<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Annu
Module annuaire Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/


require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkquery.class.php");

/**
 * @package Alkanet_Module_Annu
 *
 * @class AlkQueryAnnuAction 
 * Ensemble des requetes actions liées à l'application annuaire
 * Les méthodes intègrent également les actions ldap en modif.
 * Ces méthodes ne vérifie pas ALK_LDAP. La vérif est à effectuer avant l'appel
 */
class AlkQueryAnnuAction extends AlkQuery
{
  /** référence sur la connexion ldap */
  protected $connLdap;
  
  /** tableau associatif annuaire ldap-alkanet */
  protected $tabLdapAssoc;
  
	/**
   *  Constructeur par défaut
   * @param oDb Référence sur l'objet de connexion AlkDb
   */ 
  function __construct(&$oDb)
  {
    parent::__construct($oDb);
    
    // connLdap is null if ALK_LDAP==false
    $this->connLdap = AlkFactory::getLdapConn();
    $this->tabLdapAssoc = ( ALK_LDAP ? unserialize(ALK_LDAP_ASSOC_ANNU) : array() );
  }

  /**
   *  ajoute une coordonnées à un agent
   *        Retourne l'identifiant de la coordonnée
   * @param agent_id          identifiant de l'agent
   * @param telecom_id        identifiant du type de coordonnée
   * @param coor_valeur       valeur de la coordonnées (mail, n° tél, etc...)
   * @param coor_complement   complément d'information
   * @param coor_publication  niveau de communication
   * @return int
   */
  private function addCoord($agent_id, $telecom_id, $coor_valeur,
                            $coor_complement, $coor_publication)
  {
    $coor_id = $this->dbConn->GetNextId("SIT_COORDONNEE", "COOR_ID", "SEQ_SIT_COORDONNEE");
    $strSql = "insert into SIT_COORDONNEE (COOR_ID, AGENT_ID, TELECOM_ID, ".
      "COOR_VALEUR, COOR_COMPLEMENT, COOR_PUBLICATION) values (".$coor_id.
      ", ".$agent_id.
      ",".$telecom_id.
      ",'".$this->dbConn->analyseSql($coor_valeur)."'".
      ",'".$this->dbConn->analyseSql($coor_complement)."'".
      ",".$coor_publication.")";
    $this->dbConn->ExecuteSql($strSql);
    return $coor_id;
  }

  /**
   *  modifie une coordonnées d'un agent
   * @param coor_id           identifiant de la coordonnées
   * @param agent_id          identifiant de l'agent
   * @param telecom_id        identifiant du type de coordonnée
   * @param coor_valeur       valeur de la coordonnées (mail, n° tél, etc...)
   * @param coor_complement   complément d'information
   * @param coor_publication  niveau de communication
   */
  private function updateCoord($coor_id, $agent_id, $telecom_id, $coor_valeur,
                               $coor_complement, $coor_publication)
  {
    $strSql = "update SIT_COORDONNEE set ".
      "  COOR_VALEUR='".$this->dbConn->analyseSql($coor_valeur)."'".
      ", COOR_COMPLEMENT='".$this->dbConn->analyseSql($coor_complement)."'".
      ", COOR_PUBLICATION=".$coor_publication.
      ", TELECOM_ID=".$telecom_id.
      " where COOR_ID=".$coor_id." and AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  Efface une coordonnées ou plusieurs coordonnées
   *        si coor_id=-1 et telecom_id!=-1 : supprime toutes les coordonnées de type telecom_id de agent_id
   *        si coor_id=-1 et telecom_id=-1 : supprime toutes les coordonnées de agent_id
   *        sinon supprime la coordonnées coor_id de agent_id
   * @param coor_id           identifiant de la coordonnées
   * @param agent_id          identifiant de l'agent
   * @param telecom_id        identifiant du type de coordonnée, =-1 par défaut
   */
  private function delCoord($coor_id, $agent_id, $telecom_id="-1")
  {
    $strSql = "delete from SIT_COORDONNEE where COOR_ID=".$coor_id." and AGENT_ID=".$agent_id;
    if( $telecom_id!="-1" && $coor_id=="-1" ) {
      $strSql = "delete from SIT_COORDONNEE where TELECOM_ID=".$telecom_id." and AGENT_ID=".$agent_id;
    } elseif( $telecom_id=="-1" && $coor_id=="-1" ) {
      $strSql = "delete from SIT_COORDONNEE where AGENT_ID=".$agent_id;
    }
    $this->dbConn->ExecuteSql($strSql);
  }

  /** 
   *  Ajout d'un profil. Retourne l'id du profil généré
   * @param tabQuery       tableau contenant les informations à enregistrer
   * @return int
   */
  public function addProfil($tabQuery)
  {
    $tabValue = $tabQuery["field"];

    $profil_id = $this->dbConn->getNextId("SIT_PROFIL", "PROFIL_ID", "SEQ_SIT_PROFIL");
    $tabValue["PROFIL_ID"]    = array(ALK_SQL_NUMBER, $profil_id);
    
    $strSql = "insert into SIT_PROFIL ".$this->getPartInsertSql($tabValue);
    $this->dbConn->ExecuteSql($strSql);

    return $profil_id;
  }

  /** 
   *  Suppression du profil 
   *        La suppression des droits est faite par oSpace
   * @param profil_id  identifiant du profil à supprimer
   */
  public function delProfil($profil_id)
  {
    $strSql = "delete from SIT_AGENT_PROFIL where PROFIL_ID=".$profil_id;
    $this->dbConn->ExecuteSql($strSql);
    
    $strSql = "delete from SIT_PROFIL where PROFIL_ID=".$profil_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * @rief Supprime la filiation d'un agent
   * @param agent_id  identifiant de l'agent à traiter
   */
  public function delFiliation($agent_id)
  {
    $strSql = "delete from SIT_AGENT_SERVICE where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * @rief Supprime puis actualise la filiation d'un agent
   * @param agent_id  identifiant de l'agent à traiter
   * @param iNiv      niveau du service modifié
   * @param serv1_id  identifiant du service de niveau 1
   * @param serv2_id  identifiant du service de niveau 2
   * @param serv3_id  identifiant du service de niveau 3
   * @param serv4_id  identifiant du service de niveau 4
   */
  public function delAddFiliation($agent_id, $iNiv, $serv1_id, $serv2_id, $serv3_id, $serv4_id)
  {
    $bRes = false;
    $strList =
      ( $serv1_id!="-1" 
        ? $serv1_id.",".
        ( $serv2_id!="-1" 
          ? $serv2_id.",".
          ( $serv3_id!="-1" 
            ? $serv3_id.",".
            ( $serv4_id!="-1" 
              ? $serv4_id.","
              : "")
            : "")
          : "")
        : "");

    if( substr($strList, -1) == "," )
      $strList = substr($strList, 0, -1);

    if( $strList != "" ) {
      $this->delFiliation($agent_id);
      
     $strSql = "insert into SIT_AGENT_SERVICE (SERVICE_ID, AGENT_ID) ".
        "select SERVICE_ID, ".$agent_id.
        " from SIT_SERVICE".
        " where SERVICE_ID in (".$strList.")";
      $this->dbConn->ExecuteSql($strSql);
      $bRes = true;
    }

    return $bRes;
  }

  /**
   *  Modifie la filiation d'un agent d'un service vers un autre
   *        A partir de l'arbo
   * @param idAgent    Identifiant de l'utilisateur à déplacer
   * @param idServSrc  Identifiant du service source
   * @param idServDest Identifiant du service destination
   */
  public function updateFiliation($idAgent, $idServSrc, $idServDest)
  {
    // ajoute les services successifs jusqu'à atteindre idServDest
    $strArbre = "";
    $strSql = "select SERVICE_ARBRE from SIT_SERVICE where SERVICE_ID=".$idServDest;
    $dsServ = $this->dbConn->InitDataSet($strSql);
    if( $drServ = $dsServ->getRowIter() )
      $strArbre = $drServ->getValueName("SERVICE_ARBRE");
      
    if( $strArbre != "" ) {  
      $strArbre = substr($strArbre, 3, -1);
      $strArbre = "(".str_replace("-", ",", $strArbre).")";

      // supprime sa filiation : nettoyage avant suppression
      $this->delFiliation($idAgent);

      $strSql = "insert into SIT_AGENT_SERVICE (SERVICE_ID, AGENT_ID) ".
        "select SERVICE_ID, ".$idAgent.
        " from SIT_SERVICE".
        " where SERVICE_ID in ".$strArbre;
      $this->dbConn->ExecuteSql($strSql);
    }
  }

  /**
   *  Ajoute une mission à un utilisateur en prenant soin de la supprimer si elle existe déjà
   * @param agent_id   identifiant de l'agent
   * @param mission_id identifiant de la mission ou tableau d'identifiant de mission
   * @param bDelAdd    true par défaut pour supprimer l'association avant de la créer
   */
  public function delAddAgentMission($agent_id, $mission_id, $bDelAll=true)
  {
    if( $bDelAll == true ) {
      $strSql = "delete from SIT_AGENT_MISSION where AGENT_ID=".$agent_id;
      $this->dbConn->ExecuteSql($strSql);
    }
    
    $tabMissions = $mission_id;
    if( !is_array($tabMissions) && is_numeric($tabMissions) ) {
      $tabMissions = array($mission_id);
    }
    foreach($tabMissions as $mission_id) {
      if( $mission_id != "-1" ) {
        if( !$bDelAll ) {
          $strSql = "delete from SIT_AGENT_MISSION where AGENT_ID=".$agent_id." and MISSION_ID=".$mission_id;
          $this->dbConn->ExecuteSql($strSql);
        }
        $strSql = "insert into SIT_AGENT_MISSION (AGENT_ID, MISSION_ID) values (".$agent_id.", ".$mission_id.")";
        $this->dbConn->ExecuteSql($strSql);
      }
    }
  }
  
  /**
   *  Supprime l'association agent - mission
   * @param agent_id   identifiant de l'agent
   * @param mission_id identifiant de la mission
   */
  public function delAgentMission($agent_id, $mission_id)
  {
    $strSql = "delete from SIT_AGENT_MISSION where AGENT_ID=".$agent_id." and MISSION_ID=".$mission_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  Ajoute l'association agent - profil
   *        Dans le cas d'un profil primaire, SIT_AGENT est mis à jour.
   * @param agent_id   identifiant de l'agent
   * @param profil_id  identifiant du groupe secondaire
   * @param bPrimary   =false par défaut pour caractériser un profil secondaire, = true pour un profil primaire
   */
  public function addAgentProfil($agent_id, $profil_id, $bPrimary=false)
  {
    // nettoyage avant insertion
    $strSql = "delete from SIT_AGENT_PROFIL where AGENT_ID=".$agent_id." and PROFIL_ID=".$profil_id;
    $this->dbConn->ExecuteSql($strSql);

    if( $bPrimary ) {
      // suppression de l'ancien profil
      $strSql = "delete from SIT_AGENT_PROFIL where AGENT_ID=".$agent_id.
        " and PROFIL_ID in (select PROFIL_ID from SIT_AGENT where AGENT_ID=".$agent_id.")";
      $this->dbConn->ExecuteSql($strSql);

      $strSql = "update SIT_AGENT set PROFIL_ID=".$profil_id." where AGENT_ID=".$agent_id;
      $this->dbConn->ExecuteSql($strSql);
    }
  
    $strSql = "insert into SIT_AGENT_PROFIL (AGENT_ID, PROFIL_ID) values (".$agent_id.", ".$profil_id.")";
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Associe une liste d'agent au profil_id
   * La liste des agents est présente en request par tableau
   * @param profil_id  identifiant du profil sélectionné
   */
  public function deladdUsersToProfil($profil_id, $bAdd=true)
  {
    if( $bAdd ) {
    $tabListeAgentId = AlkRequest::_POST("listUsersId", array(), "is_array");
    } else {
      $strTabListeAgentId = AlkRequest::_GET("listUserId", "-1");
      $tabListeAgentId = explode(",", $strTabListeAgentId);
    }
                         
    $tabListeId = array();
    $iCpt = 0;
    $tabListeId[$iCpt] = "";
    foreach($tabListeAgentId as $agent_id) {
      if( $agent_id != "" && $agent_id != "-1" )
        $tabListeId[$iCpt] .= ( $tabListeId[$iCpt] == "" ? "" : "," ).$agent_id;
      if( mb_strlen($tabListeId[$iCpt])>500 ) {
        $iCpt++;
        $tabListeId[$iCpt] = "";
      }
    }
    foreach($tabListeId as $strListId) {
      $strSql = "delete from SIT_AGENT_PROFIL where AGENT_ID in (".$strListId.") and PROFIL_ID=".$profil_id;
      $this->dbConn->ExecuteSql($strSql);

      if( $bAdd ) {
        $strSql = "insert into SIT_AGENT_PROFIL (PROFIL_ID, AGENT_ID)".
          " select ".$profil_id.", AGENT_ID from SIT_AGENT where AGENT_ID in (".$strListId.")";
        $this->dbConn->ExecuteSql($strSql);
      }
    } 
    return $tabListeId;
  }

  /**
   *  Supprime l'association agent - profil
   *        l'interface garantit que profil_id est un profil secondaire
   * @param agent_id   identifiant de l'agent
   * @param profil_id  identifiant du groupe secondaire
   */
  public function delAgentProfil($agent_id, $profil_id)
  {
    $strSql = "delete from SIT_AGENT_PROFIL where AGENT_ID=".$agent_id.
      ($profil_id != "-1" ? " and PROFIL_ID=".$profil_id : "" );
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Modifie le login d'un agent
   * @param agent_id    identifiant de l'agent
   * @param agent_login login de l'agent
   */
  public function updateUserLogin($agent_id, $agent_login)
  {
    $strSql = "update SIT_AGENT set".
      " AGENT_LOGIN='".$this->dbConn->analyseSql($agent_login)."'".
      " where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
    
    $strSql = "update SIT_ABONNE set" .
      " ABONNE_PSEUDO='".$this->dbConn->analyseSql($agent_login)."'".
      " where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  Ajoute une adresse associée a un agent. Retourne l'id de l'adresse
   * @param agent_id   identifiant de l'agent
   * @param adr_adr1   adresse 1
   * @param adr_adr2   adresse 2
   * @param adr_adr3   adresse 3
   * @param adr_post   code postal
   * @param adr_ville  ville
   * @param adr_desc   description
   * @return int
   */
  public function addAddress($agent_id, $adr_adr1, $adr_adr2,
                            $adr_adr3, $adr_post, $adr_ville, $adr_desc, $adr_pays)
  {
    $adr_id = $agent_id;

    $strSql = "insert into SIT_ADRESSE (ADR_ID, AGENT_ID, ADR_ADR1,".
      "ADR_ADR2,  ADR_ADR3, ADR_POST, ADR_VILLE, ADR_DESC, ADR_PAYS) values (".$adr_id.
      ", ".$agent_id.
      ",'".$this->dbConn->analyseSql($adr_adr1)."'".
      ",'".$this->dbConn->analyseSql($adr_adr2)."'".
      ",'".$this->dbConn->analyseSql($adr_adr3)."'".
      ",'".$this->dbConn->analyseSql($adr_post)."'".
      ",'".$this->dbConn->analyseSql($adr_ville)."'".
      ",'".$this->dbConn->analyseSql($adr_desc)."'" .
      ",'".$this->dbConn->analyseSql($adr_pays)."'" .
      ")";

    $this->dbConn->ExecuteSql($strSql);
    return $adr_id;
  }
  
  /**
   *   Modifie une adresse associée a un agent
   * @param agent_id   identifiant de l'agent
   * @param adr_adr1   adresse 1
   * @param adr_adr2   adresse 2
   * @param adr_adr3   adresse 3
   * @param adr_post   code postal
   * @param adr_ville  ville
   * @param adr_desc   description
   */
  public function updateAddress($agent_id, $adr_id, $adr_adr1, $adr_adr2,
                                $adr_adr3, $adr_post, $adr_ville, $adr_desc, $adr_pays)
  {
    $this->delAddress($agent_id);
    $this->addAddress($agent_id, $adr_adr1, $adr_adr2,
                      $adr_adr3, $adr_post, $adr_ville, $adr_desc, $adr_pays);
  }

  /**
   *  Supprime l'adresse associée a un agent
   * @param agent_id   identifiant de l'agent
   */
  private function delAddress($agent_id)
  {
    $strSql="delete from SIT_ADRESSE where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  retourne une chaine correspondant au login d'un utilistateur
   *
   * @param strNom     nom d'une personne
   * @param strPrenom  prenom d'une personne
   * @return Retourne un string
   */
  public function getLogin($strNom, $strPrenom)
  {
    $iLgPrenom = 1;
    $iLgNom = ALK_ANNU_LG_MAX_LOGIN-$iLgPrenom;
    
    $strTmpNom = trim(mb_strtolower($strNom));
    $strTmpPrenom = trim(mb_strtolower($strPrenom));
  
    // enleve les accents
    $strTmpNom = strtr($strTmpNom, "éèêëäàâüùûîïôöç", "eeeeaaauuuiiooc");
    $strTmpPrenom = strtr($strTmpPrenom, "éèêëäàâüùûîïôöç", "eeeeaaauuuiiooc");
  
    // enleve les caractères non compris entre a et z.
    $strTmpNom2 = "";
    for($i=0; $i<mb_strlen($strTmpNom); $i++) {
      $cCar = mb_substr($strTmpNom, $i, 1);
      if( ord($cCar)>=ord("a") && ord($cCar)<=ord("z") )
        $strTmpNom2 .= $cCar;
    }
  
    $strTmpPrenom2 = "";
    for($i=0; $i<mb_strlen($strTmpPrenom); $i++) {
      $cCar = mb_substr($strTmpPrenom, $i, 1);
      if( ord($cCar)>=ord("a") && ord($cCar)<=ord("z") )
        $strTmpPrenom2 .= $cCar;
    }
  
    $strLogin = mb_substr($strTmpPrenom2, 0, $iLgPrenom).mb_substr($strTmpNom2, 0, $iLgNom);
  
    return $strLogin;
  }

  /**
   *  Génère un mot de passe selon le masque donnée
   * @param strMask  Mask du mot de passe, =cLLlclcL par défaut
   *                 c : chiffre 0-9
   *                 L : lettre majuscule
   *                 l : lettre minuscule
   *                 s : caractère spécial
   * @return string
   */
  public function getStrGeneratedPwd($strMask="cLLlclcL")
  {
    $strPwd = "";
    for ($i=0; $i<strlen($strMask); $i++) {
			switch ($strMask[$i]) {
      case "c":
        $iCharCode = mt_rand(48,57);
        break;
      case "l":
        $iCharCode = mt_rand(97,122);
        break;
      case "L":
        $iCharCode = mt_rand(65,90);
        break;
      case "s":
        $iCharCode = ord(";");
			}
			$strPwd .= chr($iCharCode); 
    }
    return $strPwd;	
  }
  
  /**
   * Vérifie agent_login et retourne une nouvelle proposition si non unique
   * En mode utilisation de l'adresse mail, retourne chaine vide si non unique 
   * @param idAgent  Identifiant de l'agent
   * @param strLogin Login proposé
   * @param lgMax    Longueur max du login
   */
  public function getStrVerifyLogin($idAgent, $strLogin, $lgMax)
  {
    
    if ( defined("ALK_B_ANNU_LOGIN_USE_MAIL") && ALK_B_ANNU_LOGIN_USE_MAIL == true ) {
      
      $strSql = "select COUNT(AGENT_LOGIN) as nbLoginEgal from SIT_AGENT " .
          "where AGENT_ID<>".$idAgent.
          " and ".$this->dbConn->GetLowerCase("AGENT_LOGIN")."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strLogin)."'");
      $dsLogin = $this->dbConn->InitDataSet($strSql);
      
      if( $drLogin = $dsLogin->getRowIter() ) {
        $nbLoginIdent = $drLogin->getValueName("NBLOGINEGAL");
      }
      
      if( $nbLoginIdent>0 ) {
        return "";
      }
      
      return $strLogin;
    }
  
    $iLgLogin = strlen($strLogin);
    $iLgLoginNum = $iLgLogin+1;
    if( $iLgLoginNum > $lgMax ) $iLgLoginNum = $lgMax;

    $strSql = "select a.nbLoginEgal, b.nbLoginSemblable, c.numMax ".
      " from ".
      "  (select COUNT(AGENT_LOGIN) as nbLoginEgal ".
      "   from SIT_AGENT ".
      "   where AGENT_ID<>".$idAgent.
      "     and ".$this->dbConn->GetLowerCase("AGENT_LOGIN")."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strLogin)."'").") a,".
      
      "  (select COUNT(AGENT_LOGIN) as nbLoginSemblable ".
      "   from SIT_AGENT ".
      "   where LENGTH(AGENT_LOGIN)=".$iLgLoginNum.
      "     and ".$this->dbConn->GetLowerCase($this->dbConn->GetSubstring("AGENT_LOGIN", 0, $iLgLogin))."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strLogin)."'").
      "     and ASCII(".$this->dbConn->GetSubstring("AGENT_LOGIN", $iLgLogin, 1).")>48 ".
      "     and ASCII(".$this->dbConn->GetSubstring("AGENT_LOGIN", $iLgLogin, 1).")<58".
      "  ) b,".

      "  (select max(ASCII(".$this->dbConn->GetSubstring("AGENT_LOGIN", $iLgLogin, 1)."))-48 as numMax ".
      "   from SIT_AGENT ".
      "   where LENGTH(AGENT_LOGIN)=".$iLgLoginNum.
      "     and ".$this->dbConn->GetLowerCase($this->dbConn->GetSubstring("AGENT_LOGIN", 0, $iLgLogin))."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strLogin)."'").
      "     and ASCII(".$this->dbConn->GetSubstring("AGENT_LOGIN", $iLgLogin, 1).")>48 ".
      "     and ASCII(".$this->dbConn->GetSubstring("AGENT_LOGIN", $iLgLogin, 1).")<58".
      "  ) c";

    $dsLogin = $this->dbConn->InitDataSet($strSql);
    $nbLogin = 0;
    $numMax = 1;
    if( $drLogin = $dsLogin->getRowIter() ) {
      $nbLoginIdent     = $drLogin->getValueName("NBLOGINEGAL");
      $nbLoginSemblable = $drLogin->getValueName("NBLOGINSEMBLABLE");
      $numMaxTmp           = $drLogin->getValueName("NUMMAX");
      if( $nbLoginIdent     == "" ) $nbLoginIdent = 0;
      if( $nbLoginSemblable == "" ) $nbLoginSemblable = 0;
      if( $numMaxTmp        != "" ) $numMax = $numMaxTmp+1; 

      $nbLogin = $nbLoginIdent+$nbLoginSemblable;
    }
    
    if( $nbLogin>0 ) {
      $strLogin .= $numMax;
    }
    
    return strtolower($strLogin);      
  }

  /**
   * Ajoute un agent et enregistre sa filiation avec le service de niveau 1
   *        Retourne un tableau contenant les informations pour le mailing
   * Répercute l'ajout sur SIT_ABONNE
   * @see AlkImportAnnu
   * 
   * @param user_id        identifiant de l'agent connecté responsable dGetStr_verifyLogine l'ajout
   * @param tabQuery       tableau contenant les informations à enregistrer
   * @param agent_id_dist  identifiant de l'agent connecté distant correspondant à cette fiche, =-1 par défaut
   * @return array
   */
  public function addUser($user_id, &$tabQuery, $agent_id_dist=-1)
  {
    $tabValue = $tabQuery["field"];

    $agent_id    = "-1";
    $agent_login = ( array_key_exists("AGENT_LOGIN", $tabValue) &&
                     is_array($tabValue["AGENT_LOGIN"]) && 
                     count($tabValue["AGENT_LOGIN"])==2 
                     ? $tabValue["AGENT_LOGIN"][1]
                     : "" );
    if( $agent_login != "" ) { 
      $agent_login = $this->getStrVerifyLogin($agent_id, $agent_login, ALK_ANNU_LG_MAX_LOGIN);
    }
    if( $agent_id_dist==-1 ) {
      $agent_id = $this->dbConn->GetNextId("SIT_AGENT", "AGENT_ID", "SEQ_SIT_AGENT");
    } else {
      $agent_id = $agent_id_dist;
    }

    if( $agent_id == "-1" ) 
      return array("agent_id" => -1);
      
    $agent_nom    = $tabValue["AGENT_NOM"][1];
    $agent_prenom = $tabValue["AGENT_PRENOM"][1];
    if( $agent_login=="" ) {
      if ( defined("ALK_B_ANNU_LOGIN_USE_MAIL") && ALK_B_ANNU_LOGIN_USE_MAIL == true ) {
        $agent_login = $tabValue["AGENT_MAIL"][1];
      } else {
        $agent_login = $this->getLogin($agent_nom, $agent_prenom);
      }
      $agent_login = $this->getStrVerifyLogin($agent_id, $agent_login, ALK_ANNU_LG_MAX_LOGIN);
    }

    if ( $agent_login == "" )
      return array("agent_id" => -1);

    /* agent_pwd contient un mot de passe non crypté, possibilité de le fournir à l'import */
    $agent_pwd = ( array_key_exists("AGENT_PWD", $tabValue) && 
                   is_array($tabValue["AGENT_PWD"]) && 
                   count($tabValue["AGENT_PWD"])==2 &&
                   $tabValue["AGENT_PWD"][1] != ""
                   ? $tabValue["AGENT_PWD"][1]
                   : $this->getStrGeneratedPwd() );
                   
    $agent_pwd_encrypt = AlkFactory::encrypt($agent_pwd);
    if( array_key_exists("AGENT_PWD_ENCRYPT", $tabValue) && 
        is_array($tabValue["AGENT_PWD_ENCRYPT"]) &&
        count($tabValue["AGENT_PWD_ENCRYPT"])==2 &&
        $tabValue["AGENT_PWD_ENCRYPT"][1] != "" ) {
      // on laisse la possibilité d'importer un mot de passe déjà encrypté
      $agent_pwd_encrypt = $tabValue["AGENT_PWD_ENCRYPT"][1];
      unset($tabValue["AGENT_PWD_ENCRYPT"]);
      // on ne connait la valeur non cryptée
      $agent_pwd = ""; 
    }
                   
    $agent_valide = $tabValue["AGENT_VALIDE"][1];
    $service_id = $tabValue["SERVICE_ID"][1];

    unset($tabValue["SERVICE_ID"]);
    $tabValue["AGENT_ID"]    = array(ALK_SQL_NUMBER, $agent_id);
    $tabValue["AGENT_LOGIN"] = array(ALK_SQL_TEXT,   $agent_login);
    $tabValue["AGENT_PWD"]   = array(ALK_SQL_TEXT,   $agent_pwd_encrypt);

    $tabValue["AGENT_DATE_CREA"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    $tabValue["AGENT_ID_CREA"]   = array(ALK_SQL_NUMBER, ($user_id==ALK_USER_ID_INTERNET ? ALK_USER_ID_ADMINALK : $user_id));
    $tabValue["AGENT_DATE_MAJ"]  = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    $tabValue["AGENT_ID_MAJ"]    = array(ALK_SQL_NUMBER, ($user_id==ALK_USER_ID_INTERNET ? ALK_USER_ID_ADMINALK : $user_id));
    if( $agent_valide == "1" ) {
      $tabValue["AGENT_DATEVALID"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    }

    // recuperation des services superieurs si existants
    $serv_niv2 = array_key_exists("SERV_NIV2", $tabValue) ? $tabValue["SERV_NIV2"][1] : -1;
    $serv_niv3 = array_key_exists("SERV_NIV3", $tabValue) ? $tabValue["SERV_NIV3"][1] : -1;
    $serv_niv4 = array_key_exists("SERV_NIV4", $tabValue) ? $tabValue["SERV_NIV4"][1] : -1;
    $bImport   = array_key_exists("IMPORT_ADD", $tabValue);
    if( array_key_exists("SERV_NIV2", $tabValue) ) unset($tabValue["SERV_NIV2"]);
    if( array_key_exists("SERV_NIV3", $tabValue) ) unset($tabValue["SERV_NIV3"]);
    if( array_key_exists("SERV_NIV4", $tabValue) ) unset($tabValue["SERV_NIV4"]);
    if( array_key_exists("IMPORT_ADD", $tabValue) ) unset($tabValue["IMPORT_ADD"]);
    
    // l'adresse peut être ajoutée à l'import
    $adr_adr1  = array_key_exists("ADR_ADR1",  $tabValue) ? $tabValue["ADR_ADR1"][1]  : "";
    $adr_adr2  = array_key_exists("ADR_ADR2",  $tabValue) ? $tabValue["ADR_ADR2"][1]  : "";
    $adr_adr3  = array_key_exists("ADR_ADR3",  $tabValue) ? $tabValue["ADR_ADR3"][1]  : "";
    $adr_post  = array_key_exists("ADR_POST",  $tabValue) ? $tabValue["ADR_POST"][1]  : "";
    $adr_ville = array_key_exists("ADR_VILLE", $tabValue) ? $tabValue["ADR_VILLE"][1] : "";
    $adr_desc  = array_key_exists("ADR_DESC",  $tabValue) ? $tabValue["ADR_DESC"][1]  : "";
    $adr_pays  = array_key_exists("ADR_PAYS",  $tabValue) ? $tabValue["ADR_PAYS"][1]  : "";

    if( array_key_exists("ADR_ADR1",  $tabValue) ) unset($tabValue["ADR_ADR1"]);
    if( array_key_exists("ADR_ADR2",  $tabValue) ) unset($tabValue["ADR_ADR2"]);
    if( array_key_exists("ADR_ADR3",  $tabValue) ) unset($tabValue["ADR_ADR3"]);
    if( array_key_exists("ADR_POST",  $tabValue) ) unset($tabValue["ADR_POST"]);
    if( array_key_exists("ADR_VILLE", $tabValue) ) unset($tabValue["ADR_VILLE"]);
    if( array_key_exists("ADR_DESC",  $tabValue) ) unset($tabValue["ADR_DESC"]);
    if( array_key_exists("ADR_PAYS",  $tabValue) ) unset($tabValue["ADR_PAYS"]);
    
    if( !array_key_exists("AGENT_LG", $tabValue) ) {
      $tabValue["AGENT_LG"] = array(ALK_SQL_TEXT, $GLOBALS["tabLocales"]["backoffice"][0]);
    }
    
    // création fiche agent
    $strSql = "insert into SIT_AGENT ".$this->getPartInsertSql($tabValue);
    //echo $strSql."<br>";
    $this->dbConn->ExecuteSql($strSql);

    // enregistre l'email dans les coordonnées
    $agent_mail = $tabValue["AGENT_MAIL"][1];
    $this->addCoord($agent_id, "4", $agent_mail, '', 4);
    
    // association au service de niveau 1, ou de niveau superieurs
    $this->delAddFiliation($agent_id, 1, $service_id, $serv_niv2, $serv_niv3, $serv_niv4);
    
    // ajout de l'adresse
    $this->addAddress($agent_id, $adr_adr1, $adr_adr2, $adr_adr3, $adr_post, $adr_ville, $adr_desc, $adr_pays);
    
    // association agent-profil si import
    $this->addAgentProfil($agent_id, $tabValue["PROFIL_ID"][1], true); 

    // ajout dans la table SIT_ABONNE : pas de mot de passe pour un agent authentifié
    $this->addAbonne($agent_id, $agent_nom, $agent_prenom, $agent_mail, 1, "", "", "", "", "", "", $agent_login);

    // information pour le mailing
    $tabRes["agent_id"]       = $agent_id;
    $tabRes["service_id"]     = $service_id;
    $tabRes["profil_id"]      = $tabValue["PROFIL_ID"][1];
    $tabRes["agent_civilite"] = $tabValue["AGENT_CIVILITE"][1];
    $tabRes["agent_nom"]      = $agent_nom;
    $tabRes["agent_prenom"]   = $agent_prenom;
    $tabRes["agent_mail"]     = $agent_mail;
    $tabRes["agent_login"]    = $agent_login; 
    $tabRes["agent_pwd"]      = $agent_pwd;
    $tabRes["agent_valide"]   = $agent_valide;
    $tabRes["agent_lg"]       = $tabValue["AGENT_LG"][1];
    $tabRes["agent_adminauth"]= ( isset($tabValue["AGENT_ADMINAUTH"][1]) ? $tabValue["AGENT_ADMINAUTH"][1] : 1 );
    
    //le param SERVICE_ID est ajouté après execution des requêtes pour passage par référence
    $tabValue["SERVICE_ID"] = array(ALK_SQL_NUMBER, $service_id);
    $tabQuery["field"] = $tabValue;
    
    return $tabRes;
  }

  /**
   * Modifie un agent. Retourne un tableau associatif contenant les informations pour le mailing.
   * Répercute la mise à jour sur SIT_ABONNE
   * @see AlkImportAnnu
   * 
   * @param user_id        identifiant de l'agent connecté responsable dGetStr_verifyLogine l'ajout
   * @param tabQuery       tableau contenant les informations à enregistrer
   * @param strPathUpload  chemin d'upload pour la photo
   * @return array
   */
  public function updateUser($agent_id, $tabQuery, $strPathUpload)
  {
    $tabValue = $tabQuery["field"];

    $user_id     = AlkFactory::getSProperty("user_id", "-1");
    $userServ_id = AlkFactory::getSProperty("service_id", "-1");
    $agent_id    = $tabQuery["pk"]["AGENT_ID"][1];
    $service_id  = $tabValue["SERVICE_ID"][1];
    $bOwner      = ( $user_id == $agent_id );
    $bCanInitPwd = ( AlkFactory::isAnnuAdminAll() ||
                     AlkFactory::isAnnuAdminServ() && $userServ_id==$service_id );

    if( !array_key_exists("AGENT_LG", $tabValue) ) {
      $tabValue["AGENT_LG"] = array(ALK_SQL_TEXT, $GLOBALS["tabLocales"]["backoffice"][0]);
    }

    $tabRes["agent_id"]         = $agent_id;
    $tabRes["service_id"]       = $service_id;
    $tabRes["profil_id"]        = $tabValue["PROFIL_ID"][1];
    $tabRes["agent_civilite"]   = $tabValue["AGENT_CIVILITE"][1];
    $tabRes["agent_nom"]        = $tabValue["AGENT_NOM"][1];
    $tabRes["agent_prenom"]     = $tabValue["AGENT_PRENOM"][1];
    $tabRes["agent_mail"]       = $tabValue["AGENT_MAIL"][1];
    $tabRes["agent_lg"]         = $tabValue["AGENT_LG"][1];
    $tabRes["agent_desc"]       = $tabValue["AGENT_DESC"][1];
    $tabRes["agent_priv"]       = $tabValue["AGENT_PRIV"][1];
    $strAgentLoginMem           = $tabValue["AGENT_LOGIN"][1];
    $tabRes["agent_login"]      = "";
    $tabRes["agent_pwd"]        = "";
    $tabRes["agent_pwdinit"]    = false;   
    $tabRes["agent_valide"]     = $tabValue["AGENT_VALIDE"][1];
    $tabRes["agent_valide_old"] = ( isset($tabValue["AGENT_VALIDEMEM"]) ? $tabValue["AGENT_VALIDEMEM"][1] : $tabRes["agent_valide"] );
    $tabRes["agent_adminauth"]  = -1;
    //$tabRes["alk_b_import"] mis à jour plus loin
    // traitement de l'upload
    $strFileName = doUpload("agent_photo", $agent_id."_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["AGENT_PHOTO"][1] = $strFileName;
    } else {
      unset($tabValue["AGENT_PHOTO"]);
    }

    // ne traiter le pwd que si l'on ne vient pas de l'import Annu
    if( !array_key_exists("ALK_B_IMPORT", $tabValue) ) {
      // traitement pwd
      if( $bOwner && !(defined("ALK_B_ANNU_USER_CHANGE_PWD") && ALK_B_ANNU_USER_CHANGE_PWD==false) ) {
        $strDefaultPwd = "";
        for($i=0; $i<ALK_ANNU_LG_MAX_PWD; $i++) {
          $strDefaultPwd .= "*";
        }
        if( $tabValue["AGENT_PWD"][1] == $strDefaultPwd ) {
          unset($tabValue["AGENT_PWD"]);
        } else {
          // conserver les champs pour l'envoi de mail
          $tabRes["agent_login"]  = $tabValue["AGENT_LOGIN"][1];
          $tabRes["agent_pwd"]    = $tabValue["AGENT_PWD"][1];
          
          $tabValue["AGENT_PWD"][1] = AlkFactory::encrypt($tabRes["agent_pwd"]);
        }
        unset($tabValue["AGENT_PWD2"]);
      } elseif( $bCanInitPwd && $tabValue["AGENT_PWDINIT"][1]==1) { // 
        $tabRes["agent_pwdinit"] = true;
        $agent_pwd = $this->getStrGeneratedPwd(); // regenerer un mdp ici
        
        $tabRes["agent_pwd"]      = $agent_pwd; //$tabValue["AGENT_PWD"][1];
        $tabValue["AGENT_PWD"][1] = AlkFactory::encrypt($tabRes["agent_pwd"]);
        $tabRes["agent_login"]    = $tabValue["AGENT_LOGIN"][1];
        unset($tabValue["AGENT_PWD2"]);
      } else {
        unset($tabValue["AGENT_PWD"]);
        unset($tabValue["AGENT_PWD2"]);
      }
      $tabRes["alk_b_import"] = false;
      unset($tabValue["AGENT_PWDINIT"]);
    } else {
      $tabRes["alk_b_import"] = true;
      unset($tabValue["ALK_B_IMPORT"]);
    }
    
    // vérifie si le login a été modifié
    if( $tabValue["AGENT_LOGIN"][1]!="" && $tabValue["AGENT_LOGIN"][1] != $tabValue["AGENT_LOGINMEM"][1] ) {
      $tabRes["agent_login"] = $tabValue["AGENT_LOGIN"][1];  
    }
    
    $adr_adr1 = $tabValue["ADR_ADR1"][1]; 
    $adr_adr2 = ( array_key_exists("ADR_ADR2", $tabValue) ? $tabValue["ADR_ADR2"][1] : "" ); 
    $adr_adr3 = ( array_key_exists("ADR_ADR3", $tabValue) ? $tabValue["ADR_ADR3"][1] : "" ); 
    $adr_post = $tabValue["ADR_POST"][1]; 
    $adr_ville= $tabValue["ADR_VILLE"][1];
    $adr_pays = ( array_key_exists("ADR_PAYS", $tabValue) ? $tabValue["ADR_PAYS"][1] : "" );
    $adr_desc = ( array_key_exists("ADR_DESC", $tabValue) ? $tabValue["ADR_DESC"][1] : "" );

    unset($tabValue["ADR_ADR1"]); 
    if( array_key_exists("ADR_ADR2", $tabValue) ) unset($tabValue["ADR_ADR2"]); 
    if( array_key_exists("ADR_ADR3", $tabValue) ) unset($tabValue["ADR_ADR3"]); 
    unset($tabValue["ADR_POST"]); 
    unset($tabValue["ADR_VILLE"]);
    if( array_key_exists("ADR_PAYS", $tabValue) ) unset($tabValue["ADR_PAYS"]);
    if( array_key_exists("ADR_DESC", $tabValue) ) unset($tabValue["ADR_DESC"]);
    unset($tabValue["SERVICE_ID"]);
    
    if(  !$bCanInitPwd ) {
      unset($tabValue["AGENT_ADMINAUTH"]); 
    } elseif( isset($tabValue["AGENT_ADMINAUTH"][1]) && $tabValue["AGENT_ADMINAUTH"][1] != "" ) {
      $tabRes["agent_adminauth"] = $tabValue["AGENT_ADMINAUTH"][1];
    } else {
      unset($tabValue["AGENT_ADMINAUTH"]); 
    }
    
    if( !$bCanInitPwd ) {
      unset($tabValue["AGENT_VALIDE"]);
      unset($tabValue["AGTYPE_ID"]);
    } elseif( $tabValue["AGENT_VALIDE"][1] != $tabValue["AGENT_VALIDEMEM"][1] && $tabValue["AGENT_VALIDE"][1]!="" ) {
      $tabRes["agent_valide"] = $tabValue["AGENT_VALIDE"][1];
      $tabRes["agent_valide_old"] = $tabValue["AGENT_VALIDEMEM"][1];
      $tabValue["AGENT_VALIDE"][0] = ALK_SQL_NUMBER;
      $tabValue["AGENT_DATEVALID"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
      
      if( $tabRes["agent_valide"] == 1 ) {
        // génère un nouveau mot de passe
        $agent_pwd = $this->getStrGeneratedPwd(); // regenerer un mdp ici
        $tabRes["agent_pwd"]   = $agent_pwd;
        $tabValue["AGENT_PWD"] = array(ALK_SQL_TEXT, AlkFactory::encrypt($tabRes["agent_pwd"]));
        $tabRes["agent_login"] = $tabValue["AGENT_LOGIN"][1];
        unset($tabValue["AGENT_PWD2"]);
      }
    } else {
      unset($tabValue["AGENT_VALIDE"]);
      if( $tabRes["alk_b_import"] ) {
        unset($tabValue["AGENT_PWD"]);
      }
    }
    unset($tabValue["AGENT_VALIDEMEM"]);
    if( $tabRes["alk_b_import"] ) {
      if( $tabValue["AGENT_LOGIN"][1] == "" ) {
        unset($tabValue["AGENT_LOGIN"]);
      }
    } else {
      unset($tabValue["AGENT_LOGIN"]);
    }
    unset($tabValue["AGENT_LOGINMEM"]);
    
    $this->updateAddress($agent_id, $agent_id, $adr_adr1, $adr_adr2, $adr_adr3, $adr_post, $adr_ville, $adr_desc, $adr_pays);

    // association agent-profil principal si import et modif
    if( $tabRes["alk_b_import"] ) {
      if( $tabValue["PROFIL_IDMEM"][1]!=$tabValue["PROFIL_ID"][1] ) {
        $this->delAgentProfil($agent_id, $tabValue["PROFIL_IDMEM"][1]);
        $this->addAgentProfil($agent_id, $tabValue["PROFIL_ID"][1]);
      } else {
        unset($tabValue["PROFIL_ID"]);
      }
      unset($tabValue["PROFIL_IDMEM"]);
    } else {
      unset($tabValue["PROFIL_ID"]);
    }

    $tabValue["AGENT_DATE_MAJ"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    $tabValue["AGENT_ID_MAJ"]   = array(ALK_SQL_NUMBER, ($user_id==ALK_USER_ID_INTERNET ? ALK_USER_ID_ADMINALK : $user_id));
    
    $strSql = "update SIT_AGENT set ".$this->getPartUpdateSql($tabValue).
      " where AGENT_ID=".$agent_id;
    //echo $strSql."<br/>";
    $this->dbConn->ExecuteSql($strSql);
        
    //dans le cas de la mise à jour de la fiche personnelle, le agent_valide est vide et l'abonné abonne_valide ne doit pas être modifié
    $this->updateAbonne(array(), "-1", $agent_id, $tabRes["agent_nom"], $tabRes["agent_prenom"], $tabRes["agent_mail"], ($tabRes["agent_valide"] != "" ? $tabRes["agent_valide"] : -1), 
                        "", "", "", "", "", "", $strAgentLoginMem);
    
    return $tabRes;
  }

  /**
   * Supprime un agent vers la corbeille
   * @see AlkImportAnnu
   * 
   * @param agent_id  identifiant de l'agent
   * @param bRegion   vrai si connexion départemantale à supprimer
   */
  public function delUser($agent_id, $bRegion)
  {
    $this->delFiliation($agent_id);
    $this->delAddAgentMission($agent_id, -1, true);
		
    $this->delCoord(-1, $agent_id);
    $this->delAddress($agent_id);
    $this->delAgentProfil($agent_id, "-1");
    
    // on efface la responsabilité du service
    $strSql = "update SIT_SERVICE set AGENT_ID=null where AGENT_ID=".$agent_id; 
    $this->dbConn->ExecuteSql($strSql);
    
    $strSql = "update SIT_AGENT set".
      " AGENT_VALIDE=-1".
      ", AGENT_ADMINAUTH=0".
      ", AGENT_LOGIN=".$this->dbConn->getConcat("'id'", "AGENT_ID").
      ", AGENT_PWD='pwd'".
      ", AGENT_MAIL=''".
      ", AGENT_PRIV=0".
      ", CONT_ID_START=0".
      ", PROFIL_ID=1".
      ", AGENT_DATE_MAJ=".$this->dbConn->getDateCur().
      " where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
    
    // tentative de suppression définitive
    $strSql = "delete from SIT_AGENT where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql, false);
  }

  // 
  // Met à jour les mots de passe d'un service
  // 02_agent_sql.php
  //
  function maj_agentPwd($service_id, $strPwdDefaut)
  {
    $strSql = "update SIT_AGENT set".
      " AGENT_PWD='".$this->dbConn->analyseSql($strPwdDefaut)."'".
      " where AGENT_ID in ".
      "(select distinct AGENT_ID from SIT_AGENT_SERVICE where SERVICE_ID=".$service_id.")";

    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  Maj du mot de passe de l'agent_id'
   *
   * @param agent_pwd  password encodé
   * @param agent_id  identifiant de l'utilisateur connecté
   */ 
  public function updateUserPwd($agent_id, $agent_pwd)
  {   
    $strSql = "UPDATE SIT_AGENT" .
       " SET AGENT_PWD='".$this->dbConn->analyseSql($agent_pwd)."'".
    	 ", AGENT_DATE_MAJ=".$this->dbConn->getDateCur().
       " WHERE AGENT_ID=".$agent_id;  
    return $this->dbConn->ExecuteSql($strSql);
  }

  //
  // Modifie la propriété publipostage d'un agent
  // 
  function maj_publipostageAgent($agent_id, $iPublipostage)
  {
    $strSql = "update SIT_AGENT set".
      " AGENT_PUBLIPOSTAGE =".$iPublipostage.
      " where AGENT_ID=".$agent_id;
      
    $this->dbConn->ExecuteSql($strSql);
  }
  // 
  // Met à jour les mots de passe d'un service
  // 02_agent_sql.php
  //
  function add_loginDpt($agent_id, $dept_id, $strLoginDpt, $strPwdDpt)
  {
    $strSql = "insert into SIT_AGENT_LOGINDPT (AGENT_ID, SITDEPT_ID, LOGIN_DPT, PWD_DPT) values (".
      $agent_id.
      ", '".$this->dbConn->analyseSql($dept_id)."'".
      ", '".$this->dbConn->analyseSql($strLoginDpt)."'".
      ", '".$this->dbConn->analyseSql($strPwdDpt)."'".
      ")";

    $this->dbConn->ExecuteSql($strSql);
  }

  // 
  // Met à jour les mots de passe d'un service
  // 02_agent_sql.php
  //
  function del_loginDpt($agent_id, $dept_id)
  {
    $strSql = "delete from SIT_AGENT_LOGINDPT where AGENT_ID=".$agent_id." and SITDEPT_ID=".$dept_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  Ajoute une fonction avancée. Retourne l'id de la fonction créée
   *
   * @param
   */
  function add_fonctionAv($fonc_intitule, $fonc_desc, $strPhotoPostName, $strPathUpload)
  {
    $fonc_id = $this->dbConn->GetNextId("SIT_FONCTION", "FONC_ID", "SEQ_SIT_FONCTION");

    //pj
    $strFileName = DoUpload($strPhotoPostName, "fonc_".$fonc_id."_", $strPathUpload, 0, "");
    if( !is_string($strFileName) ) 
      $strFileName = "";    

    $strSql = "insert into SIT_FONCTION (FONC_ID, FONC_INTITULE, FONC_DESC, FONC_PJ)".
      " values (".$fonc_id.
      ", '".$this->dbConn->analyseSql($fonc_intitule)."'".
      ", '".$this->dbConn->analyseSql($fonc_desc)."'".
      ", '".$this->dbConn->analyseSql($strFileName)."'".
      ")";
      
    $this->dbConn->ExecuteSql($strSql);    
    return $fonc_id;
  }

  /**
   *  modifie une fonction avancée.
   *
   * @param fonc_id           Identifiant de la fonction
   * @param fonc_intitule     
   * @param fonc_desc         
   * @param strPhotoPostName  nom du ctrl upload
   * @param iSupPhoto         =1 si photo actuelle est à supprimer
   * @param strPathUpload     
   */
  function maj_fonctionAv($fonc_id, $fonc_intitule, $fonc_desc, $strPhotoPostName, $iSupPhoto, $strPathUpload)
  {
    // photo
    $bImageMaj = false;
    $strFileName = DoUpload($strPhotoPostName, "fonc_".$fonc_id."_", 
                            $strPathUpload, $iSupPhoto, "fonc_".$fonc_id."_*");
    if( is_string($strFileName) ) 
      $bImageMaj = true;
    
    $strSql = "update SIT_FONCTION set".
      "  FONC_INTITULE='".$this->dbConn->analyseSql($fonc_intitule)."'".
      ", FONC_DESC='".$this->dbConn->analyseSql($fonc_desc)."'".
      ($bImageMaj==true ? ", FONC_PJ='".$this->dbConn->analyseSql($strFileName)."'" : "").      
      " where FONC_ID=".$fonc_id;

    $this->dbConn->ExecuteSql($strSql);    
  }	

  /**
   *  Supprime une fonction avancée
   *
   * @param fonc_id  Identifiant de la fonction
   * @return Retourne un bool : true si ok, false sinon
   */
  function del_fonctionAv($fonc_id)
  {
    $strSql = "delete from SIT_FONCTION where FONC_ID=".$fonc_id;
    return $this->dbConn->ExecuteSql($strSql);
  }

  /**
   *  Ajoute une association agent-fonction ou service-fonction
   *        en fonction de strType
   *
   * @param strType  = "Serv" ou "Agent"
   * @param idObjet  Identifiant de strType
   */
  function deladd_assoWithFonctionAv($strType, $idObjet)
  {
    $this->del_assoWithFonctionAv($strType, $idObjet);

    // ajout
    $tabFonc = Request("listeFonc", REQ_POST, array());
    if( is_array($tabFonc) && count($tabFonc)>0 ) {
      foreach($tabFonc as $fonc_id) {
        if( $fonc_id != "-1") {
          if( $strType == "Agent" ) {
            $strSql = "insert into SIT_AGENT_FONCTION (AGENT_ID";
          } else {
            $strSql = "insert into SIT_SERVICE_FONCTION (SERVICE_ID";
          }
          $strSql .= ", FONC_ID) values (".$idObjet.", ".$fonc_id.")";
          $this->dbConn->ExecuteSql($strSql);
        }
      }
    }
  }

  /**
   *  Supprime toutes associations d'un agent ou service en fonction de strType
   *
   * @param strType  = "Serv" ou "Agent"
   * @param idObjet  Identifiant de strType
   */
  function del_assoWithFonctionAv($strType, $idObjet)
  {
    if( $strType == "Agent" )
      $strSql = "delete from SIT_AGENT_FONCTION where AGENT_ID=".$idObjet;
    else
      $strSql = "delete from SIT_SERVICE_FONCTION where SERVICE_ID=".$idObjet;
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Ajoute un nouveau service. Retourne le service_id
   * @see AlkImportAnnuServ
   *
   * @param dsServPere
   * @param service_pere
   * @param service_intitule
   * @param service_intitule_court
   * @param service_adresse
   * @param service_tel
   * @param service_fax
   * @param service_email
   * @param service_web
   * @param service_etat
   * @param inter_id        non utilisé
   * @param cont_id_start
   * @param agent_id
   * @param fonc_id
   * @return int
   */
  function addServiceImport(&$dsServPere, $service_pere, $service_intitule, $service_intitule_court, 
                            $service_adresse, $service_tel, $service_fax, $service_email,
                            $service_web, $service_etat, $cont_id_start, $agent_id, $fonc_id,
                            $dept_id=0, $service_ldap=0, $service_ldapdn="", $service_ldapuid="")
  {
    $service_id = $this->dbConn->GetNextId("SIT_SERVICE", "SERVICE_ID", "SEQ_SIT_SERVICE");
  
    $strfieldAux = "";
    $strValueAux = "";
    if( defined("ALK_B_DIRECTEUR_SERVICE_ANNU") && ALK_B_DIRECTEUR_SERVICE_ANNU==true ) {
      $strfieldAux .= ", AGENT_ID";
      $strValueAux .= ", ".( $agent_id == "-1" ? "null" : $agent_id );
      if( defined("ALK_B_FONCTION_SERV_ANNU") && ALK_B_FONCTION_SERV_ANNU==true ) {
        $strfieldAux .= ", FONC_ID";
        $strValueAux .= ", ".( $fonc_id == "-1" ? "null" : $fonc_id );
      }
    }
  
    // collecte les infos du pere
    if( $drServPere = $dsServPere->getRowIter() ) {
        $service_niveau = $drServPere->getValueName("SERVICE_NIVEAU")+1;
        $service_racine = $drServPere->getValueName("SERVICE_RACINE");
        if( $service_niveau == 1 )
          $service_racine = $service_id;
        $service_arbre = $drServPere->getValueName("SERVICE_ARBRE");
        $service_arbre .= $service_id."-";
        $service_rang = $service_id;
      if( $service_niveau > 1 ) {
        // force la valeur sur celle du pere
        $dept_id      = $drServPere->getValueName("DEPT_ID");
        $service_etat = $drServPere->getValueName("SERVICE_ETAT");
      }
    } else {
      return "-1";
    }
  
    $strSql = "insert into SIT_SERVICE (".
      " SERVICE_ID, SERVICE_INTITULE, SERVICE_INTITULE_COURT, SERVICE_NIVEAU,".
      " SERVICE_ARBRE, SERVICE_PERE, SERVICE_RANG, SERVICE_RACINE, SERVICE_EFFACABLE,".
      " SERVICE_ADRESSE, SERVICE_TEL, SERVICE_FAX, SERVICE_EMAIL, SERVICE_WEB, ".
      " SERVICE_ETAT, CONT_ID_START, SERVICE_DATE_MAJ, DEPT_ID, SERVICE_LDAP, SERVICE_LDAPDN, SERVICE_LDAPUID".$strfieldAux.
      ") values (".$service_id.
      ", '".$this->dbConn->analyseSql($service_intitule)."'".
      ", '".$this->dbConn->analyseSql($service_intitule_court)."'".
      ", ".$service_niveau.
      ", '".$this->dbConn->analyseSql($service_arbre)."'".
      ", ".$service_pere.
      ", ".$service_rang.
      ", ".$service_racine.
      ", 1".
      ", '".$this->dbConn->analyseSql($service_adresse)."'".
      ", '".$this->dbConn->analyseSql($service_tel)."'".
      ", '".$this->dbConn->analyseSql($service_fax)."'".
      ", '".$this->dbConn->analyseSql($service_email)."'".
      ", '".$this->dbConn->analyseSql($service_web)."'".
      ", ".$service_etat.
      ", ".$cont_id_start.
      ", ".$this->dbConn->getDateCur().
      ", ".$dept_id.
      ", ".$service_ldap.
      ", '".$this->dbConn->analyseSql($service_ldapdn)."'".
      ", '".$this->dbConn->analyseSql($service_ldapuid)."'".
      $strValueAux.
      ")";
  
    $this->dbConn->ExecuteSql($strSql);
    return $service_id;
  }

  /**
   * Mise à jour d'un service
   * @see AlkImportAnnuServ
   *
   * @param service_id
   * @param service_intitule
   * @param service_intitule_court
   * @param service_adresse
   * @param service_tel
   * @param service_fax
   * @param service_email
   * @param service_web
   * @param cont_id_start
   * @param agent_id
   * @param fonc_id
   */
  function updateServiceImport($service_id, $service_intitule, $service_intitule_court, 
                               $service_adresse, $service_tel, $service_fax, $service_email,
                               $service_web, $cont_id_start, $agent_id, $fonc_id, $service_etat="-1",
                               $dept_id=-1, $service_ldap=0, $service_ldapdn="", $service_ldapuid="")
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    
    $strfieldValueAux = "";
    if( defined("ALK_B_DIRECTEUR_SERVICE_ANNU") && ALK_B_DIRECTEUR_SERVICE_ANNU==true ) {
      $strfieldValueAux .= ", AGENT_ID=".( $agent_id == "-1" ? "null" : $agent_id );
      if( defined("ALK_B_FONCTION_SERV_ANNU") && ALK_B_FONCTION_SERV_ANNU==true ) {
        $strfieldValueAux .= ", FONC_ID=".( $fonc_id == "-1" ? "null" : $fonc_id );
      }
    }
  
    $strSql = "update SIT_SERVICE set ".
      "  SERVICE_INTITULE='".$this->dbConn->analyseSql($service_intitule)."'".
      ", SERVICE_INTITULE_COURT='".$this->dbConn->analyseSql($service_intitule_court)."'".
      ", SERVICE_ADRESSE='".$this->dbConn->analyseSql($service_adresse)."'".
      ", SERVICE_TEL='".$this->dbConn->analyseSql($service_tel)."'".
      ", SERVICE_FAX='".$this->dbConn->analyseSql($service_fax)."'".
      ", SERVICE_EMAIL='".$this->dbConn->analyseSql($service_email)."'".
      ", SERVICE_WEB='".$this->dbConn->analyseSql($service_web)."'".
      $strfieldValueAux.
      ( $cont_id_start!="" ? ", CONT_ID_START=".$cont_id_start : "" ).
      ", SERVICE_DATE_MAJ=".$this->dbConn->getDateCur().
      ( $user_id == ALK_USER_ID_ADMINALK && $dept_id!="-1" 
        ? ", DEPT_ID=".$dept_id
        : "" ).
      ( $user_id == ALK_USER_ID_ADMINALK && $service_etat!="-1" 
        ? ", SERVICE_ETAT=".$service_etat
        : "" ).
      ", SERVICE_LDAP=".$service_ldap.
      ", SERVICE_LDAPDN='".$this->dbConn->analyseSql($service_ldapdn)."'".
      ", SERVICE_LDAPUID='".$this->dbConn->analyseSql($service_ldapuid)."'".
      "  where SERVICE_ID=".$service_id;
  
    $this->dbConn->ExecuteSql($strSql);
  }
   
  /**
   * Création d'un service et retourne l'id du service créé
   * @param tabQuery  tableau contenant les propriétés du service à créer
   * @return int
   */
  public function addService($tabQuery, $strPathUpload)
  {
    $tabValue = $tabQuery["field"];
    $tabValue["SERVICE_ID"] = array(ALK_SQL_NUMBER, $this->dbConn->getNextId("SIT_SERVICE", "SERVICE_ID", "SEQ_SIT_SERVICE"));
    
    if( isset($tabValue["RESPONSABLE"]) ) {
      unset($tabValue["RESPONSABLE"]); 
    }
    $tabValue["SERVICE_DATE_CREATION"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    //$tabRang = array("field_pere" => "SERVICE_PERE", "field_rang" => "SERVICE_RANG");
    $tabArbre = array("field_id" => "SERVICE_ID", "field_pere" => "SERVICE_PERE", "field_niveau" => "SERVICE_NIVEAU", 
                      "field_racine" => "SERVICE_RACINE", "field_arbre" => "SERVICE_ARBRE");
    $strFileName = doUpload("service_logo", $tabValue["SERVICE_ID"]."_", $strPathUpload);
    if( is_string($strFileName) ) {
    	$tabValue["SERVICE_LOGO"][1] = $strFileName;
    } else {
    	unset($tabValue["SERVICE_LOGO"]);
    }
    $this->setHierarchyForQuery($tabValue, "SIT_SERVICE", $tabArbre);
    
    $strSql = "insert into SIT_SERVICE ".$this->getPartInsertSql($tabValue);
    $this->dbConn->ExecuteSql($strSql);
    
    return $tabValue["SERVICE_ID"][1];
  }

  /**
   * Création d'un service et retourne l'id du service créé
   * @param service_id identifiant du service à modifier
   * @param tabQuery   tableau contenant les propriétés du service à créer
   * @return int
   */
  public function updateService($service_id, $tabQuery, $strPathUpload)
  {
    $tabValue = $tabQuery["field"];
    
    $strFileName = doUpload("service_logo", $service_id."_", $strPathUpload);
    if( is_string($strFileName) ) {
    	$tabValue["SERVICE_LOGO"][1] = $strFileName;
    } else {
    	unset($tabValue["SERVICE_LOGO"]);
    }  
    $strSql = "update SIT_SERVICE set ".$this->getPartUpdateSql($tabValue)." where SERVICE_ID=".$service_id;
    $this->dbConn->ExecuteSql($strSql);
  }
    
  /**
   * Suppression d'un service
   * @see AlkImportAnnuServ
   *
   * @param service_id
   * @return boolean
   */
  public function delService($service_id)
  {
    $strSql = "delete from SIT_SERVICE where SERVICE_ID=".$service_id;
    $bRes = $this->dbConn->ExecuteSql($strSql, false);
    return $bRes;
  }
    
  /**
   * Suppression d'un service
   * @see AlkImportAnnuServ
   *
   * @param service_id identifiant du service
   * @param manager_id identifiant du chef de service
   */
  public function addServiceManager($service_id, $manager_id)
  {
    $strSql = "update SIT_SERVICE set".
      " AGENT_ID=".$manager_id.
      " where SERVICE_ID=".$service_id;

    $this->dbConn->ExecuteSql($strSql);           
  }
  
  // ************ gestion des abonnés ************ //
    
  /**
   *  Vérifie abonne_pseudo et retourne une nouvelle proposition si non unique
   * @param idAbonne  Identifiant de l'abonne
   * @param strPseudo Pseudo proposé
   * @param lgMax    Longueur max du pseudo
   */
  public function getStrVerifyPseudo($idAbonne, $strPseudo, $lgMax)
  {
    $iLgPseudo = strlen($strPseudo);
    $iLgPseudoNum = $strPseudo+1;
    if( $iLgPseudoNum > $lgMax ) $iLgPseudoNum = $lgMax;

    $strSql = "select a.nbPseudoEgal, b.nbPseudoSemblable, c.numMax ".
      " from ".
      "  (select COUNT(ABONNE_PSEUDO) as nbPseudoEgal ".
      "   from SIT_ABONNE ".
      "   where ABONNE_ID<>".$idAbonne.
      "     and ".$this->dbConn->GetLowerCase("ABONNE_PSEUDO")."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strPseudo)."'").") a,".
      
      "  (select COUNT(ABONNE_PSEUDO) as nbPseudoSemblable ".
      "   from SIT_ABONNE ".
      "   where LENGTH(ABONNE_PSEUDO)=".$iLgPseudoNum.
      "     and ".$this->dbConn->GetLowerCase($this->dbConn->GetSubstring("ABONNE_PSEUDO", 0, $iLgPseudo))."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strPseudo)."'").
      "     and ASCII(".$this->dbConn->GetSubstring("ABONNE_PSEUDO", $iLgPseudo, 1).")>48 ".
      "     and ASCII(".$this->dbConn->GetSubstring("ABONNE_PSEUDO", $iLgPseudo, 1).")<58".
      "  ) b,".

      "  (select max(ASCII(".$this->dbConn->GetSubstring("ABONNE_PSEUDO", $iLgPseudo, 1)."))-48 as numMax ".
      "   from SIT_ABONNE ".
      "   where LENGTH(ABONNE_PSEUDO)=".$iLgPseudoNum.
      "     and ".$this->dbConn->GetLowerCase($this->dbConn->GetSubstring("ABONNE_PSEUDO", 0, $iLgPseudo))."=".$this->dbConn->GetLowerCase("'".$this->dbConn->analyseSql($strPseudo)."'").
      "     and ASCII(".$this->dbConn->GetSubstring("ABONNE_PSEUDO", $iLgPseudo, 1).")>48 ".
      "     and ASCII(".$this->dbConn->GetSubstring("ABONNE_PSEUDO", $iLgPseudo, 1).")<58".
      "  ) c";
    
    $dsPseudo = $this->dbConn->InitDataSet($strSql);
    $nbPseudo = 0;
    $numMax = 1;
    
    if( $drPseudo = $dsPseudo->getRowIter() ) {
      $nbPseudoIdent     = $drPseudo->getValueName("NBPSEUDOEGAL");
      $nbPseudoSemblable = $drPseudo->getValueName("NBPSEUDOSEMBLABLE");
      $numMax           = $drPseudo->getValueName("NUMMAX");
      if( $nbPseudoIdent     == "" ) $nbPseudoIdent = 0;
      if( $nbPseudoSemblable == "" ) $nbPseudoSemblable = 0;
      if( $numMax           != "" ) {
        $numMax++; 
      } else {
        $numMax = 1;
      }

      $nbPseudo = $nbPseudoIdent+$nbPseudoSemblable;
    }
    
    if( $nbPseudo>0 ) {
      $strPseudo .= $numMax;
    }
    
    return strtolower($strPseudo);      
  }
    
  /**
   * Ajout d'un abonné. Si l'ajout 
   * Retourne l'identifiant de l'abonné créé si ok, retourne -1 si la création a échoué (dû au doublon d'adresse mail)
   *
   * @param agent_id    identifiant de l'agent
   * @param strNom      nom de l'abonné
   * @param strPrenom   prénom de l'abonné
   * @param strMail     mail de l'abonné
   * @param bValide     =1 pour valide (par défaut), =0 pour non valide
   * @param strAdresse  adresse (vide par défaut)
   * @param strCp       code postal
   * @param strVille    ville
   * @param strPays     pays
   * @param strTel      tel
   * @param strMdp      mot de passe en clair
   * @param strPseudo   pseudo
   * @param abonne_desc description de l'abonné
   * @return int
   */
  public function addAbonne( $agent_id, $strNom, $strPrenom, $strMail, $bValide=1, 
                             $strAdresse="", $strCp="", $strVille="", $strPays="", $strTel="", $strMdp="", $strPseudo="", $abonne_desc="")
  {
    $abonne_id = $this->dbConn->getNextId("SIT_ABONNE", "ABONNE_ID", "SEQ_SIT_ABONNE");
    
    if( $strPseudo != "" ) { 
      $strPseudo = $this->getStrVerifyPseudo($abonne_id, $strPseudo, ALK_ANNU_LG_MAX_PSEUDO);
    }
    
    $tabQuery["ABONNE_ID"]         = array(ALK_SQL_NUMBER, $abonne_id);
    $tabQuery["AGENT_ID"]          = array(ALK_SQL_NUMBER, $agent_id);
    $tabQuery["ABONNE_NOM"]        = array(ALK_SQL_TEXT, $strNom);
    $tabQuery["ABONNE_PRENOM"]     = array(ALK_SQL_TEXT, $strPrenom);
    $tabQuery["ABONNE_PSEUDO"]     = array(ALK_SQL_TEXT, $strPseudo);
    $tabQuery["ABONNE_MAIL"]       = array(ALK_SQL_TEXT, $strMail);
    $tabQuery["ABONNE_VALIDE"]     = array(ALK_SQL_NUMBER, $bValide);
    $tabQuery["ABONNE_ADRESSE"]    = array(ALK_SQL_TEXT, $strAdresse);
    $tabQuery["ABONNE_CP"]         = array(ALK_SQL_TEXT, $strCp);
    $tabQuery["ABONNE_VILLE"]      = array(ALK_SQL_TEXT, $strVille);
    $tabQuery["ABONNE_PAYS"]       = array(ALK_SQL_TEXT, $strPays);
    $tabQuery["ABONNE_TEL"]        = array(ALK_SQL_TEXT, $strTel);
    $tabQuery["ABONNE_PWD"]        = array(ALK_SQL_TEXT, $strMdp);
    $tabQuery["ABONNE_DATE_INSCR"] = array(ALK_SQL_DATETIME, $this->dbConn->GetDateCur());
    $tabQuery["ABONNE_DESC"]       = array(ALK_SQL_TEXT, $abonne_desc);
    
    $strSql = "insert into SIT_ABONNE ".$this->getPartInsertSql($tabQuery);
    $bRes = $this->dbConn->executeSql($strSql, false);  
       
    return ( $bRes ? $abonne_id : -1 );
  }
  
  /**
   * Mise à jour d'un abonné soit par rapport à abonne_id, soit par rapport à agent_id si abonne_id=-1
   *
   * @param abonne_id   identifiant de l'abonné, peut valoir -1 si agent_id<>null
   * @param agent_id    identifiant de l'agent
   * @param strNom      nom de l'abonné
   * @param strPrenom   prénom de l'abonné
   * @param strMail     mail de l'abonné
   * @param iValide     =1 pour valide (par défaut), =0 pour non valide
   * @param strAdresse  adresse (vide par défaut)
   * @param strCp       code postal
   * @param strVille    ville
   * @param strPays     pays
   * @param strTel      tel
   * @param strMdp      mot de passe en clair
   * @param strPseudo   pseudo
   * @param abonne_desc description de l'abonné
   * @return boolean
   */
  public function updateAbonne($tabQuery, $abonne_id, $agent_id, $strNom="", $strPrenom="", $strMail="", $iValide=1, 
                               $strAdresse="", $strCp="", $strVille="", $strPays="", $strTel="", $strMdp="", $strPseudo="", $abonne_desc="")
  {
    if( empty($tabQuery) ) {
      if( $abonne_id != "-1" ) {
        $tabQuery["AGENT_ID"]      = array(ALK_SQL_NUMBER, $agent_id);
      }
      $tabQuery["ABONNE_NOM"]      = array(ALK_SQL_TEXT, $strNom);
      $tabQuery["ABONNE_PRENOM"]   = array(ALK_SQL_TEXT, $strPrenom);
      if( $strPseudo != "" ) {
        $tabQuery["ABONNE_PSEUDO"] = array(ALK_SQL_TEXT, $strPseudo);
      }
      if( $strMail != "" ) {
        $tabQuery["ABONNE_MAIL"]   = array(ALK_SQL_TEXT, $strMail);
      }
      if( $iValide != -1 ) {
        $tabQuery["ABONNE_VALIDE"]   = array(ALK_SQL_NUMBER, $iValide);
      }
      $tabQuery["ABONNE_ADRESSE"]  = array(ALK_SQL_TEXT, $strAdresse);
      $tabQuery["ABONNE_CP"]       = array(ALK_SQL_TEXT, $strCp);
      $tabQuery["ABONNE_VILLE"]    = array(ALK_SQL_TEXT, $strVille);
      $tabQuery["ABONNE_PAYS"]     = array(ALK_SQL_TEXT, $strPays);
      $tabQuery["ABONNE_TEL"]      = array(ALK_SQL_TEXT, $strTel);
      $tabQuery["ABONNE_DESC"]     = array(ALK_SQL_TEXT, $abonne_desc);
      if( $strMdp != "" ) {
        $tabQuery["ABONNE_PWD"]    = array(ALK_SQL_TEXT, $strMdp);
      }
      if( $iValide == "0" ) {
        $tabQuery["ABONNE_DATE_DESINSCR"] = array(ALK_SQL_DATETIME, $this->dbConn->GetDateCur());
      }
    } else {
      unset($tabQuery["ABONNE_ID"]);
      if( $abonne_id == "-1" ) {
        unset($tabQuery["AGENT_ID"]);
      }
      if( isset($tabQuery["ABONNE_DATE_INSCR"]) ) unset($tabQuery["ABONNE_DATE_INSCR"]);
      if( isset($tabQuery["ABONNE_DATE_DESINSCR"]) ) unset($tabQuery["ABONNE_DATE_DESINSCR"]);
      if( isset($tabQuery["ABONNE_VALIDE_ORIG"]) ) {
        if( $tabQuery["ABONNE_VALIDE_ORIG"][1] == "1" && $tabQuery["ABONNE_VALIDE"][1] == "0" ) {
          // désabonnement
          $tabQuery["ABONNE_DATE_DESINSCR"] = array(ALK_SQL_DATETIME, $this->dbConn->GetDateCur());
        } 
        unset($tabQuery["ABONNE_VALIDE_ORIG"]);
      }
    }
        
    $strSql = "update SIT_ABONNE set ".$this->getPartUpdateSql($tabQuery).
      " where ".
      ( $abonne_id != -1 
        ? "ABONNE_ID=".$abonne_id 
        : "AGENT_ID=".$agent_id );
    return $this->dbConn->executeSql($strSql);
  }
    
  /**
   * Suppression d'un abonné. 
   * Ne pas appelé cette méthode directement mais plutot : oAppliAnnu->delAbonne()
   *
   * @param abonne_id  identifiant de l'abonné ou liste d'identifiants séparés par une virgule
   * @return boolean
   */
  public function delAbonne($abonne_id)
  {
    $strSql = "delete from SIT_ABONNE_LISTE where ABONNE_ID in (".$abonne_id.")";
    $this->dbConn->executeSql($strSql);
          
    // si il n'est plus rattaché à d'autre appli, on le supprime
    // sélection des abonnés
    $tabAboIdNotDel = array();
    $strSql = "select ABONNE_ID from SIT_ABONNE where ABONNE_ID in (".$abonne_id.")";
    $dsAbo = $this->dbConn->initDataSet($strSql);
    while( $drAbo = $dsAbo->getRowIter() ) {
      $idAbonne = $drAbo->getValueName("ABONNE_ID");

      // tentative de suppression
      $strSql = "delete from SIT_ABONNE where ABONNE_ID=".$idAbonne;
      $bRes = $this->dbConn->ExecuteSql($strSql, false);
      if( !$bRes ) {
        $tabAboIdNotDel[] = $idAbonne;
      }
    }

    if( !empty($tabAboIdNotDel) ) {
      // la tentative de suppression a échoué
      // on préserve juste le pseudo et la fiche pour maintenance la propriété des infos 
      $strSql = "update SIT_ABONNE set ABONNE_NOM='', ABONNE_PRENOM='', ABONNE_MAIL='', ABONNE_VALIDE=-1 " .
        " where ABONNE_ID in (".implode(",", $tabAboIdNotDel).")";
      $this->dbConn->ExecuteSql($strSql);
    }
  }
  
  /**
   * effectue une action sur les fiches d'abonnés
   * les actions :
   *   =1 activation des fiches en attente ou désactivées
   *   =2 désactivation des fiches en attente ou actives
   *   =3 réactivation des fiches désactivées
   *   =4 suppression des fiches désactivées ou en attente d'activation
   *   =5 retrait des abonnés de la liste
   * @param iAction     identifiant de l'action
   * @param abonne_id   string chaine des identifiants des abonnés séparés par des virgules
   */
  public function setActionToAbonne($iAction, $abonne_id=-1, $liste_id=-1)
  {
    $strSql = "";
    switch( $iAction ) {
    case "1": // activation des fiches en attente ou désactivées
      $strSql = "update SIT_ABONNE set" .
        "  ABONNE_VALIDE=1" .
        ", ABONNE_DATE_DESINSCR=null" .
        " where (ABONNE_VALIDE=0 or ABONNE_VALIDE is null) and ABONNE_ID in (".$abonne_id.")";
      break;
      
    case "2": // désactivation des fiches en attente ou actives
      $strSql = "update SIT_ABONNE set" .
        "  ABONNE_VALIDE=0".
        ", ABONNE_DATE_DESINSCR=".$this->dbConn->getDateCur().
        " where ABONNE_DATE_DESINSCR is null and ABONNE_ID in (".$abonne_id.")";
      break;
      
    case "3": // réactivation des fiches désactivées
      $strSql = "update SIT_ABONNE set" .
        "  ABONNE_VALIDE=1" .
        ", ABONNE_DATE_DESINSCR=null" .
        " where ABONNE_VALIDE=0 and ABONNE_DATE_DESINSCR is not null and ABONNE_ID in (".$abonne_id.")";
      break;
      
    case "4": // suppression des fiches désactivées ou en attente d'activation
      $tabAboId = array();
      $strSql = "select ABONNE_ID from SIT_ABONNE where ABONNE_VALIDE=0 and ABONNE_ID in (".$abonne_id.")";
      $dsAbonne = $this->dbConn->initDataset($strSql);
      while( $drAbonne = $dsAbonne->getRowIter() ) {
        $tabAboId[] = $drAbonne->getValueName("ABONNE_ID");
      }
      if( !empty($tabAboId) ) {
        $oAppliAnnu = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
        $oAppliAnnu->delAbonne(implode(",", $tabAboId));
      }
      $strSql = "";
      break;
    
    case "5": // retrait des abonnés de la liste
      $this->delAbonnement($abonne_id, $liste_id);
      break;
    }
    if( $strSql != "" ) {
      $this->dbConn->executeSql($strSql);
    }
  } 
  
  /**
   * ajoute les utilisateurs d'un espace à une liste
   * @param $cont_id    identifiant de l'espace
   * @param $liste_id   identifiant de la liste
   */
  public function syncEspaceListe($cont_id=-1, $liste_id=-1)
  {
    $this->dbConn->initTransaction();
    $iRes = 1;
    
    $strSql = "delete from SIT_ABONNE_LISTE where ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (select AGENT_ID from SIT_AGENT_CONT where CONT_ID=".$cont_id.")) and LISTE_ID=".$liste_id;
    $iRes &= $this->dbConn->executeSql($strSql, false);
    //echo $strSql."<br/>";
    
    $strSql = "insert into SIT_ABONNE_LISTE (LISTE_ID, ABONNE_ID) select ".$liste_id." as LISTE_ID, ABONNE_ID from SIT_ABONNE where AGENT_ID in (select AGENT_ID from SIT_AGENT_CONT where CONT_ID=".$cont_id.")";
    $iRes &= $this->dbConn->executeSql($strSql, false);
    //echo $strSql."<br/>";
    
    if ( $iRes == 1 ) {
      $this->dbConn->commitTransaction();
      return true;
    } else {
      $this->dbConn->rollBackTransaction();
      return false;
    }
  }
  
  /**
   * Ajout d'utilisateurs à une liste
   * @param user_id     identifiant ou liste d'id d'utilisateurs spéraré par des virgules
   * @param liste_id    identifiant de la liste
   * @return true ok, false sinon
   */
  public function addUserToListe($user_id=-1, $liste_id=-1)
  {
    $this->dbConn->initTransaction();
    $iRes = 1;
    
    $strSql = "delete from SIT_ABONNE_LISTE where ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$user_id.")) and LISTE_ID=".$liste_id;
    $iRes &= $this->dbConn->executeSql($strSql, false);
    //echo $strSql."<br/>";
    
    $strSql = "insert into SIT_ABONNE_LISTE (LISTE_ID, ABONNE_ID) select ".$liste_id." as LISTE_ID, ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$user_id.")";
    $iRes &= $this->dbConn->executeSql($strSql, false);
    //echo $strSql."<br/>";
    
    if ( $iRes == 1 ) {
      $this->dbConn->commitTransaction();
      return true;
    } else {
      $this->dbConn->rollBackTransaction();
      return false;
    }
  }
  
  /**
   * Suppression d'utilisateurs d'une liste
   * @param user_id     identifiant ou liste d'id d'utilisateurs spéraré par des virgules
   * @param liste_id    identifiant de la liste
   * @return true ok, false sinon
   */
  public function delUserFromListe($user_id=-1, $liste_id=-1)
  {
    $this->dbConn->initTransaction();
    $iRes = 1;
    
    $strSql = "delete from SIT_ABONNE_LISTE where ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$user_id.")) and LISTE_ID=".$liste_id;
    $iRes &= $this->dbConn->executeSql($strSql, false);
    
    if ( $iRes == 1 ) {
      $this->dbConn->commitTransaction();
      return true;
    } else {
      $this->dbConn->rollBackTransaction();
      return false;
    }
  }
  
  /**
   * Ajout d'un abonnement d'un abonné à une liste
   * Ne fait rien si l'abonnement à la liste existe déjà pour cet abonné
   * @param abonne_id  identifiant de l'abonné 
   * @param liste_id   identifiant de la liste à associer à l' abonné, peut prendre la valeur -1 pour ne pas prendre en compte ce paramètre
   * @param strDate    date de l'abonnement (optionnel, affecte la date courante si non fourni)
   * @param bAuto      booléen true pour abonnement automatique (hérité d'une autre liste), false sinon (défaut=false)
   */
  public function addAbonnement($abonne_id, $liste_id, $strDateCur="", $bAuto=false)
  {
    if( $liste_id != "-1" ) {
      $strSql = "select * from SIT_ABONNE_LISTE where LISTE_ID=".$liste_id." and ABONNE_ID=".$abonne_id;
      $oDsAbonneListe = $this->dbConn->initDataSet($strSql);
      if ( $oDsAbonneListe->getCountTotDr() == 0 ) {
        $strSql = "insert into SIT_ABONNE_LISTE (LISTE_ID, ABONNE_ID, ABONNELISTE_DATE, ABONNELISTE_AUTO) values (".$liste_id.",".$abonne_id.",".( $strDateCur == "" ? $this->dbConn->getDateCur() : $strDateCur ).", ".( $bAuto ? 1 : 0 ).")";
        $this->dbConn->executeSql($strSql);
      }
    }
  }
  
  /**
   * Suppression d'un abonnement
   *
   * @param abonne_id       identifiant de l'abonnée ou liste d'identifiant
   * @param liste_id        identifiant de la liste de désabonnement
   * @param agent_id        identifiant de l'agent, =-1 par défaut 
   * @param strDate         date de l'abonnement, supprime l'abonnement uniquement si correspond à la date effective de l'abonnement sinon ne fait rien (chaîne vide par défaut, test non effectué)
   * @return boolean
   */
  public function delAbonnement($abonne_id, $liste_id, $agent_id=-1, $strDate="")
  {
    $strWhere2 = ( $agent_id != "-1" && $agent_id != ALK_USER_ID_INTERNET
                   ? "select ABONNE_ID from SIT_ABONNE where AGENT_ID=".$agent_id
                   : $abonne_id );
    
    if( $liste_id != "-1" ) {
      $strSql = "delete from SIT_ABONNE_LISTE where ABONNE_ID in (".$strWhere2.")" .
        " and LISTE_ID=".$liste_id.
        ( $strDate != "" 
          ? " and ABONNELISTE_DATE = '".$strDate."'"
          : "" );
      $this->dbConn->ExecuteSql($strSql);
    }
  }  
  
  /**
   * Suppression de tous les abonnements à une ou plusieurs listes
   * @param liste_id        identifiant de la ou des listes de désabonnement (séparés par des virgules)
   * @param $AutoOnly       booléen à true pour ne supprimer que les abonnements automatiques (hérités d'une autre liste), false sinon (défaut=false)
   */
  public function delAllAbonnement($liste_id, $bAutoOnly=false)
  {
    $strSql = "delete from SIT_ABONNE_LISTE where LISTE_ID in (".$liste_id.")".
      ( $bAutoOnly 
        ? " and ABONNELISTE_AUTO=1" 
        : "" );
    $this->dbConn->ExecuteSql($strSql);
  }
  
  // ************ gestion des liste d'abonnés ************ //

  /**
   *  Ajoute une liste de diffusion
   *
   * @param liste_intitule    tableau des intitulés par langue
   * @param liste_desc        tableau des description par langue
   * @param liste_prive       type de la portée de la liste privé/espace=0, privé/sit=3, public/espace=1, public/sit=2
   * @param agent_id          identifiant de l'agent créateur
   * @param cont_id           identifiant de l'espace
   * @param liste_id_to_copy  =-1, 
   * @param liste_type        =0 pour mes listes
   *                          =1 pour liste synd
   *                          =2 pour liste d'abonnement aux catégories et données
   *                          =3 liste d'abonnement explicite (lettre d'information)
   *                          =4 liste des utilisateurs de l'espace
   * @param liste_valide
   * @return Retourne un entier : identifiant de la liste créée
   */
  public function addListe($liste_intitule, $liste_desc, $liste_prive, $agent_id, $cont_id, $liste_id_to_copy=-1, $liste_type=0, $liste_valide=1)
  {
    $liste_id = $this->dbConn->GetNextId("SIT_LISTE", "LISTE_ID", "SEQ_SIT_LISTE");

    $strIntituleChamp = "";
    $strIntituleValues = "";
    $strDescChamp = "";
    $strDescValues = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntituleChamp  .= ", LISTE_INTITULE".$tabLg["bdd"];
      $strDescChamp  .= ", LISTE_DESC".$tabLg["bdd"];
      if( $liste_id_to_copy == -1 ) {
        $strDescValues .= ", '".$this->dbConn->analyseSql($liste_desc[$key])."'";
        $strIntituleValues .= ", '".truncFileName($this->dbConn->analyseSql($liste_intitule[$key]), 80)."'";
      }
    }

    if( $liste_prive == "2" || $liste_prive == "3" )
      $cont_id = "0";

    if( $liste_id_to_copy == -1 ) {
      $strSql = "insert into SIT_LISTE (LISTE_ID, LISTE_PRIVE, AGENT_ID, CONT_ID, LISTE_TYPE, LISTE_VALIDE".
        $strIntituleChamp.$strDescChamp.") values (".$liste_id.
        ", ".$liste_prive.
        ", ".$agent_id.
        ", ".$cont_id.
        ", ".$liste_type.
        ", ".$liste_valide.
        $strIntituleValues.
        $strDescValues.
        ")";
    } else {
      $strSql = "insert into SIT_LISTE (LISTE_ID, LISTE_PRIVE, AGENT_ID, CONT_ID, LISTE_TYPE".
        $strIntituleChamp.$strDescChamp.") select ".$liste_id.", ".$liste_prive.", ".
        $agent_id.", ".$cont_id.", LISTE_TYPE".$strIntituleChamp.$strDescChamp.
        " from SIT_LISTE where LISTE_ID=".$liste_id_to_copy;
    }
    
    $this->dbConn->ExecuteSql($strSql);
    return $liste_id;
  }
  
  /**
   *  Modifie une liste de diffusion
   *
   * @param liste_id          Identifiant de la liste à modifier
   * @param liste_intitule    tableau des intitulés par langue
   * @param liste_desc        tableau des description par langue
   * @param liste_prive       type de la portée de la liste privé/espace=0, privé/sit=3, public/espcae=1, public/sit=2
   * @param agent_id          identifiant de l'agent créateur
   * @param cont_id           identifiant de l'espace
   * @param liste_type        =0 pour mes listes
   *                          =1 pour liste synd
   *                          =2 pour liste d'abonnement aux catégories et données
   *                          =3 liste d'abonnement explicite (lettre d'information)
   *                          =4 liste des utilisateurs de l'espace
   * @param liste_valide
   */
  public function updateListe($liste_id, $liste_intitule, $liste_desc, $liste_prive, $agent_id, $cont_id, $liste_type="-1", $liste_valide="-1")
  {
    $strIntitule = "";
    $strDesc = "";
    $strAgent = "";
    foreach($this->tabLangue as $key => $tabLg) {
      $strIntitule  .= ", LISTE_INTITULE".$tabLg["bdd"]."='".$this->dbConn->analyseSql($liste_intitule[$key])."'";
      $strDesc .= ", LISTE_DESC".$tabLg["bdd"]."='".$this->dbConn->analyseSql($liste_desc[$key])."'";
    }

    // force l'espace à null pour les portées à tout le sit
    // possibilité de modifier l'agent si portée à tout le sit
    if( $liste_prive == "2" || $liste_prive == "3" ) {
      $cont_id = "0";
      $strAgent = ", AGENT_ID=".$agent_id;
    }

    $strSql = "update SIT_LISTE set".
      "  LISTE_PRIVE=".$liste_prive.
      ( $liste_type != "-1"
        ? ", LISTE_TYPE=".$liste_type
        : "" ).
      ( $liste_valide != "-1"
        ? ", LISTE_VALIDE=".$liste_valide
        : "" ).
      ", CONT_ID=".$cont_id.
        $strIntitule.
        $strDesc.
        $strAgent.
      " where LISTE_ID=".$liste_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Modifie la validation d'une liste
   * @param liste_id     identifiant de la liste à modifier
   * @param liste_valide =0 pour non valide, =1 pour valide
   */
  public function setListeValide($liste_id, $liste_valide="1")
  {
    if( $liste_valide != "1" ) {
      $liste_valide = "0";
    }
    $strSql = "update SIT_LISTE set".
      " LISTE_VALIDE=".$liste_valide.
      " where LISTE_ID=".$liste_id;
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Modifie la propriété lock d'une liste de diffusion
   *
   * @param liste_id         Identifiant de la liste à modifier
   * @param bLock            valeur du champ LOCK
   */
  public function maj_lockListeDif($liste_id, $bLock)
  {    
      $strSql = "update SIT_LISTE set".
      " LISTE_LOCK=".$bLock.
      " where LISTE_ID in (".$liste_id.")";
    $this->dbConn->ExecuteSql($strSql);
  } 

  /**
   * Supprime une liste de diffusion
   *
   * @param liste_id  Identifiant de la liste à supprimer ou liste d'identifiants
   */
  public function delListe($liste_id)
  {
    $strSql = "select * from SIT_LISTE where LISTE_ID in (".$liste_id.")";
    $dsListe = $this->dbConn->initDataset($strSql);
    if ($drListe = $dsListe->getRowIter()){
      $liste_lock  = $drListe->getValueName("LISTE_LOCK");
      $liste_type  = $drListe->getValueName("LISTE_TYPE");

      $strSql = "delete from SIT_LISTE_SLISTE" .
        " where LISTE_ID in (".$liste_id.")".
        ( $liste_type == 2 
          ? " or SLISTE_ID in (".$liste_id.")" 
          : "" );
      $this->dbConn->ExecuteSql($strSql);

      if( $liste_lock == "1" ) {
        //Suppression cachée de la liste  
        $strSql = "update SIT_LISTE set LISTE_VALIDE=-1 where LISTE_ID in (".$liste_id.")";
        $this->dbConn->ExecuteSql($strSql);
      } else {
        // Suppression complète de la liste
        $strSql = "delete from SIT_ABONNE_LISTE where LISTE_ID in (".$liste_id.")";
        $this->dbConn->ExecuteSql($strSql);
    
        $strSql = "delete from SIT_LISTE where LISTE_ID in (".$liste_id.")";
        $this->dbConn->ExecuteSql($strSql);
      }
    }           
  }

  /**
   * Ajoute un ou plusieurs abonnés à une liste de diffusion
   * Si l'abonné n'existe pas, il est créé avec l'état valide
   * 
   * Si liste_id vaut -1 et strMails ne contient qu'une ligne, la fonction teste l'existance de l'abonné, le créé s'il n'existe pas et retourne son abonne_id
   * retourne la liste des abonne_id associés à la liste
   *
   * @param liste_id  Identifiant de la liste de destination des abonnés
   * @param strMails  Liste d'adresses Mail au format Prenom Nom <adresse mail> ou adresse mail uniquement 1 / ligne
   * @param bDateCur  booléen à true pour ajouter la date d'abonnement à la liste (date courante, défaut), false sinon
   * @return array
   */
  public function addAbonneToListe($liste_id, $strMails, $bDateCur=true)
  {
    $strRegexp = "^([-!#\$%&'*+./0-9=?A-Z^_`a-z{|}~^?])".
      "+@([-!#\$%&'*+/0-9=?A-Z^_`a-z{|}~^?]+\\.)+[a-zA-Z]{2,6}\$";
    
    $tabAbonneId = array();
    $strMails = mb_ereg_replace("\r", "", $strMails);
    $tabMails = explode("\n", $strMails);
    foreach($tabMails as $strNomPrenomMail) {
      $tabNPM = explode("<", $strNomPrenomMail, 2);
      if( count($tabNPM) == 1 ) {
        $strMail = trim($strNomPrenomMail);
        if( !mb_eregi($strRegexp, $strMail) ) continue;
        $strNom = substr($strMail, 0, (strpos($strMail, "@")===false ? strlen($strMail) : strpos($strMail, "@")));
        $strPrenom = "";
      } elseif( count($tabNPM) == 2 ) {
        $strMail = trim(mb_ereg_replace(">", "", $tabNPM[1]));
        if( !mb_eregi($strRegexp, $strMail) ) continue;
        $tabNP = explode(" ", trim($tabNPM[0]), 2);
        if( count($tabNP) == 1 ) {
          $strNom = ( trim($tabNP[0])=="" 
                      ? substr($strMail, 0, (strpos($strMail, "@")===false ? strlen($strMail) : strpos($strMail, "@")))
                      : trim($tabNP[0]));
          $strPrenom = "";
        } elseif( count($tabNP) == 2 ) {
          $strNom = trim($tabNP[1]);
          $strPrenom = trim($tabNP[0]);
        } else {
          $strNom = substr($strMail, 0, (strpos($strMail, "@")===false ? strlen($strMail) : strpos($strMail, "@")));
          $strPrenom = "";
        }
      } else {
        continue;
      }
      
      $strSql = "select ABONNE_ID, AGENT_ID from SIT_ABONNE where ABONNE_MAIL='".$this->dbConn->analyseSql($strMail)."'";
      $dsAbonne = $this->dbConn->initDataSet($strSql);
      if( $drAbonne = $dsAbonne->getRowIter() ) {
        $abonne_id = $drAbonne->getValueName("ABONNE_ID");
        $agent_id = $drAbonne->getValueName("AGENT_ID");
      } else {
        $strPseudo = str_replace("@", ".", $strMail);
        $abonne_id = $this->addAbonne(ALK_USER_ID_INTERNET, $strNom, $strPrenom, $strMail, 1, "", "", "", "", "", "", $strPseudo);
        $agent_id = ALK_USER_ID_INTERNET;
      }
      $tabAbonneId[] = array($abonne_id, trim($strPrenom." ".$strNom), $strMail, $agent_id);

      if( $liste_id != -1 ) {
        $strSql = "insert into SIT_ABONNE_LISTE (ABONNE_ID, LISTE_ID".( $bDateCur ? ", ABONNELISTE_DATE" : "" ).") values (".$abonne_id.", ". $liste_id.( $bDateCur ? ", ".$this->dbConn->getDateCur() : "" ).")";
        $this->dbConn->ExecuteSql($strSql, false);
      }
    }
    return $tabAbonneId;
  }
  
  /**
   * Ajout les abonnés d'une ou plusieurs listes à une autre liste (<liste_id_from> ∪ <liste_id_to> = <liste_id_to>)
   * @param liste_id_from   identifiant d'une ou plusieurs listes (séparés par des virgules)
   * @param liste_id_to     identifiant de la liste d'arrivée
   * @param bCopyDate       booléen true pour copier la date d'abonnement, false pour affecter la date courante
   * @param iAuto           -1 copie l'information d'abonnement automatique des abonnements de la liste de départ vers la liste d'arrivée
   *                        0 affecte l'information d'abonnement non automatique aux abonnements de la liste d'arrivée
   *                        1 affecte l'information d'abonnement automatique aux abonnements de la liste d'arrivée
   */
  public function addAbonnesFromListeToListe($liste_id_from, $liste_id_to, $bCopyDate=true, $iAuto=-1)
  {
    $strSql = "select sa.ABONNE_ID, sal.ABONNELISTE_DATE from SIT_ABONNE sa" .
        " inner join SIT_ABONNE_LISTE sal on sal.ABONNE_ID=sa.ABONNE_ID" .
        " where sal.LISTE_ID in (".$liste_id_from.")";
    $oDsAbonnes = $this->dbConn->initDataset($strSql);
    
    while ( $oDrAbonnes = $oDsAbonnes->getRowIter() ) {
      $abonne_id = $oDrAbonnes->getValueName("ABONNE_ID");
      $date_abonnement = $oDrAbonnes->getValueName("ABONNELISTE_DATE");
      $auto_abonnement = $oDrAbonnes->getValueName("ABONNELISTE_AUTO");
      $this->addAbonnement($abonne_id, $liste_id_to, ( $bCopyDate && $date_abonnement != "" ? "'".$date_abonnement."'" : $this->dbConn->getDateCur() ), ( $iAuto == -1 ? $auto_abonnement : $iAuto ));
    }
  }
  
  /**
   * Supprime les abonnements des abonnés d'une ou plusieurs listes d'une liste (<liste_id_to> - <liste_id_from> = <liste_id_to>)
   * @param liste_id        identifiant de la liste dans laquelle les abonnements sont supprimés
   * @param liste_id_from   identifiant d'une ou plusieurs listes (séparés par des virgules)
   */
  public function delAbonnesListeFromListe($liste_id_to, $liste_id_from)
  {
    $strSql = "delete from SIT_ABONNE_LISTE sal1" .
        " where sal1.ABONNE_ID in ( select sal2.ABONNE_ID from SIT_ABONNE_LISTE sal2 where sal2.LISTE_ID in (".$liste_id_from.") )" .
        " and sal1.LISTE_ID=".$liste_id_to;
    
    $this->dbConn->ExecuteSql($strSql, false);
  }
  
  /**
   * Intersecte une liste d'abonnement avec une ou plusieurs listes (<liste_id_to> ∩ <liste_id_from> = <liste_id_to>)
   * @param liste_id_to     identifiant de la liste de destination de l'intersection, également utilisé pour l'intersection
   * @param liste_id_from   identifiant d'une ou plusieurs listes (séparés par des virgules)
   */
  public function interAbonnesListeAndListe($liste_id_to, $liste_id_from)
  {
    $strSql = "delete from SIT_ABONNE_LISTE sal1" .
        " where sal1.ABONNE_ID not in ( select sal2.ABONNE_ID from SIT_ABONNE_LISTE sal2 where sal2.LISTE_ID in (".$liste_id_from.") )" .
        " and sal1.LISTE_ID=".$liste_id_to;
        
    $this->dbConn->ExecuteSql($strSql, false);
  }
  
  /**
   * Ajoute un abonné associé à un agent de l'annuaire à une liste
   *
   * @param liste_id  identifiant de la liste
   * @param agent_id  identifiant ou liste d'identifiants des agents à abonner
   */
  public function delAddAbonneFromAnnuToListe($liste_id, $agent_id)
  {
    // menage avant insertion
    $strSql = "delete from SIT_ABONNE_LISTE" .
      " where LISTE_ID=".$liste_id.
      "   and ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$agent_id."))";
    $this->dbConn->ExecuteSql($strSql);

    // ajoute l'agent dans la liste
    $strSql = "insert into SIT_ABONNE_LISTE (LISTE_ID, ABONNE_ID) ".
      "select ".$liste_id.", ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$agent_id.")";
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Ajoute un abonné de l'annuaire à une liste
   *
   * @param liste_id    identifiant de la liste
   * @param abonne_id   identifiant ou liste d'identifiants des abonnés
   */
  public function delAddAbonneToListe($liste_id, $abonne_id)
  {
    // menage avant insertion
    $strSql = "delete from SIT_ABONNE_LISTE" .
      " where LISTE_ID=".$liste_id.
      "   and ABONNE_ID in (".$abonne_id.")";
    $this->dbConn->ExecuteSql($strSql);

    // ajoute l'agent dans la liste
    $strSql = "insert into SIT_ABONNE_LISTE (LISTE_ID, ABONNE_ID) ".
      "select ".$liste_id.", ABONNE_ID from SIT_ABONNE where ABONNE_ID in (".$abonne_id.")";
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   *  Duplique la liste de diffusion
   *
   * @param liste_id  identifiant de la liste à dupliquer
   * @param agent_id  identifiant de l'utilisateur connecté
   * @param cont_id   identifiant de l'espace courant
   */
  public function copyListeDif($liste_id, $agent_id, $cont_id)
  {
    // duplique la liste de diffustion
    $liste_id_dup = $this->addListe(array(1 => ""), array(1 => ""), 1, $agent_id, $cont_id, $liste_id);

    // duplique les abonnés : uniquement pour oracle et postgres (cf class queryAnnuActionMysql pour Mysql)
    $strSql = "insert into SIT_ABONNE_LISTE (ABONNE_ID, LISTE_ID)".
      " select ABONNE_ID, ".$liste_id_dup.
      " from SIT_ABONNE_LISTE" .
      " where LISTE_ID=".$liste_id;
    $this->dbConn->ExecuteSql($strSql);
  }

}
?>