<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Pattern
Module fournissant les classes de base Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH."classes/pattern/alkappli.class.php");
require_once(ALK_ALKANET_ROOT_PATH."classes/pattern/alkintoptions.int.php");

/**
 * @package Alkanet_Class_Pattern
 * 
 * @class AlkOptionsManager
 * @brief Cette classe sert de gestionnaire pour la vérification des droits des options d'application.
 *        Une instance de ce gestionnaire est injectée dans chaque application (<b>AlkAppli->optionsManager</b>).
 *        Le gestionnaire ne sera fonctionnel que pour les applications *réellement* instanciées, dont le appli_id est différent de -1.
 *        La gestion des droits sur les options d'application se fait au niveau de la liste des applications.
 *        Pour qu'une application puisse gérer ses options, elle doit surchager la fonction <b>getDeclaredOptions()</b> et renvoyer 
 *        un objet de configuration des options de l'application (classe <b>AlkOptionsBuilder</b>), de cette manière :
 * <pre>
 * class CustomAppli extends AlkAppli {
 * 
 *   public function getDeclaredOptions()
 *   {
 *     $options = parent::getDeclaredOptions(); // ou faire un new AlkOptionsBuilder()
 *     $options->addOption('mon_option_1', 'Ma première option', 'une description longue')
 *             ->addOption('mon_option_2', 'Ma seconde option', 'une description longue');
 *     return $options;
 *   }
 *   
 * }
 * </pre>
 * Les droits sur les options peuvent ensuite être testés avec les fonctions offertes par ce gestionnaire :
 * <pre>
 * if( $this->optionsManager->isGloballyActivated('mon_option_1') ) { ... } // passer par le manager associé à l'application courante
 * if( AlkOptionsManager::getInstance($this)->isGloballyActivated('mon_option_1') ) { ... } // appel static en injectant l'application comme contexte d'exécution
 * if( $this->oAppli->optionsManager->isGranted('mon_option_2') ) { ... } // appel depuis un formulaire par exemple, en testant les droits Acl
 * </pre>
 * 
 * @see AlkOptionsBuilder
 */
class AlkOptionsManager
{

  private static $registry = array();  
  
  protected $options_provider;
  
  protected $query_provider = null;
  
  protected $cached_options = null;
  
  protected $bAclAvailable = false;
  
  /**
   * Constructeur privé
   * @param AlkIntOptionsProvider $options_provider
   */
  private function __construct(AlkIntOptionsProvider $options_provider)
  {
    $this->options_provider = $options_provider;
    $this->bAclAvailable    = (AlkFactory::getModuleName(ALK_ATYPE_ID_ACL) != '') && file_exists(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE.AlkFactory::getModuleName(ALK_ATYPE_ID_ACL));
    if( $this->bAclAvailable ) {
      require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."acl/classes/alkaclmanager.class.php");
    }
    
    // affecter l'objet responsable des requêtes qui implémente AlkIntOptions (AlkAppliEspace)
    if($options_provider instanceof AlkIntOptions ) {
      // éviter d'appeler le AlkFactory::getAppli() sur l'appli qui implémente AlkIntOptions (éviter les références cycliques)
      $this->setQueryProvider( $options_provider );
    } else if( $options_provider->getAppliId() == -1 ) {
      // cas non supporté, il s'agit d'une appli autre que espace mais avec un appli_id à -1
      //trigger_error(__CLASS__." - Impossible d'initialiser le manager d'options avec l'objet de type ".get_class($options_provider), E_USER_ERROR);
    } else {
      // dans tous les autres cas, c'est l'espace qui est le query provider
      $this->setQueryProvider( AlkFactory::getAppli(ALK_ATYPE_ID_ESPACE) );
    }
  }
  
  /**
   * Retourne l'instance du manager d'options global (par défaut) ou dédié à une application (si $appli_id renseigné) 
   * @param AlkIntOptionsProvider $options_provider
   */
  public static function getInstance(AlkIntOptionsProvider $options_provider)
  {
    if( ! array_key_exists($options_provider->getAppliId(), self::$registry) ) {
      //var_dump(__CLASS__.'::'.__FUNCTION__.' registering object '.get_class($options_provider).' with appli_id = '.$options_provider->getAppliId());
      self::$registry[$options_provider->getAppliId()] = new AlkOptionsManager($options_provider);
    }
    
    return self::$registry[$options_provider->getAppliId()];
  }
  
  /**
   * Affecte l'objet qui implémente AlkIntOptions et qui est responsable des requêtes 
   *
   * @param AlkIntOptions $object
   */
  private function setQueryProvider(AlkIntOptions $object)
  {
    $this->query_provider = $object;
  }
  
  /**
   * Retourne le cache des options de l'application
   * Le tableau est indexé avec les alias des options (-> doivent être unique pour une même application)
   * Les options contenues dans le tableau contiennent les clés suivantes :
   *  - option_id : int
   *  - option_alias : string
   *  - option_intitule : string
   *  - option_desc : string
   *  - option_activee : boolean
   *  - option_droits_avances : boolean
   *  - appli_id : int
   *
   * @return array cached options
   */
  private function getCachedOptions()
  {
    if( $this->cached_options === null ) {
      $this->loadOptions();
    }
    return $this->cached_options;
  }
  
  /**
   * Fonction de chargement des options pour l'application courante
   * Se base sur un cache d'options pour éviter de les recharger à chaque appel aux fonctions de vérification des droits
   */
  private function loadOptions()
  {
    $this->cached_options = array();
    
    if( null === $this->query_provider) {
      // si le query provider est null, impossible de charger les options, abandonner...
      return;
    }
    
    // pas d'options sur les applis autres que espace (impl AlkIntOptions) si le appi_id est à -1
    if( $this->options_provider->getAppliId() == -1 && !($this->options_provider instanceof AlkIntOptions) ) {
      // TODO supprimer le warning après une phase de tests suffisante
      trigger_error(__CLASS__.'::'.__FUNCTION__." - pas d'options sur les applis autres que espace (impl AlkIntOptions) si le appi_id est à -1 (TODO: MESSAGE À SUPPRIMER)", E_USER_WARNING);
      return;
    }
    
    $options = $this->options_provider->getDeclaredOptions();
    
    $declared_options = $options->toArray();
    $declared_aliases = array_keys( $declared_options );
    $instanciated_aliases = array();
    
    // chargement des options d'application depuis la base
    $ds = $this->query_provider->getDsOptions( $this->options_provider->getAppliId() );
    // TODO WARNING! utilisation de l'iterateur de AlkDs, vérifier la compatibilité entre versions de modules (pattern 3.5.15)
    foreach( $ds as $dr ) {
      $option = array();
      $option['option_id']             = $dr['OPTION_ID'];
      $option['option_alias']          = $dr['OPTION_ALIAS'];
      $option['option_intitule']       = $dr['OPTION_INTITULE'];
      $option['option_desc']           = $dr['OPTION_DESC'];
      $option['option_activee']        = $dr['OPTION_ACTIVEE'] == '1' ? true : ($dr['OPTION_ACTIVEE'] == '0' ? false : null) ;
      $option['option_droits_avances'] = $dr['OPTION_DROITS_AVANCES'] == '1' ? true : false;
      $option['atype_id']              = $dr['OPTION_ID'];
      $option['atype_intitule']        = $dr['OPTION_ID'];
  
      $this->cached_options[$option['option_alias']] = $option;
      $instanciated_aliases[] = $option['option_alias'];
    }
    // est-ce qu'il faut créer de nouvelles options ?
    $new_options = array_diff($declared_aliases, $instanciated_aliases);
    foreach( $new_options as $alias ) {
      $this->query_provider ->addOption(
          $this->options_provider->getAppliId(), 
          $declared_options[$alias]['alias'], 
          $declared_options[$alias]['intitule'], 
          $declared_options[$alias]['description']
      );
    }
    // est-ce qu'il faut supprimer des options non déclarées ?
    $options_to_remove= array_diff($instanciated_aliases, $declared_aliases);
    foreach( $options_to_remove as $alias) {
      $this->query_provider->removeOption( $this->cached_options[$alias]['option_id'] );
    }
  }
  
  /**
   * Indique si une option est activée de manière globale
   * L'option est retrouvée grâce à son alias
   *
   * @param string $option_alias alias de l'option à tester
   * @return boolean
   */
  public function isGloballyActivated($option_alias)
  {
    $cache = $this->getCachedOptions();
    if ( empty($cache) || false === array_key_exists($option_alias, $cache) ) {
      //trigger_error(get_class($this)." - L'option '$option_alias' n'existe pas pour cette application (".get_class($this->options_provider).", appli_id = ".$this->options_provider->getAppliId().")", E_USER_WARNING);
      return false;
    }
    return isset($cache[$option_alias]['option_activee']) && $cache[$option_alias]['option_activee'] === true;
  }
  
  /**
   * Indique si une option est désactivée de manière globale
   * L'option est retrouvée grâce à son alias
   *
   * @param string $option_alias alias de l'option à tester
   * @return boolean
   */
  public function isGloballyDeactivated($option_alias)
  {
    $cache = $this->getCachedOptions();
    if ( empty($cache) || false === array_key_exists($option_alias, $cache) ) {
      //trigger_error(get_class($this)." - L'option '$option_alias' n'existe pas pour cette application (".get_class($this->options_provider).", appli_id = ".$this->options_provider->getAppliId().")", E_USER_WARNING);
      return false;
    }
    return isset($cache[$option_alias]['option_activee']) && $cache[$option_alias]['option_activee'] === false;
  }
  
  /**
   * Indique si une option a des droits avancés (gérés via les ACL)
   * L'option est retrouvée grâce à son alias
   *
   * @param string $option_alias alias de l'option à tester
   * @return boolean
   */
  public function hasAcl($option_alias)
  {
    $cache = $this->getCachedOptions();
    if ( empty($cache) || false === array_key_exists($option_alias, $cache) ) {
      //trigger_error(get_class($this)." - L'option '$option_alias' n'existe pas pour cette application (".get_class($this->options_provider).", appli_id = ".$this->options_provider->getAppliId().")", E_USER_WARNING);
      return false;
    }
    return isset($cache[$option_alias]['option_droits_avances']) && $cache[$option_alias]['option_droits_avances'] === true;
  }
  
  /**
   * Indique si l'option est activée pour l'utilsiateur connecté (par défaut)
   * ou pour tout autre utilisateur. Teste automatiquement si l'option est activée ou non
   * soit de manière globale soit via les ACL en fonction de la configuration de l'option.
   * C'est la fonction à privilégier pour vérifier l'activation ou non d'une option
   *
   * @param string $option_alias  alias de l'option à tester
   * @param int $user_id          utilisateur connecté si null (par défaut), ou identifiant d'un utilisateur
   * @return boolean
   */
  public function isGranted($option_alias, $user_id=null)
  {
    if( $this->optionIsGloballyActivated($option_alias) ) return true;
    if( $this->optionIsGloballyDeactivated($option_alias) ) return false;
    if( $this->bAclAvailable && isset($cache[$option_alias]['option_id']) ) {
      return AlkAclManager::isGranted('RESSTYPE_ESPACE_OPTIONAPPLI', $cache[$option_alias]['option_id'], 'ROLE_ESPACE_OPTIONAPPLI', $user_id);
    }
    return false;
  }

}

?>