<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Espace
Module gestionnaire d'espaces.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkquery.class.php");

/**
 * @package Alkanet_Module_Espace
 * @class AlkQueryEspaceAction
 * Ensemble des requetes actions liées à la gestion des espaces de travail
 */
class AlkQueryEspaceAction extends AlkQuery
{

  /**
  * @brief Constructeur par défaut
  */ 
  public function __construct(&$oDb)
  {
    parent::__construct($oDb);
  }

  /**
   * Destructeur par défaut
   */
  public function __destruct()
  {
  
  }

  /**
   * Ajoute à l'espace cont_id tous les agents valides 
   *        non animateur et appartenant aux services de type état
   *
   * @param cont_id  Identifiant de l'espace
   */
  function add_agentNonAnimateurToEspacePublic($cont_id)
  {
  	// menage avant insertion
    $strSql = "delete from SIT_AGENT_CONT where CONT_ID=".$cont_id." and CONT_ADMIN=0";
    $this->dbConn->ExecuteSql($strSql);
	
    // ajoute tous les agents valides non administrateur et appartenant aux services de type état
    $strSql = "insert into SIT_AGENT_CONT (AGENT_ID, CONT_ID, CONT_ADMIN) ".
      " select a.AGENT_ID, ".$cont_id.", 0 ".
      " from".
      "   (select a1.AGENT_ID".
      "    from SIT_AGENT a1".
 			"      left join (select * from SIT_AGENT_CONT where CONT_ADMIN=1 and CONT_ID=".$cont_id.") ac on ac.AGENT_ID=a1.AGENT_ID".
 			"    where ac.AGENT_ID is null and a1.AGENT_VALIDE>-1) a".
      "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
      "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      " where ".
      "  s.SERVICE_ETAT=1 and s.SERVICE_NIVEAU=1".
      "  and ase.AGENT_ID=a.AGENT_ID and ase.SERVICE_ID=s.SERVICE_ID";
    $this->dbConn->ExecuteSql($strSql);
  }
    
  /**
   * Ajoute une application dans un espace.
   *        Place les droits par défaut sur les profils.
   *        Retourne son identifiant
   *
   * @param cont_id          Identifiant de l'espace
   * @param tabQuery         tableau associatif contenant les propriétés
   * @param appli_id_defaut  Identifiant de l'appli actuellement par défaut
   * @param appli_id_src     =-1 par défaut. Contient d'identifiant de l'application 
   *                         duppliquée pour permettre la copie des droits sur profil
   * @param appli_id_dist    identifiant de l'application distante associée (-1 si non renseignée)
   * @return Retourne un entier
   */
  function addAppli($cont_id, $tabQuery, $appli_id_defaut, $appli_id_src="-1", $appli_id_dist="-1")
  {
    $tabValue = $tabQuery["field"];
    $user_id = AlkFactory::getSProperty("user_id", "-1");

    $appli_id = ( $appli_id_dist == -1 
                  ? $this->dbConn->GetNextId("SIT_APPLI", "APPLI_ID", "SEQ_SIT_APPLI")
                  : $appli_id_dist );

    $tabValue["APPLI_ID"] = array(ALK_SQL_NUMBER, $appli_id);
    $appli_defaut = $tabValue["APPLI_DEFAUT"][1];
    unset($tabValue["APPLI_DEFAUT"]);

    $tabValue["AGENT_IDCREA"] = array(ALK_SQL_NUMBER, $user_id);
    $tabValue["APPLI_DATECREA"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    $tabValue["AGENT_IDMAJ"] = array(ALK_SQL_NUMBER, $user_id);
    $tabValue["APPLI_DATEMAJ"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
     
    // on ignore le cont_id (si on vient de collab)
    $strSql = "insert into SIT_APPLI ".$this->getPartInsertSql($tabValue, Array("CONT_ID"));
    $this->dbConn->ExecuteSql($strSql);

    // association conteneur-appli
    $strSql = "insert into SIT_APPLI_CONT (APPLI_ID, CONT_ID) values (".$appli_id.", ".$cont_id.")";
    $this->dbConn->ExecuteSql($strSql);
    
    // appli par defaut dans l'espace
    if( ($appli_defaut==1 && $appli_id_defaut!=$appli_id) ||
        ($appli_defaut==0 && $appli_id_defaut==$appli_id) ) {
      $idAppli = ( $appli_defaut==0 ? "0" : $appli_id );
      $strSql = "update SIT_CONTENEUR set CONT_APPLI_DEFAUT=".$idAppli.
        " where CONT_ID=".$cont_id." and ".$idAppli.
        " in (select a.APPLI_ID".
        "     from SIT_APPLI a".
        "       inner join SIT_APPLI_TYPE at on a.ATYPE_ID=at.ATYPE_ID".
        "     where at.ATYPE_URL is null or at.ATYPE_URL='' )";
      $this->dbConn->ExecuteSql($strSql);
    }

    /* place les droits sur les profils
    $strDroitMax = ( $bDroitPub == true ? "4" : "2" );
    if( $appli_id_src == "-1" ) {
      // initialise les droits sur profil : droit max pour profil 1, droit aucun pour les autres
      $strSql = "insert into SIT_PROFIL_APPLI (PROFIL_ID, APPLI_ID, APPLI_DROIT_ID)".
        " select PROFIL_ID, ".$appli_id.", ".
        $this->dbConn->CompareSql("PROFIL_ID", "=", "1", $strDroitMax, "0").
        " from SIT_PROFIL";
    } else {
      // recopie les droits sur profil d'une autre application
      $strSql = "insert into SIT_PROFIL_APPLI (PROFIL_ID, APPLI_ID, APPLI_DROIT_ID)".
        " select PROFIL_ID, ".$appli_id.", APPLI_DROIT_ID".
        " from SIT_PROFIL_APPLI".
        " where APPLI_ID=".$appli_id_src;
    }
    $this->dbConn->ExecuteSql($strSql);*/

    return $appli_id;
  }
  
  /**
   * Recopie les droits des profils génériques d'une application src vers dest
   * Duplique également les droits et les profils spécifiques 
   * @param appli_id_src      identifiant de l'application source
   * @param appli_id_dest     identifiant de l'application nouvellement créée 
   * @param cont_id_src       identifiant de l'espace source
   * @param cont_id_dest      identifiant de l'espace nouvellement créée 
   * @param tabProfilSpeAssoc tableau des associations entre profils spécifiques sources et dupliqué
   */
  public function copyAppliRights($appli_id_src, $appli_id_dest, $cont_id_src, $cont_id_dest, $tabProfilSpeAssoc=array())
  {
    $strSql = "delete from SIT_PROFIL_APPLI where APPLI_ID=".$appli_id_dest;
	$this->dbConn->ExecuteSql($strSql);
  
    $strSql = "insert into SIT_PROFIL_APPLI (PROFIL_ID, APPLI_ID, APPLI_DROIT_ID)".
      " select pa.PROFIL_ID, ".$appli_id_dest.", pa.APPLI_DROIT_ID".
      " from SIT_PROFIL_APPLI pa".
      "   inner join SIT_PROFIL p on pa.PROFIL_ID=p.PROFIL_ID".
      " where pa.APPLI_ID=".$appli_id_src." and p.CONT_ID=0";
    $this->dbConn->ExecuteSql($strSql);
    
    // recopie les profils spécifiques de cont_id_src vers $cont_id_dest
    $oQueryAnnuAction = AlkFactory::getQueryAction(ALK_ATYPE_ID_ANNU);
    
    $strSql = "select * from SIT_PROFIL where CONT_ID=".$cont_id_src;
    $dsProfil = $this->dbConn->initDataSet($strSql);
    while( $drProfil = $dsProfil->getRowIter() ) {
      $profil_id_src = $drProfil->getValueName("PROFIL_ID");
      
      if( !array_key_exists("_".$profil_id_src, $tabProfilSpeAssoc) ) {
        // ajout du nouveau profil spécifique 
        $tabQuery["field"]["PROFIL_INTITULE"]  = array(ALK_SQL_TEXT, $drProfil->getValueName("PROFIL_INTITULE"));
        $tabQuery["field"]["PROFIL_DESC"]      = array(ALK_SQL_TEXT, $drProfil->getValueName("PROFIL_DESC"));
        $tabQuery["field"]["PROFIL_RANG"]      = array(ALK_SQL_NUMBER, 0);
        $tabQuery["field"]["PROFIL_EFFACABLE"] = array(ALK_SQL_NUMBER, 1);
        $tabQuery["field"]["CONT_ID"]          = array(ALK_SQL_NUMBER, $cont_id_dest);

        $profil_id = $oQueryAnnuAction->addProfil($tabQuery);
        $tabProfilSpeAssoc["_".$profil_id_src] = $profil_id;
      } else {
        $profil_id = $tabProfilSpeAssoc["_".$profil_id_src]; 
      } 
      
      // recopie des droits associés au profil specifique dupliqué
      $strSql = "insert into SIT_PROFIL_APPLI (PROFIL_ID, APPLI_ID, APPLI_DROIT_ID)".
        " select ".$profil_id.", ".$appli_id_dest.", APPLI_DROIT_ID".
        " from SIT_PROFIL_APPLI".
        " where PROFIL_ID=".$profil_id_src." and APPLI_ID=".$appli_id_src;
      $this->dbConn->ExecuteSql($strSql);  
      
      // recopie l'association entre agents et profil specifique dupliqué
      $strSql = "insert into SIT_AGENT_PROFIL (PROFIL_ID, AGENT_ID)".
        " select ".$profil_id.", AGENT_ID".
        " from SIT_AGENT_PROFIL".
        " where PROFIL_ID=".$profil_id_src;
      $this->dbConn->ExecuteSql($strSql);
    }
    
    // suppression des éventuels droits déjà présents mis par la fonction setRightProfilToAgentByEspace
    $strSql = "delete from SIT_AGENT_APPLI where APPLI_ID=".$appli_id_dest;
    $this->dbConn->ExecuteSql($strSql);
    
    // recopie des droits spécifiques des agents
    $strSql = "insert into SIT_AGENT_APPLI (AGENT_ID, APPLI_ID, APPLI_DROIT_ID) ".
      " select a.AGENT_ID, ".$appli_id_dest.", a.APPLI_DROIT_ID".
      " from SIT_AGENT_APPLI a".
      " where a.APPLI_ID=".$appli_id_src;
    $this->dbConn->ExecuteSql($strSql);
    
    /* recopie des droits spécifiques des agents
    $strSql = "insert into SIT_ABONNE_APPLI (ABONNE_ID, APPLI_ID) ".
      " select ABONNE_ID, ".$appli_id_dest.
      " from SIT_ABONNE_APPLI".
      " where APPLI_ID=".$appli_id_src;
    $this->dbConn->ExecuteSql($strSql);*/
    
    return $tabProfilSpeAssoc;
  }
  
  /**
   * Ajoute une application dans un espace.
   *        Place les droits par défaut sur les profils.
   *        Retourne son identifiant
   *
   * @param cont_id          Identifiant de l'espace
   * @param appli_id         Identifiant de l'application
   * @param tabQuery         tableau associatif contenant les propriétés
   * @param appli_id_defaut  Identifiant de l'appli actuellement par défaut
   */
  function updateAppli($cont_id, $appli_id, $tabQuery, $appli_id_defaut)
  {
    $tabValue = $tabQuery["field"];
    $user_id = AlkFactory::getSProperty("user_id", "-1");

    // met à jour la fiche
    $tabValue["AGENT_IDMAJ"] = array(ALK_SQL_NUMBER, $user_id);
    $tabValue["APPLI_DATEMAJ"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    $appli_defaut = $tabValue["APPLI_DEFAUT"][1];
    unset($tabValue["APPLI_DEFAUT"]);

    $strSql = "update SIT_APPLI set ".$this->getPartUpdateSql($tabValue)." where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);

    // appli par defaut dans l'espace
    if( ($appli_defaut==1 && $appli_id_defaut!=$appli_id) ||
        ($appli_defaut==0 && $appli_id_defaut==$appli_id) ) {
      $idAppli = ( $appli_defaut==0 ? "0" : $appli_id );
      $strSql = "update SIT_CONTENEUR set CONT_APPLI_DEFAUT=".$idAppli." where CONT_ID=".$cont_id;
      $this->dbConn->ExecuteSql($strSql);
    }
  }

  /**
   * Supprime l'application appli_id de l'espace cont_id .
   *
   * @param cont_id          Identifiant de l'espace
   * @param appli_id         Identifiant de l'application
   * @param appli_id_defaut  Identifiant de l'appli actuellement par défaut
   */
  function delAppli($cont_id, $appli_id, $appli_id_defaut)
  {
    // supprime l'association conteneur-appli
    $strSql = "delete from SIT_APPLI_CONT where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);
    
    // supprime l'association agent-appli
    $strSql = "delete from SIT_AGENT_APPLI where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);

    /* supprime l'association agent-appli
    $strSql = "delete from SIT_APPLI_LISTE where APPLI_ID=".$appli_id;
    $strSql = "delete from SIT_ABONNE_APPLI where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);*/
    
    // supprime l'association profil-appli
    $strSql = "delete from SIT_PROFIL_APPLI where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);
  
    // suppression des stats
    $strSql = "delete from SIT_STATISTIQUE where TYPE_CONNEXION=2 and ID_TYPE_CONNEXION=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);

    // suppression des paramétrages
    $strSql = "delete from SIT_APPLI_PARAM where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);

    if( defined("ALK_B_ATYPE_GEDIT") && ALK_B_ATYPE_GEDIT==true ) {
      // suppression des blocs associés à l'appli
      $strSql = "delete from GEDIT_01_APPLI_BLOC where APPLI_ID=".$appli_id;
      $this->dbConn->ExecuteSql($strSql);
      
      // réinitialisation des associations des pages
      $strSql = "update GEDIT_01_PAGE set PAGE_APPLIID=-1 where PAGE_APPLIID=".$appli_id;
      $this->dbConn->ExecuteSql($strSql);
    }

    // suppression de la syndication associée aux infos de l'appli supprimée
    if( defined("ALK_B_ATYPE_SYND") && ALK_B_ATYPE_SYND==true ) {
      $oQuerySyndAction = AlkFactory::getQueryAction(ALK_ATYPE_ID_SYND);
      $oQuerySyndAction->delAppli($appli_id);
    }

    // suppression des données indexées
    if( defined("ALK_B_ATYPE_SEARCH") && ALK_B_ATYPE_SEARCH==true && !is_null($this->oQuerySearchAction) ) {
      $this->oQuerySearchAction->delAppli($appli_id);
    } 

	// supprime les liaisons entre applis
	$strSql = "update SIT_APPLI set APPLI_LIEE=null where APPLI_LIEE=".$appli_id;
	$this->dbConn->ExecuteSql($strSql);
    
    // supprime la fiche appli
    $strSql = "delete from SIT_APPLI where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);
  
    // si l'appli était par défaut
    if( $appli_id_defaut == $appli_id )  {
      $strSql = "update SIT_CONTENEUR set CONT_APPLI_DEFAUT=0 where CONT_ID=".$cont_id;
      $this->dbConn->ExecuteSql($strSql);
    }
	
  }

  /**
   * Fixe les droits aux agents de profil_id sur les applis de l'espace cont_id
   *
   * @param cont_id   Identifiant de l'espace
   * @param profil_id Identifiant du profil
   *
  function setRightProfilToAgentFromEspace($cont_id, $profil_id=-1)
  {
    // suppression des droits des utilisateurs
    $strSql = "delete from SIT_AGENT_APPLI".
      " where AGENT_ID in".
      "   (select distinct ap.AGENT_ID".
      "    from SIT_AGENT_PROFIL ap".
      "      inner join SIT_AGENT_CONT ac on ap.AGENT_ID=ac.AGENT_ID".
      "    where ac.CONT_ID=".$cont_id.( $profil_id!="-1" ? " and ap.PROFIL_ID=".$profil_id : "" ).")".
      " and APPLI_ID in".
      "   (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.")";
    $this->dbConn->ExecuteSql($strSql);

    // insertion des droits pour les agents de profil_id sur les applis de l'espace
    $strSql = "insert into SIT_AGENT_APPLI (AGENT_ID, APPLI_ID, APPLI_DROIT_ID)".
      "select agc.AGENT_ID, pa.APPLI_ID,".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)")." ".
      " from SIT_AGENT_PROFIL ap".
      "  inner join SIT_PROFIL_APPLI pa on pa.PROFIL_ID=ap.PROFIL_ID".
      "  inner join SIT_APPLI_CONT ac on pa.APPLI_ID=ac.APPLI_ID".
      "  inner join SIT_AGENT_CONT agc on (agc.AGENT_ID=ap.AGENT_ID and ac.CONT_ID=agc.CONT_ID)".
      " where agc.CONT_ID=".$cont_id.( $profil_id != -1 ? " and ap.PROFIL_ID=".$profil_id : "" ).
      " group by agc.AGENT_ID, pa.APPLI_ID, agc.CONT_ADMIN".
      " having ".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)").">0";
    $this->dbConn->ExecuteSql($strSql);
  }*/

  /**
   * Applique les droits sur les profils pour l'application appli_id
   *
   * @param appli_id   Identifiant de l'application 
   * @param tabRight   Tableau contenant les couples (profil_id, droit_id) séparés par un tiret
   */
  function setRightsOnAppli($appli_id, $tabRight)
  {
    // menage
    $strSql = "delete from SIT_PROFIL_APPLI where APPLI_ID=".$appli_id;
    $this->dbConn->ExecuteSql($strSql);

    // ajoute les droits saisis
    $strSqlBase = "insert into SIT_PROFIL_APPLI (APPLI_ID, PROFIL_ID, APPLI_DROIT_ID) values (".$appli_id.",";    
    if( is_array($tabRight) ) {
      foreach($tabRight as $strRight) {
        $tabTmp = explode("-", $strRight);
        $strSql = $strSqlBase.$tabTmp[0].", ".$tabTmp[1].")";
        $this->dbConn->ExecuteSql($strSql);
      }
    }
    
    // précalcul des droits pour chaque utilisateur
    $this->setRightProfilToAgentByAppli($appli_id);
  }
  
  /**
   * fixe les droits aux agents de l'espace / aux droits sur profil sur une application
   *
   * @param cont_id    Identifiant de l'application (peut valoir -1 pour tous les espaces)
   * @param agent_id   Identifiant de l'agent (=-1 par défaut pour ts les agents de l'espace)
   */
  private function setRightProfilToAgentByEspace($cont_id, $agent_id="-1")
  {
    if( $agent_id == "" ) {
      $agent_id = "-1";
    }
    
    // suppression des droits des utilisateurs
    $strSql = "delete from SIT_AGENT_APPLI where ".
      ( $cont_id>0
        ? " APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.")"
        : " APPLI_ID>0" ).
      ( $agent_id != "-1" 
        ? " and AGENT_ID in (".$agent_id.")"
        : "" );
    $this->dbConn->ExecuteSql($strSql);

    /* gestion des abonnés aux applis
    $strSql = "delete from SIT_ABONNE_APPLI where ".
      ( $cont_id>0
        ? " APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.")"
        : " APPLI_ID>0" ).
      ( $agent_id != "-1" 
        ? " and ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$agent_id."))"
        : "" );
    $this->dbConn->ExecuteSql($strSql);*/
      
    // insertion des droits pour les agents de profil_id sur l'appli
    $strSql = "insert into SIT_AGENT_APPLI (AGENT_ID, APPLI_ID, APPLI_DROIT_ID)".
      " select agc.AGENT_ID, pa.APPLI_ID,".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)")." ".
      " from SIT_AGENT_PROFIL ap".
      "  inner join SIT_PROFIL_APPLI pa on pa.PROFIL_ID=ap.PROFIL_ID".
      "  inner join SIT_APPLI_CONT ac on pa.APPLI_ID=ac.APPLI_ID".
      "  inner join SIT_AGENT_CONT agc on (agc.AGENT_ID=ap.AGENT_ID and ac.CONT_ID=agc.CONT_ID)".
      " where" .
      ( $cont_id>-1
        ? " agc.CONT_ID=".$cont_id
        : " agc.CONT_ID>0" ).
      ( $agent_id != "-1" 
        ? " and agc.AGENT_ID in (".$agent_id.")"
        : "" ).
      " group by agc.AGENT_ID, pa.APPLI_ID, agc.CONT_ADMIN".
      " having ".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)").">0";
    $this->dbConn->ExecuteSql($strSql);
    
    /* insertion des abonnés
    $strSql = "insert into SIT_ABONNE_APPLI (ABONNE_ID, APPLI_ID)".
      " select ab.ABONNE_ID, aa.APPLI_ID".
      " from SIT_AGENT_APPLI aa".
      "   inner join SIT_ABONNE ab on ab.AGENT_ID=aa.AGENT_ID" .
      "   inner join SIT_APPLI a on aa.APPLI_ID=a.APPLI_ID".
      " where a.ATYPE_ID in (".$this->getATypeIdListAbonne().")".
      ( $cont_id>0
        ? " and aa.APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.")"
        : " and aa.APPLI_ID>0" ).
      ( $agent_id != "-1" 
        ? " and ab.AGENT_ID in (".$agent_id.")"
        : "" );
    $this->dbConn->ExecuteSql($strSql);*/
  }
  
  /**
   * Retourne la liste des types d'applis gérant des abonnés
   * @return string
   */
  function getATypeIdListAbonne()
  {
    return ALK_ATYPE_ID_COMMENT.", ".ALK_ATYPE_ID_NEWSLETTER.", ".ALK_ATYPE_ID_FORUM; 
  }
  
  /**
   * fixe les droits aux agents de l'espace / aux droits sur profil sur une application
   *
   * @param appli_id   Identifiant de l'application
   * @param agent_id   Identifiant de l'agent (=-1 par défaut pour ts les agents de l'epace)
   */
  private function setRightProfilToAgentByAppli($appli_id, $agent_id="-1")
  {
    if( $agent_id == "" ) {
      $agent_id = "-1";
    }
    
    // suppression des droits des utilisateurs
    $strSql = "delete from SIT_AGENT_APPLI where APPLI_ID in (".$appli_id.")".
      ( $agent_id != "-1" 
        ? " and AGENT_ID in (".$agent_id.")"
        : "" );
    $this->dbConn->ExecuteSql($strSql);
    
    /* suppression des abonnés aux applis
    $strSql = "delete from SIT_ABONNE_APPLI where APPLI_ID in (".$appli_id.")".
      ( $agent_id != "-1" 
        ? " and ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$agent_id."))"
        : "" );
    $this->dbConn->ExecuteSql($strSql);*/
      
    // insertion des droits pour les agents de profil_id sur l'appli
    $strSql = "insert into SIT_AGENT_APPLI (AGENT_ID, APPLI_ID, APPLI_DROIT_ID)".
      " select agc.AGENT_ID, pa.APPLI_ID,".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)")." ".
      " from SIT_AGENT_PROFIL ap".
      "  inner join SIT_PROFIL_APPLI pa on pa.PROFIL_ID=ap.PROFIL_ID".
      "  inner join SIT_APPLI_CONT ac on pa.APPLI_ID=ac.APPLI_ID".
      "  inner join SIT_AGENT_CONT agc on (agc.AGENT_ID=ap.AGENT_ID and ac.CONT_ID=agc.CONT_ID)".
      " where ac.APPLI_ID=".$appli_id.
      ( $agent_id != "-1" 
        ? " and agc.AGENT_ID in (".$agent_id.")"
        : "" ).
      " group by agc.AGENT_ID, pa.APPLI_ID, agc.CONT_ADMIN".
      " having ".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)")." >0";

    $this->dbConn->ExecuteSql($strSql);
    
    /* ajout des abonnés
    $strSql = "insert into SIT_ABONNE_APPLI (ABONNE_ID, APPLI_ID)".
      " select ab.ABONNE_ID, aa.APPLI_ID".
      " from SIT_AGENT_APPLI aa".
      "   inner join SIT_ABONNE ab on ab.AGENT_ID=aa.AGENT_ID" .
      "   inner join SIT_APPLI a on aa.APPLI_ID=a.APPLI_ID".
      " where a.ATYPE_ID in (".$this->getATypeIdListAbonne().")".
      "  and aa.APPLI_ID in (".$appli_id.")".
      ( $agent_id != "-1" 
        ? " and ab.AGENT_ID in (".$agent_id.")"
        : "" );
    $this->dbConn->ExecuteSql($strSql);*/
  }

  /**
   * Invitation d'un ou plusieurs utilisateurs à un espace (tous si cont_id=-1, tous les espaces publics si cont_id=-2)
   * Retourne un tableau vide
   * pour le cas typeAdd=0, retourne un tableau contenant les identifiants des utilisateurs qui ont perdu l'animation d'un espace 
   * @param typeAdd         type d'ajout 
   * @param user_id         identifiant ou liste d'id d'utilisateurs. -1 pour tous les utilisateurs ayant les privilèges priv_id
   * @param cont_id         identifiant de l'espace. 
   * @param priv_id         ensemble des privilèges, ALK_PRIV_SPACE_NONE par défaut
   * @param strListIdToDel  liste d'identifiant d'utilisateur à supprimer avant insertion
   * @return array
   */
  public function addUserToSpace($typeAdd, $user_id, $cont_id=-1, $priv_id=ALK_PRIV_SPACE_NONE, $strListIdToDel="")
  {
    $tabUsers = array();
    if( $typeAdd==0 ) {
      // ménage avant insert : suppression des animateurs qui vont le rester
      if( $strListIdToDel!="" ) {
        $this->removeUserFromSpace($cont_id, $strListIdToDel);
      } else {
        $this->removeUserFromSpace($cont_id, $user_id);
      }
      // il reste les animateurs dont il faut retirer l'animation
      // désactive l'animation pour les animateurs restants
      // réapplique les droits 
      $strSql = "select AGENT_ID from SIT_AGENT_CONT where CONT_ADMIN=1 and CONT_ID=".$cont_id;
      $dsAC = $this->dbConn->initDataset($strSql);
      while( $drAC = $dsAC->getRowIter() ) {
        $tabUsers[] = $drAC->getValueName("AGENT_ID");
      }
      
      $strSql = "update SIT_AGENT_CONT set CONT_ADMIN=0 where CONT_ADMIN=1 and CONT_ID=".$cont_id;
      $this->executeSql($strSql);
    }

    if( trim($user_id) == "" ) return array();

    $bDept = false;
    if( ALK_B_SPACE_ADMIN_ALL_DEPT==true && $typeAdd != "1" && $typeAdd != "3") {
      $strSql = "select count(DEPT_ID) as nb from SIT_DEPT where DEPT_VISIBLE=1";
      $dsNb = $this->dbConn->initDataset($strSql);
      if( $drNb = $dsNb->getRowIter() ) { 
        $iNb = $drNb->getValueName("NB");
        if( $iNb == "" ) $iNb = 0; 
        $bDept = ( $iNb == 0 ? false : true );
      }
    }

    $tabSql = array();
    switch( $typeAdd ) {
    case 0: // ajout d'animateur : user_id!="", cont_id>0,  priv_id=ALK_PRIV_SPACE_ANIM
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
        " select ".$cont_id.", a.AGENT_ID, 1".
        " from SIT_AGENT a".
        " where a.AGENT_ID in (".$user_id.") and a.AGENT_VALIDE>-1";
      break;

    case 1: // invitation d'une liste d'utilisateurs à l'espace : user_id!="", cont_id>0, priv_id=ALK_PRIV_SPACE_USER
      $user_id = $this->dbConn->delDoublon($user_id);
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
        " select ".$cont_id.", a.AGENT_ID, 0".
        " from SIT_AGENT a".
        " where a.AGENT_ID in (".$user_id.") and a.AGENT_VALIDE>-1";
      break;

    case 2: // création espace, ajout des admins all + user courant : user_id=-1 && cont_id>0
      $currentUser_id = AlkFactory::getSProperty("user_id", "-1");
      if( !$bDept ) {
        // ajout du créateur et de tous les animateurs de tous les espaces
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
          " select distinct ".$cont_id.", a.AGENT_ID, 1".
          " from SIT_AGENT a".
          " where a.AGENT_VALIDE>-1 and (a.AGENT_ID=".$currentUser_id." or (a.AGENT_PRIV & ".$priv_id.")>0)";        
      } else {
        // ajout du créateur et 
        // si espace public : util alkante + tous les animateurs de tous les espaces liés aux dépts du service des animateurs
        $cont_public = "0";
        $strSql = "select CONT_PUBLIC from SIT_CONTENEUR where CONT_ID=".$cont_id;
        $dsCont = $this->dbConn->initDataset($strSql);
        if( $drCont = $dsCont->getRowIter() ) {
          $cont_public = $drCont->getValueName("CONT_PUBLIC");
        }
        
        if( $cont_public == "0" ) {
          // espace privé
          $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
            " select distinct ".$cont_id.", a.AGENT_ID, 1".
            " from SIT_AGENT a".
            " where a.AGENT_ID=".$currentUser_id;
        } else {
          // espace public
          $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
            " select distinct ".$cont_id.", a.AGENT_ID, 1".
            " from SIT_AGENT a".
            " where a.AGENT_ID in (".$currentUser_id.", ".ALK_USER_ID_ADMINALK.")".        
            " union".
        " select ".$cont_id.", a.AGENT_ID, 1".
        " from SIT_AGENT a".
            "   inner join SIT_AGENT_SERVICE ase on a.AGENT_ID=ase.AGENT_ID".
            "   inner join SIT_SERVICE s on (ase.SERVICE_ID=s.SERVICE_ID and s.SERVICE_NIVEAU=1 and s.SERVICE_ETAT=1)".
            "   inner join SIT_DEPT_CONT dc on s.DEPT_ID=dc.DEPT_ID".
            "   inner join SIT_CONTENEUR c on (dc.CONT_ID=c.CONT_ID and c.CONT_PUBLIC=1 and c.CONT_ID=".$cont_id.")".
            " where a.AGENT_VALIDE>-1 and a.AGENT_ID<>".$currentUser_id." and a.AGENT_ID<>".ALK_USER_ID_ADMINALK.
            " and (a.AGENT_PRIV & ".$priv_id.")>0";
        }
      }
      break;

    case 3: // ajout espace public ou modif espace qui devient public : ajout utilisateur serv etat : user_id=listeID, cont_id>0
      $currentUser_id = AlkFactory::getSProperty("user_id", "-1");
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
        " select ".$cont_id.", a.AGENT_ID, 0".
        " from SIT_AGENT a".
        " where a.AGENT_ID in (".$user_id.")";
      break;

    case 4: // ajout d'un utilisateur avec privilège : user_id=x
      //$tabSql[] = "delete from SIT_ABONNE_APPLI where ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID=".$user_id.")";
      $tabSql[] = "delete from SIT_AGENT_APPLI where AGENT_ID=".$user_id;
      $tabSql[] = "delete from SIT_AGENT_CONT where AGENT_ID=".$user_id;
      
      if( !$bDept ) {
        // ajoute l'utilisateur à tous les espaces
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
        " select c.CONT_ID, ".$user_id.", ".(($priv_id & ALK_PRIV_SPACE_ADMIN)==ALK_PRIV_SPACE_ADMIN ? "1" : "0").
        " from SIT_CONTENEUR c".
        " where c.CONT_ID>0";
      } else {
        // ajoute l'utilisateur à tous les espaces publics liés au dept de son service
        $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
          " select c.CONT_ID, ".$user_id.", ".(($priv_id & ALK_PRIV_SPACE_ADMIN)==ALK_PRIV_SPACE_ADMIN ? "1" : "0").
          " from SIT_CONTENEUR c".
          "   inner join SIT_DEPT_CONT dc on c.CONT_ID=dc.CONT_ID".
          "   inner join SIT_SERVICE s on (s.DEPT_ID=dc.DEPT_ID and s.SERVICE_NIVEAU=1 and s.SERVICE_ETAT=1)".
          "   inner join SIT_AGENT_SERVICE ase on (s.SERVICE_ID=ase.SERVICE_ID and ase.AGENT_ID=".$user_id.")".
          " where c.CONT_ID>0 and c.CONT_PUBLIC=1";
      }
      break;

    case 5: // ajout d'un utilisateur sans privilège d'un service de type etat : l'invite à tous les espace public
      if( $bDept ) {
        // invite aux espaces publics lié au dept de son service
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
        " select c.CONT_ID, ".$user_id.", 0".
        " from SIT_CONTENEUR c".
        "   inner join SIT_DEPT_CONT dc on dc.CONT_ID=c.CONT_ID".
          "   inner join SIT_SERVICE s on (s.DEPT_ID=dc.DEPT_ID and s.SERVICE_NIVEAU=1 and s.SERVICE_ETAT=1)".
        "   inner join SIT_AGENT_SERVICE ase on (s.SERVICE_ID=ase.SERVICE_ID and ase.AGENT_ID=".$user_id.")". 
          " where c.CONT_ID>0 and c.CONT_PUBLIC=1".
        " group by c.CONT_ID";
      } else {
        // invite à tous les espaces publics
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
          " select c.CONT_ID, ".$user_id.", 0".
          " from SIT_CONTENEUR c".
          " where c.CONT_ID>0 and c.CONT_PUBLIC=1";
      }
      break;
    
    case 6: // invite un utilisateur sur un espace avec un privilège
      $cont_admin = ( ($priv_id & ALK_PRIV_SPACE_ADMIN)==ALK_PRIV_SPACE_ADMIN ||
                      ($priv_id & ALK_PRIV_SPACE_ANIM)==ALK_PRIV_SPACE_ANIM
                      ? "1" 
                      : "0" );
      $tabSql[] = "insert into SIT_AGENT_CONT (CONT_ID, AGENT_ID, CONT_ADMIN)".
        " values (".$cont_id.", ".$user_id.", ".$cont_admin.")";
      break;
    }
    
    foreach($tabSql as $strSql) {
      //echo $strSql."<br><br>";
      $this->dbConn->ExecuteSql($strSql);
    }
    
    // précalcul des droits
    switch( $typeAdd ) {
    case 0: // ajout d'animateur : user_id!="", cont_id>0,  priv_id=ALK_PRIV_SPACE_ANIM
    case 1: // invitation d'une liste d'utilisateurs à l'espace : user_id="", cont_id>0, priv_id=ALK_PRIV_SPACE_USER
    case 4: // ajout d'un utilisateur avec privilège : user_id=x
    case 6: // invite un utilisateur sur un espace avec un privilège
      $this->setRightProfilToAgentByEspace($cont_id, $user_id);
      break;

    case 2: // création espace, ajout des admins all + user courant : user_id=-1 && cont_id>0
    case 3: // ajout espace public ou modif espace qui devient public : ajout utilisateur serv etat : user_id=listeID, cont_id>0
      $this->setRightProfilToAgentByEspace($cont_id);
      break;

    case 5: // ajout d'un utilisateur sans privilège d'un service de type etat : l'invite à tous les espace public
      $this->setRightProfilToAgentByEspace(-1, $user_id);
      break;
    }
    return $tabUsers;
  }

  /**
   * Retire les utilisateurs de tous les espaces
   *        Suppression des logs, les stats ne sont pas supprimées
   * @param strListUserId  liste des agent_id sélectionnés 
   */
  function removeUser($strListUserId)
  {
    $strListUserId = $this->dbConn->delDoublon($strListUserId);

    /* supprime assoc abonne-appli
    $strSql = "delete from SIT_ABONNE_APPLI where ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID in (".$strListUserId."))";
    $this->dbConn->ExecuteSql($strSql);*/

    // supprime assoc agent-appli
    $strSql = "delete from SIT_AGENT_APPLI where AGENT_ID in (".$strListUserId.")";
    $this->dbConn->ExecuteSql($strSql);

    // supprime assoc agent-cont
    $strSql = "delete from SIT_AGENT_CONT where AGENT_ID in (".$strListUserId.")";
    $this->dbConn->ExecuteSql($strSql);
  }

  /** 
   * Ajout des droits liés au profil
   * @param profil_id  identifiant du profil
   * @param cont_id    identifiant de l'espace lié au profil =0 pour tous les espace = profil gen, >0 pour un espace = profil spé
   */
  public function addProfil($profil_id, $cont_id="0")
  {
    // ménage avant insertion
    $this->delProfil($profil_id);
    
    // ajout du droit 0 avec le profil sur toutes les applis
    $strSql = "insert into SIT_PROFIL_APPLI (APPLI_ID, PROFIL_ID, APPLI_DROIT_ID)".
      " select a.APPLI_ID, ".$profil_id.", 0".
      ( $cont_id == "0"
        ? " from SIT_APPLI a where a.APPLI_ID>0"
        : " from SIT_APPLI a inner join SIT_APPLI_CONT ac on a.APPLI_ID=ac.APPLI_ID where ac.CONT_ID=".$cont_id." and a.APPLI_ID>0" );
    $this->dbConn->ExecuteSql($strSql);
  }  

  /** 
   * Suppression des droits liés au profil à supprimer
   * @param profil_id  identifiant du profil à supprimer
   */
  public function delProfil($profil_id)
  {
    $strSql = "delete from SIT_PROFIL_APPLI where PROFIL_ID=".$profil_id;
    $this->dbConn->ExecuteSql($strSql);
  }  

  /*
   * Transfère de propriété d'un utilisateur à un autre
   *        Méthode à appeler avant suppression définitive d'un utilisateur
   *        redirige les éléments de stats pour ne pas perturber les résultats globaux
   *        les logs sont supprimés
   * @param idUserFrom  identifiant de l'utilisateur perdant la propriété des ses données
   * @param idUserTo    identifiant de l'utilisateur récupérant la propriété des données
   */
  function replaceUser($idUserFrom, $idUserTo)
  {
    $strSql = "update SIT_STAT_APPLI set AGENT=".$idUserTo." where AGENT_ID=".$idUserFrom;
    $this->dbConn->ExecuteSql($strSql);

    $strSql = "update SIT_STATISTIQUE set AGENT=".$idUserTo." where AGENT_ID=".$idUserFrom;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Retire les utilisateurs de l'espace cont_id
   *
   * @param cont_id         Identifiant de l'espace, si <=0, suppression sur tous les espaces
   * @param strListUserId   liste des agent_id sélectionnés, passé par référence (suppression des doublons) 
   * @param priv_id         privilège de ceux à supprimer. =ALK_PRIV_SPACE_NONE par défaut.
   *                        si =ALK_PRIV_SPACE_ANIM, suppression de tous les animateurs de l'espace
   */
  function removeUserFromSpace($cont_id, &$strListUserId, $priv_id=ALK_PRIV_SPACE_NONE)
  {
    $strListUserId = $this->dbConn->delDoublon($strListUserId);

    if( trim($strListUserId) == "" && $priv_id==ALK_PRIV_SPACE_NONE ) return;
    if( $cont_id<=0 && $priv_id!=ALK_PRIV_SPACE_NONE ) return;

    $strWhereCont = ( $cont_id>0 ? "CONT_ID=".$cont_id ." and " : "" );
    $strWhereCont2 = $strWhereCont.
      ( $priv_id==ALK_PRIV_SPACE_NONE
        ? "AGENT_ID in (".$strListUserId.")"
        : "( CONT_ADMIN=1 or AGENT_ID in (".$strListUserId."))" );

    $strSql = "delete from SIT_AGENT_APPLI".
      " where ".
      ( $cont_id > 0
        ? "APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.") and "
        : "" ).
      " AGENT_ID in (select AGENT_ID from SIT_AGENT_CONT where ".$strWhereCont2.")";
    $this->dbConn->ExecuteSql($strSql);

    /*$strSql = "delete from SIT_ABONNE_APPLI".
      " where ".
      ( $cont_id > 0
        ? "APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.") and "
        : "" ).
      " ABONNE_ID in (select ABONNE_ID " .
      "               from SIT_AGENT_CONT ac " .
      "                 inner join SIT_ABONNE ab on ab.AGENT_ID=ac.AGENT_ID " .
      "               where ".str_replace("AGENT_ID", "ac.AGENT_ID", $strWhereCont2).")";
    $this->dbConn->ExecuteSql($strSql);*/

    // supprime assoc agent-cont
    $strSql = "delete from SIT_AGENT_CONT where ".$strWhereCont2;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Applique les droits spécifiques de l'utilisateur aux applis de l'espace
   * @param cont_id  identifiant d'utilisateur
   * @param user_id  identifiant de l'espace
   * @param tabRight ensemble des droits sélectionnés
   */
  function setUserRightsOnSpace($cont_id, $user_id, $tabRight, $tabMem)
  {
    // menage avant insertion
    $strSql = "delete from SIT_AGENT_APPLI".
      " where APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.")".
      " and AGENT_ID=".$user_id;
    $this->dbConn->ExecuteSql($strSql);
    
    /* menage sit_abonne_appli
    $strSql = "delete from SIT_ABONNE_APPLI".
      " where APPLI_ID in (select APPLI_ID from SIT_APPLI_CONT where CONT_ID=".$cont_id.")".
      " and ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID=".$user_id.")";
    $this->dbConn->ExecuteSql($strSql);*/
    
    // recherche des diffirences 
    $tabSql = array();
    $strListAppliId = "";
    if( is_array($tabRight) ) { //&& is_array($tabMem) && count($tabRight)==count($tabMem) ) {
      $strSqlBase = "insert into SIT_AGENT_APPLI (AGENT_ID, APPLI_ID, APPLI_DROIT_ID) values (".$user_id.",";    
      foreach($tabRight as $i => $strRight) {
        $tabTmp = explode("-", $strRight);
        if( /*$tabTmp[1] != $tabMem[$i] &&*/ $tabTmp[1]>0 ) {
          $tabSql[] = $strSqlBase.$tabTmp[0].", ".$tabTmp[1].")";
          $strListAppliId .= ",".$tabTmp[0];
        }
      }
      /*if( $strListAppliId != "" ) {
        $strListAppliId = substr($strListAppliId, 1); // retrait première virgule
        $tabSql[] = "insert into SIT_ABONNE_APPLI (APPLI_ID, ABONNE_ID)".
          " select a.APPLI_ID, ab.ABONNE_ID".
          " from SIT_APPLI a, SIT_ABONNE ab".
          " where ab.AGENT_ID=".$user_id.
          "   and a.APPLI_ID in (".$strListAppliId.")".
          "   and a.ATYPE_ID in (".$this->getATypeIdListAbonne().")";
      }*/
    }

    // indique que cet agent à des droits spécifiques
    $cont_droit = ( empty($tabSql) ? "0" : "1" );
    $strSql = "update SIT_AGENT_CONT set CONT_DROIT=".$cont_droit." where AGENT_ID=".$user_id;
    $this->dbConn->ExecuteSql($strSql);

    if( !empty($tabSql) ) {
      // enregistre les droits spécifiques
      foreach($tabSql as $strSql) {
        $this->dbConn->ExecuteSql($strSql);
      }
    }
  }

  /**
   * Enregistre les droits des profils aux applis de l'espace
   * @param cont_id  identifiant d'utilisateur
   * @param tabRight ensemble des droits sélectionnés
   */
  public function setProfilRightsOnSpace($cont_id, $tabRight)
  {
    if( !is_array($tabRight) ) return;
    
    // enregistrement
    foreach($tabRight as $strRight) {
      $tabTmp = explode("-", $strRight);
      $strSql = "update SIT_PROFIL_APPLI set APPLI_DROIT_ID=".$tabTmp[2].
        " where PROFIL_ID=".$tabTmp[1]." and APPLI_ID=".$tabTmp[0];
      $this->dbConn->ExecuteSql($strSql);
    }
    
    // recalcul des droits des applis de cont_id / agents
    $this->setRightProfilToAgentByEspace($cont_id);
  }

  /**
   * Applique les droits d'un utilisateur par rapport à ses profils sur l'ensemble de l'Extranet
   * @param user_id  identifiant de l'utilisateur 
   */
  public function setProfilRightToUser($user_id)
  {
    // suppression des droits des utilisateurs
    $strSql = "delete from SIT_AGENT_APPLI where AGENT_ID in (".$user_id.")";
    $this->dbConn->ExecuteSql($strSql);

    // gestion des abonnés aux applis
    //$strSql = "delete from SIT_ABONNE_APPLI where ABONNE_ID in (select ABONNE_ID from SIT_ABONNE where AGENT_ID=".$user_id.")";
    //$this->dbConn->ExecuteSql($strSql);
      
    // insertion des droits pour les agents de profil_id sur l'appli
    $strSql = "insert into SIT_AGENT_APPLI (AGENT_ID, APPLI_ID, APPLI_DROIT_ID)".
      " select agc.AGENT_ID, pa.APPLI_ID,".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)")." ".
      " from SIT_AGENT_PROFIL ap".
      "  inner join SIT_PROFIL_APPLI pa on pa.PROFIL_ID=ap.PROFIL_ID".
      "  inner join SIT_APPLI_CONT ac on pa.APPLI_ID=ac.APPLI_ID".
      "  inner join SIT_AGENT_CONT agc on (agc.AGENT_ID=ap.AGENT_ID and ac.CONT_ID=agc.CONT_ID)".
      " where agc.CONT_ID>0 and agc.AGENT_ID in (".$user_id.")".
      " group by agc.AGENT_ID, pa.APPLI_ID, agc.CONT_ADMIN".
      " having ".
      $this->dbConn->compareSql("agc.CONT_ADMIN", "=", "1", "4", "max(pa.APPLI_DROIT_ID)").">0";
    $this->dbConn->ExecuteSql($strSql);
    
    // insertion des abonnés
    /*$strSql = "insert into SIT_ABONNE_APPLI (ABONNE_ID, APPLI_ID)".
      " select ab.ABONNE_ID, aa.APPLI_ID".
      " from SIT_AGENT_APPLI aa".
      "   inner join SIT_ABONNE ab on ab.AGENT_ID=aa.AGENT_ID" .
      "   inner join SIT_APPLI a on aa.APPLI_ID=a.APPLI_ID".
      " where a.ATYPE_ID in (".$this->getATypeIdListAbonne().")".
      " and aa.APPLI_ID>0".
      " and ab.AGENT_ID=".$user_id;
    $this->dbConn->ExecuteSql($strSql);*/
  }

  /**
   * Déplace l'espace cont_id_move vers cont_id_dest
   *
   * @param cont_id_move Identifiant de l'espace à déplacer
   * @param cont_id_dest Identifiant du nouvel espace parent
   */
  public function moveSpace($cont_id_move, $cont_id_dest)
  {
    // deplace cont_id_move vers cont_id_dest

    $strSql = "select * from SIT_CONTENEUR where CONT_ID=".$cont_id_move;
    $dsMCont = $this->dbConn->initDataset($strSql);
    if( $drMCont = $dsMCont->getRowIter() ) {
      // info de l'espace a deplacer avant déplacement
      $mcont_pere  = $drMCont->getValueName("CONT_PERE");
      $mcont_rang  = $drMCont->getValueName("CONT_RANG");
      $mcont_niv   = $drMCont->getValueName("CONT_NIVEAU");
      $mcont_arbre = $drMCont->getValueName("CONT_ARBRE");

      $lgMCont_arbre = strlen($mcont_arbre);
 
      $strSql = "select * from SIT_CONTENEUR where CONT_ID=".$cont_id_dest;
      $dsPCont = $this->dbConn->initDataset($strSql);
      if( $drPCont = $dsPCont->getRowIter() ) {
        // info sur le nouvel espace pere
        $pcont_arbre = $drPCont->getValueName("CONT_ARBRE");
        $pcont_niv   = $drPCont->getValueName("CONT_NIVEAU");
        
        // info de l'espace déplacé après déplacement
        $cont_niv = $pcont_niv+1;
        $cont_arbre = $pcont_arbre.$cont_id_move."-";
        $nivDiff = $cont_niv-$mcont_niv;

        // met à jour le père de l'espace déplacé
        $strSql = "update SIT_CONTENEUR set CONT_PERE=".$cont_id_dest." where CONT_ID=".$cont_id_move;
        $this->dbConn->ExecuteSql($strSql);

        // recalcul l'arbre de l'espace déplacé et de sa sous arborescence
        $strArbreSql = $this->dbConn->getConcat("'".$cont_arbre."'",
                                                $this->dbConn->getSubstring("CONT_ARBRE", $lgMCont_arbre));

        $strSql = "update SIT_CONTENEUR set".
          "  CONT_NIVEAU=CONT_NIVEAU+(".$nivDiff.")".
          ", CONT_ARBRE=".$strArbreSql.
          " where CONT_ARBRE like '".$mcont_arbre."%'";
        $this->dbConn->ExecuteSql($strSql);

        // Pas de mise a jour du rang
      }
    }
  }

  /**
   * Retire le privilège animateur à un utilisateur sur ts les espaces
   *
   * @param agent_id Identifiant de l'agent
   */
  function del_privAnimateurFromAgent($agent_id)
  {
    $strSql = "update SIT_AGENT_CONT set".
      " CONT_ADMIN=0".
      " where CONT_ADMIN=1 and AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Retire le privilège animateur à un utilisateur sur ts les espaces
   *
   * @param agent_id Identifiant de l'agent
   */
  function add_privAnimateurToAgent($agent_id)
  {
    $strSql = "update SIT_AGENT_CONT set".
      " CONT_ADMIN=1".
      " where CONT_ADMIN=0 and AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Création d'un espace (paramètres liés à la base)
   *
   * @return Retourne un entier = identifiant de l'espace créé
   */
  function addSpace($user_id, $user_priv, $tabQuery, $cont_appli_defaut, $strPathUpload)
  {
    $tabValue = $tabQuery["field"];

    $cont_id = $tabValue["CONT_ID"][1] = $this->dbConn->GetNextId("SIT_CONTENEUR", "CONT_ID", "SEQ_SIT_CONTENEUR");
    $tabValue["CONT_ARBRE"][1] .= $cont_id."-";

    // traitement de l'upload
    $strFileName = doUpload("cont_logo", $cont_id."_l1_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["CONT_LOGO"][1] = $strFileName;
    } else {
      unset($tabValue["CONT_LOGO"]);
    }
    $strFileName = doUpload("cont_logo2", $cont_id."_l2_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["CONT_LOGO2"][1] = $strFileName;
    } else {
      unset($tabValue["CONT_LOGO2"]);
    }
    $strFileName = doUpload("cont_bandeau", $cont_id."_b_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["CONT_BANDEAU"][1] = $strFileName;
    } else {
      unset($tabValue["CONT_BANDEAU"]);
    }

    if( $tabValue["CONT_ALIAS"][1] != "" ) {
      // traitement du raccourci
      if( !$this->SaveLinkUrlSpace($cont_id, $cont_appli_defaut, "", $tabValue["CONT_ALIAS"][1]) ) {
        $tabValue["CONT_ALIAS"][1] = "";
      }
    }

    $tabValue["AGENT_ID"] = array(ALK_SQL_NUMBER, $user_id);
    $tabValue["CONT_DATE_CREA"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    $tabValue["AGENT_ID_MAJ"] = array(ALK_SQL_NUMBER, $user_id);
    $tabValue["CONT_DATE_MAJ"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());

    // création fiche espace
    $strSql = "insert into SIT_CONTENEUR ".$this->getPartInsertSql($tabValue);
    //echo $strSql."<br>";
    $this->dbConn->ExecuteSql($strSql);

    return $cont_id;
  }

  /**
   * Modifie un espace (paramètres liés à la base)
   */
  function updateSpace($user_id, $user_priv, $cont_id, $tabQuery, $cont_appli_defaut, $strPathUpload,
                       $cont_public_old, $cont_rang_old, $cont_alias_old)
  {
    $tabValue = $tabQuery["field"];
    
    // traitement de l'upload
    $strFileName = doUpload("cont_logo", $cont_id."_l1_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["CONT_LOGO"][1] = $strFileName;
    } else {
      unset($tabValue["CONT_LOGO"]);
    }
    $strFileName = doUpload("cont_logo2", $cont_id."_l2_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["CONT_LOGO2"][1] = $strFileName;
    } else {
      unset($tabValue["CONT_LOGO2"]);
    }
    $strFileName = doUpload("cont_bandeau", $cont_id."_b_", $strPathUpload);
    if( is_string($strFileName) ) {
      $tabValue["CONT_BANDEAU"][1] = $strFileName;
    } else {
      unset($tabValue["CONT_BANDEAU"]);
    }
    
    if( $cont_alias_old != $tabValue["CONT_ALIAS"][1] ) {
      // traitement du raccourci
      if( !$this->SaveLinkUrlSpace($cont_id, $cont_appli_defaut, $cont_alias_old, $tabValue["CONT_ALIAS"][1]) ) {
        $tabValue["CONT_ALIAS"][1] = "";
      }
    }
    
    $tabValue["AGENT_ID_MAJ"] = array(ALK_SQL_NUMBER, $user_id);
    $tabValue["CONT_DATE_MAJ"] = array(ALK_SQL_EXPR, $this->dbConn->getDateCur());
    
    // enregistre les modifs de la fiche conteneur
    $strSql = "update SIT_CONTENEUR set ".$this->getPartUpdateSql($tabValue)." where CONT_ID=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Supprime un espace (applis de l'espace dèjà supprimées) avant cet appel
   *
   * @param cont_id    Identifiant de l'espace à supprimer
   * @param cont_rang  Rang de l'espace
   * @param cont_logo  Nom du logo associée
   * @param cont_pere  identifiant de l'espace parent
   * @param cont_alias Intitulé de l'alias
   */
  function delSpace($cont_id, $cont_pere, $cont_rang, $cont_alias)
  {
    if( $cont_alias != "" )
      $this->DelLinkUrlSpace($cont_alias);

    // on garde le logo pour la bibliotheque

    // suppression des associations profils spé - agents
    $strSql = "delete from SIT_AGENT_PROFIL".
      " where PROFIL_ID in (select PROFIL_ID from SIT_PROFIL where CONT_ID=".$cont_id.")";
    $this->dbConn->ExecuteSql($strSql);
    
    // suppression des profils spé asscocié à l'espace
    $strSql = "delete from SIT_PROFIL where CONT_ID=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);

    // suppression des agents
    $strSql = "delete from SIT_AGENT_CONT where CONT_ID=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);

    // suppression association avec dept
    $strSql = "delete from SIT_DEPT_CONT where CONT_ID=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);

    // suppression des stats
    $strSql = "delete from SIT_STATISTIQUE where TYPE_CONNEXION=1 and ID_TYPE_CONNEXION=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);

    // suppression de l'espace
    $strSql = "delete from SIT_CONTENEUR where CONT_EFFACABLE=1 and CONT_ID=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Met à jour les associations entre espace et entité géographique
   * @param cont_id   identifiant de l'espace
   * @param tabDeptId tableau d'identifiants d'entités géographiques
   */
  public function delAddAssocContDept($cont_id, $tabDeptId)
  {
    $strSql = "delete from SIT_DEPT_CONT where CONT_ID=".$cont_id;
    $this->dbConn->ExecuteSql($strSql);
    
    $strListId = "0";
    if( count($tabDeptId)>0 ) {
      $strListId = implode(",", $tabDeptId);
    }
    $strSql = "insert into SIT_DEPT_CONT (CONT_ID, DEPT_ID)".
      " select ".$cont_id.", DEPT_ID from SIT_DEPT where DEPT_ID in (".$strListId.")";
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Met à jour les associations entre espace et entité géographique
   * @param cont_id   identifiant de l'espace
   * @param tabDeptId tableau d'identifiants d'entités géographiques
   */
  public function copyAssocContDept($cont_id_src, $cont_id_dest)
  {
    $strSql = "delete from SIT_DEPT_CONT where CONT_ID=".$cont_id_dest;
    $this->dbConn->ExecuteSql($strSql);
    
    $strSql = "insert into SIT_DEPT_CONT (CONT_ID, DEPT_ID)".
      " select ".$cont_id_dest.", DEPT_ID from SIT_DEPT_CONT where CONT_ID=".$cont_id_src;
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Création d'un fichier html
   *        nécessaire à la redirection auto vers un espace
   *
   * @param  cont_id     identifiant de l'espace
   * @param  appli_id    identifiant de l'appli par défaut
   * @param  strOldAlias ancien alias
   * @param  strAlias    alias de l'espace
   * @return Booleen : true si créé, false sinon
   */
  function SaveLinkUrlSpace($cont_id, $appli_id, $strOldAlias, $strAlias)
  {
    $queryCont = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
    $user_id = AlkFactory::getSProperty("user_id", "-1");

    // suppression de l'ancien s'il existe
    if( $strOldAlias!="" && $strOldAlias!="''" ) {
      $strPathFileName = ALK_ROOT_PATH."/".$strOldAlias.".htm";
      if( file_exists($strPathFileName) && is_file($strPathFileName) ) {
        unlink($strPathFileName);
      }
    }
   
    if( $strAlias=="" || $strAlias=="''" ) return false;

    $appli_id = ( $appli_id == 0 ? -1 : $appli_id );

    $strToken = AlkRequest::getEncodeParam("cont_id=".$cont_id."&appli_id=".$appli_id);
    $strUrl = ALK_ALKANET."?token=".$strToken;
    
    $strHtml = "<html>\n".
      "<head>\n".
      "<title>".$strAlias."</title>\n".
      "<META HTTP-EQUIV=\"Refresh\" CONTENT=\"0;URL=".$strUrl."\">\n".
      "</head>\n".
      "<body></body>\n".
      "</html>\n";
    
    // suppression du nouveau s'il existe
    $strPathFileName = ALK_ROOT_PATH."/".$strAlias.".htm";
    if( file_exists($strPathFileName) && is_file($strPathFileName) ) {
      unlink($strPathFileName);
    }

    // création du nouveau
    $bRes = false;
    $hFile = @fopen($strPathFileName, 'w');
    if( $hFile )	{
      $res = @fwrite($hFile, $strHtml);
      $bRes = ( $res>0 );
      fclose($hFile);
    }
    
    return $bRes;
  }

  /**
   * Supprime le raccourci
   *
   * @param strAlias Non de l'alias à supprimer
   */
  function DelLinkUrlSpace($strAlias)
  {
    if( $strAlias!="" && $strAlias!="''" ) {
      $strPathFileName = ALK_ROOT_PATH."/".$strAlias.".htm";
      if( file_exists($strPathFileName) && is_file($strPathFileName) )
        @unlink($strPathFileName);
    }
  }  

  /**
   * Recopie les agents de cont_id_src vers cont_id_dest (avec le privilège)
   * Recopie les droits des profils génériques, pas les droits des profils spécifiques
   * @param cont_id_src  Identifiant du conteneur source
   * @param cont_id_dest Identifiant du conteneur destination
   */
  public function dupSpaceUser($cont_id_src, $cont_id_dest)
  {
    // menage avant insertion
    $strSql = "delete from SIT_AGENT_CONT where CONT_ID=".$cont_id_dest;
    $this->dbConn->ExecuteSql($strSql);

    // recopie des agents
    $strSql = "insert into SIT_AGENT_CONT (AGENT_ID, CONT_ID, CONT_ADMIN) ".
      " select ac.AGENT_ID, ".$cont_id_dest.", ac.CONT_ADMIN".
      " from SIT_AGENT_CONT ac".
      " where ac.CONT_ID=".$cont_id_src;
    $this->dbConn->ExecuteSql($strSql);

    // pas d'appli à ce moment car 1ere étape de recopie, pas de recopie de droits
  }

  /**
   * Efface les associatons profil - appli
   *       09_profil_sql
   *
   * @param profil_id Identifiant du profil à supprimer
   */
  function del_profilAppli($profil_id)
  {
    $strSql = "delete from SIT_PROFIL_APPLI where PROFIL_ID=".$profil_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Met à jour les statistiques de l'agent
   * lib_sqlmenu.php
   *
   * @param agent_id        identifiant de l'agent
   * @param type_conn       type de connexion 0 = sit, 1=espace, 2=appli
   * @param type_conn_id    identifiant de l'espace ou de l'appli en fonction de type_conn, =0 pour sit
   */
  public function maj_statsByAgent($agent_id, $type_conn, $type_conn_id)
  {
    $strSql = "update SIT_STATISTIQUE set".
      " NB_CONNEXION=NB_CONNEXION+1".
      " where AGENT_ID=".$agent_id.
      " and MOIS_CONNEXION=".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
      " and ANNEE_CONNEXION=".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true).
      " and TYPE_CONNEXION=".$type_conn.
      " and ID_TYPE_CONNEXION=".$type_conn_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Ajoute les statistiques d'un agent
   * lib_sqlmenu.php
   *
   * @param agent_id        identifiant de l'agent
   * @param type_conn       type de connexion 0=sit, 1=espace, 2=appli
   * @param type_conn_id    identifiant de l'espace ou de l'appli en fonction de type_conn, =0 pour sit
   */
  public function add_statsByAgent($agent_id, $type_conn, $type_conn_id)
  {
    $strSql = "insert into SIT_STATISTIQUE (AGENT_ID, MOIS_CONNEXION, ANNEE_CONNEXION,".
      " NB_CONNEXION, TYPE_CONNEXION, ID_TYPE_CONNEXION) values (".$agent_id.
      ", ".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
      ", ".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true).
      ", 1".
      ", ".$type_conn.
      ", ".$type_conn_id.
      ")";
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * met à jour la statistique liée à une donnée data_id de l'application appli_id identifiée par action_id
   * @param atype_id    type de l'application
   * @param appli_id    identifiant de l'application
   * @param datatype_id type de l'information
   * @param data_id     identifiant de l'information
   * @param agent_id    identifiant de l'agent
   * @param action_id   identifiant de l'action déclenchée
   */
  public function updateStatAppli($atype_id, $appli_id, $datatype_id, $data_id, $agent_id, $action_id)
  {
    $strKey = $atype_id."_".$appli_id."_".$datatype_id."_".$data_id."_".$agent_id."_".$action_id; 

    $bLog = ( $datatype_id=="" && $data_id=="" );
    $bAdd = $bLog; 

    if( !$bLog && !isset($_SESSION["ALK_STAT_APPLI"][$strKey]) ) {
      $strSql = "select * from SIT_STAT_APPLI".
        " where ATYPE_ID=".$atype_id." and APPLI_ID=".$appli_id.
        " and DATATYPE_ID=".$datatype_id." and DATA_ID=".$data_id.
        " and AGENT_ID=".$agent_id." and ACTION_ID=".$action_id. 
        " and STAT_MOIS=".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
        " and STAT_ANNEE=".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true);
      $dsStat = $this->dbConn->initDataSet($strSql);
      $bAdd = ( $dsStat->getCountTotDr() == 0 ? true : false );  
    }
    
    if( !$bLog ) {
      $_SESSION["ALK_STAT_APPLI"][$strKey] = true;
    }
    if( $bAdd ) {
      $strSql = "insert into SIT_STAT_APPLI (ATYPE_ID, APPLI_ID, DATATYPE_ID, DATA_ID, AGENT_ID, STAT_MOIS, STAT_ANNEE,".
        " STAT_NOMBRE, ACTION_ID, STAT_DATE) values (".$atype_id.", ".$appli_id.", ".$datatype_id.", ".$data_id.", ".$agent_id.
        ", ".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
        ", ".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true).
        ", 1, ".$action_id.", ".$this->dbConn->GetDateCur().")";  
    } else {
      $strSql = "update SIT_STAT_APPLI set".
        " STAT_NOMBRE=STAT_NOMBRE+1".
        " where ATYPE_ID=".$atype_id." and APPLI_ID=".$appli_id.
        " and DATATYPE_ID=".$datatype_id." and DATA_ID=".$data_id.
        " and AGENT_ID=".$agent_id." and ACTION_ID=".$action_id. 
        " and STAT_MOIS=".$this->dbConn->GetDateFormat("MM", $this->dbConn->GetDateCur(), false, true).
        " and STAT_ANNEE=".$this->dbConn->GetDateFormat("YYYY", $this->dbConn->GetDateCur(), false, true);
    }

    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Met à jour le conteneur de démarrage d'un agent
   * @param agent_id  identifiant de l'agent
   * @param cont_id   identifiant du conteneur
   */
  public function maj_agentContIdStart($agent_id, $cont_id)
  {
    $strSql = "update SIT_AGENT set CONT_ID_START=".$cont_id." where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Met à jour la date de connexion de l'agent en cours
   * sit_menu/login_verif.php
   *
   * @param agent_id    Identifiant de l'agent
   * @param strInfo     Information sur le navigateur client
   * @param strWindSize taille de l'écran de l'utilisateur connecté (largeurxhauteur en pixels)
   */  
  public function maj_agentDateConn($agent_id, $strInfo, $strWindSize="", $strClientIp="")
  {
    $strInfo = "type=".$strInfo;
    if( $strWindSize != "" )
      $strInfo .= " screen=".$strWindSize;
   if( $strClientIp != "" )
      $strInfo .= " ip=".$strClientIp;

    $strSql = "update SIT_AGENT set".
      " AGENT_DATE_CONN=".$this->dbConn->GetDateCur().
      ", AGENT_BROWSER='".$this->dbConn->AnalyseSql($strInfo)."'".
      " where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Met à jour le mot de passe de l'agent
   *
   * @param agent_id   Identifiant de l'agent connecté
   * @param agent_pwd  Mot de passe de l'agent
   */
  function maj_agentPwd($agent_id, $agent_pwd)
  {
    $strSql = "update SIT_AGENT set AGENT_PWD='".$this->dbConn->AnalyseSql($agent_pwd)."' where AGENT_ID=".$agent_id;
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Met à jour la visibilité d'un espace pour un agent
   * @param cont_id  identifiant de l'espace
   * @param user_id  identifiant de l'agent connecté
   * @param iVisible =0 non visible, =1 visible
   */
  public function updateSpaceVisible($cont_id, $user_id, $iVisible)
  {
    $strSql = "update SIT_AGENT_CONT set ".
      " CONT_VISIBLE=".$iVisible.
      " where CONT_ID=".$cont_id." and AGENT_ID=".$user_id;
    $this->dbConn->ExecuteSql($strSql);
  }

  /**
   * Construit puis exécute les requêtes d'import
   * @param tabFieldImport      tableau contenant les infos à importer
   * @param tabPk               tableau d'association entre les nouveaux et anciens appli_id 
   * @param agent_id            identifiant du propriétaire des données importées
   * @param cont_id             identifiant de l'espace sur lequel, on importe les données
   * @param strPathZipExtract   chemin contenant les fichiers joints
   */
  public function import($tabFieldImport, $tabPk, $agent_id, $cont_id, $strPathZipExtract)
  {
    // détermination du service de l'agent
    $strLog = "";
    
    $service_id = 2;
    $strSql = "select s.SERVICE_ID".
      " from SIT_AGENT_SERVICE ase".
      "   inner join SIT_SERVICE s on ase.SERVICE_ID=s.SERVICE_ID".
      " where s.SERVICE_NIVEAU=1 and ase.AGENT_ID=".$agent_id;
    $dsServ = $this->dbConn->initDataset($strSql);
    if( $drServ = $dsServ->getRowIter() ) {
      $service_id = $drServ->getValueName("SERVICE_ID");
    } else {
      $strLog .= "Impossible de trouver le service du nouvel agent propriétaire.\r\n";
    }
    
    $tabFile = array();
    $tabPath = array("iedit_pj"   => ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD."iedit/", // branche 3.1.0 uniquement : .$cont_id."/",
                     "fdoc_01_pj" => ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD."fdoc/",
                     );
    $tabSql = array();
    
    foreach($tabFieldImport as $strTable => $tabRecords) {
      $strPk = $tabRecords["pk"];
      
      if( strpos($strPk, ",") === false ) {
        //echo $strTable.".".$strPk."<br>";
      } else {
        $strPk = "";
      }
      foreach($tabRecords as $numRes=>$tabFieldsI) {
        if( !is_numeric($numRes) ) continue;

        // remplace l'id de la pk
        if( $strPk != "" ) {
          $oldId = $tabFieldsI[$strPk][1];
          $newId = $this->dbConn->getNextId($strTable, $strPk, (strpos($strTable, "_pj")===false ? "SEQ_".$strTable : "SEQ_DATA_PJ"));
          if( $oldId != -1 )
            $tabPk[$strPk]["_".$oldId] = $newId; 
          //echo "PK ".$strPk." old=".$oldId." new=".$newId."<br>";
          $tabFieldsI[$strPk][1] = $newId;
        }
        
        foreach($tabFieldsI as $strFieldName => $tabData) {
          if( $strFieldName == $strPk ) continue;
          
          // cas particulier où la clé étrangère ne correspond pas à la clé primaire
          $strFk = ( $strTable == "fdoc_01_pj" && $strFieldName=="data_id" 
                     ? "doc_id"
                     : $strFieldName );   
          
          // remplace l'id du propriétaire
          if( !(strpos($strFieldName, "agent") === false) ) {
            $tabFieldsI[$strFieldName][1] = $agent_id; 
          }
          // remplace l'id du service propriétaire
          elseif( !(strpos($strFieldName, "service_id") === false) ) {
            $tabFieldsI[$strFieldName][1] = $service_id;
          }
          // remplace l'id des fk : si le champ correspond à une clé primaire
          elseif( isset($tabPk[$strFk]) ) {
            $oldId = $tabData[1]; 
            if( isset($tabPk[$strFk]["_".$oldId]) ) {
              $tabFieldsI[$strFieldName][1] = $tabPk[$strFk]["_".$oldId];
            } else {
              $strLog .= "Pas d'equivalent à ".$strTable.".".$strFieldName." - ".$strPk."=".$oldId."\r\n";
            }
          } 
          // remplace l'id du parent
          elseif( !(strpos($strFieldName, "_pere") === false) || !(strpos($strFieldName, "_parent") === false) ) {
            $oldId = $tabData[1];
            if( $oldId > 0 ) {  
              if( isset($tabPk[$strPk]["_".$oldId]) ) {
                $tabFieldsI[$strFieldName][1] = $tabPk[$strPk]["_".$oldId];
              } else {
                $strLog .= "Pas d'equivalent pour le parent ".$strTable.".".$strFieldName."=".$oldId."\r\n";
              }
            }
          }
          // remplace les id dans arbre
          elseif( !(strpos($strFieldName, "_arbre") === false) ) {
            $oldIds = substr($tabData[1], 1, -1);
            $tabOldIds = explode("-", $oldIds);
            $tabNewIds = array();
            foreach($tabOldIds as $oldId) {
              if( $oldId > 0 ) {
                if( isset($tabPk[$strPk]["_".$oldId]) ) {
                  $tabNewIds[] = $tabPk[$strPk]["_".$oldId];
                } else {
                  $strHtml .= "Pas d'equivalent pour l'arbre ".$strTable.".".$strFieldName." ".$tabData[1]." = ".$oldId."\r\n";
                }
              } else {
                $tabNewIds[] = "0";
              } 
            }
            $tabFieldsI[$strFieldName][1] = "-".implode("-", $tabNewIds)."-";
          }
          // mémorise les noms de fichiers à déplacer
          elseif( $strFieldName == "pj_file" || !(strpos($strFieldName, "_pj") === false) ) {
            $tabFile[$strTable][] = $tabData[1];
          }
          
        }
        
        $tabSql[] = "insert into ".$strTable." ".$this->getPartInsertSql($tabFieldsI);
      }
    }
    
    // exécution des requêtes
    $nbReq = 0;
    if( $strLog == "" ) {
      foreach($tabSql as $strSql) { 
        //echo $strSql."<br/>"; 
        $bRes = $this->dbConn->executeSql($strSql);
        if( !$bRes ) {
          $strLog .= "Erreur sur la requête : ".$strSql."\r\n";
          break;
        }
        $nbReq++;
      }
      $strLog .= "Nombre de requêtes exécutées : ".$nbReq."/".count($tabSql)."\r\n";
    }
    
    // déplacement des fichiers
    if( $nbReq > 0 && $nbReq == count($tabSql) ) {
      foreach($tabFile as $strTable => $tabAppliFile) {
        if( isset($tabPath[$strTable]) ) {
          foreach($tabAppliFile as $strFileName) {
            $bRes = @rename($strPathZipExtract.$strFileName, $tabPath[$strTable].$strFileName);
            if( !$bRes ) {
              $strLog .= "Impossible de déplacer ".$strFileName." vers ".str_replace(ALK_ALKANET_ROOT_PATH, ALK_ROOT_DIR, $tabPath[$strTable])."\r\n";
            } else {
              $strLog .= "Déplacement du fichier ".$strFileName." vers ".str_replace(ALK_ALKANET_ROOT_PATH, ALK_ROOT_DIR, $tabPath[$strTable])."\r\n";
            }
          }
        } else {
          $strLog .= "Pas de répertoire destination pour la table ".$strTable."\r\n"; 
        }
      }
    }
        
    return $strLog;
  }
  
  /**
   * Ajoute une nouvelle tâche planifiée
   * @param valeur des champs de la table SIT_CRON_TASK
   * @return identifiant de la tâche ajoutée
   *         -1 si erreur
   */
  public function addCronTask($cont_id=0, $atype_id, $appli_id=-1, $action_id, $task_typetask=0, $task_date="", 
                              $task_jourssemaine=0, $task_joursmois=0, $task_mois=0, $task_typetime=0, $task_time="")
  {
    $task_id = $this->dbConn->getNextId("SIT_CRON_TASK", "TASK_ID", "SEQ_SIT_CRON_TASK");
    
    $strSql = "insert into SIT_CRON_TASK (" .
      "  TASK_ID" .
      ", TASK_DATECREA" .
      ", CONT_ID" .
      ", ATYPE_ID" .
      ", APPLI_ID" .
      ", ACTION_ID" .
      ", task_TYPETASK" .
      ", task_DATE" .
      ", task_JOURSSEMAINE" .
      ", task_JOURSMOIS" .
      ", task_MOIS" .
      ", task_TYPETIME" .
      ", task_TIME" .
      ") values (" .
      "  ".$task_id .
      ", ".$this->dbConn->getDateCur() .
      ", ".$cont_id .
      ", ".$atype_id .
      ", ".$appli_id .
      ", ".$action_id .
      ", ".$task_typetask .
      ", ".( $task_date != "" ? $this->dbConn->getDateFormat("DD/MM/YYYY", "'".$task_date."'") : "NULL" ) .
      ", ".$task_jourssemaine .
      ", ".$task_joursmois .
      ", ".$task_mois .
      ", ".$task_typetime .
      ", '".$this->dbConn->analyseSql($task_time)."'" .
      ")";
    
    $iRes = $this->dbConn->executeSql($strSql);
    if ( $iRes > 0 ) {
      return $task_id;
    } else {
      return -1;
    }
  }
  
  /**
   * Met à jour une tâche planifiée existante
   * @param valeur des champs de la table SIT_CRON_TASK
   * @return identifiant de la tâche modifiée
   *         -1 si erreur
   */
  public function updateCronTask($task_id, $cont_id=0, $atype_id, $appli_id=-1, $action_id, $task_typetask=0, $task_date="", 
                              $task_jourssemaine=0, $task_joursmois=0, $task_mois=0, $task_typetime=0, $task_time="")
  {
    $strSql = "update SIT_CRON_TASK set " .
      "  CONT_ID=".$cont_id .
      ", ATYPE_ID=".$atype_id .
      ", APPLI_ID=".$appli_id .
      ", ACTION_ID=".$action_id .
      ", task_TYPETASK=".$task_typetask .
      ", task_DATE=".( $task_date != "" ? $this->dbConn->getDateFormat("DD/MM/YYYY", "'".$task_date."'") : "NULL" ) .
      ", task_JOURSSEMAINE=".$task_jourssemaine .
      ", task_JOURSMOIS=".$task_joursmois .
      ", task_MOIS=".$task_mois .
      ", task_TYPETIME=".$task_typetime .
      ", task_TIME='".$this->dbConn->analyseSql($task_time)."'" .
      " where TASK_ID=".$task_id;
    
    $iRes = $this->dbConn->executeSql($strSql);
    if ( $iRes > 0 ) {
      return $task_id;
    } else {
      return -1;
    }
  }
  
  /**
   * Supprime une tâche planifiée existante
   * @param task_id   identifiant de la tâche planifiée
   * @return identifiant de la tâche supprimée
   *         -1 si erreur
   */
  public function deleteCronTask($task_id)
  {
    $strSql = "delete from SIT_CRON_TASK where TASK_ID=".$task_id;
    
    $iRes = $this->dbConn->executeSql($strSql);
    if ( $iRes > 0 ) {
      return $task_id;
    } else {
      return -1;
    }
  }
  
  /**
   * Met à jour les constantes
   * @param tableau tableau des constantes à mettre à jour
   * @param appli_id identifiant de l'appli courante par rapport à laquelle il faut mettre à jour la constante
   * @param cont_id identifiant de l'espace courant par rapport auquel il faut mettre à jour la constante
   * @param atype_id atype_id de l'appli courante
   * 
   */
  public function updateConstantes($tableau, $appli_id, $cont_id, $atype_id){
    
    // ménage
    $strSql = " delete from SIT_CONSTANTE ".
              " where ATYPE_ID=".$atype_id." and APPLI_ID=".$appli_id. " and CONT_ID=".$cont_id; 
    $this->dbConn->executeSql($strSql);
        
    foreach($tableau as $const_intitule =>$const_valeur){      
      if ($const_valeur != ALK_FIELD_NOT_VIEW && strpos($const_intitule, "ALK_") !== false){        
        // insertion 
        $strSql =" insert into SIT_CONSTANTE (ATYPE_ID, CONT_ID, APPLI_ID, CONST_INTITULE, CONST_VALEUR, CONST_LABEL, CONST_TYPE, ".
                                              "CONST_ADMIN, CONST_LEVEL, CONST_BLOC)".
                 " select ".$atype_id.",".$cont_id.",".$appli_id.",CONST_INTITULE,'".$this->dbConn->analyseSql($const_valeur)."',".
                 " CONST_LABEL, CONST_TYPE, CONST_ADMIN, CONST_LEVEL, CONST_BLOC".
                 " from SIT_CONSTANTE ".
                 " where CONST_INTITULE='".$const_intitule."' and CONT_ID=-1";
        $this->dbConn->executeSql($strSql);
      }
    }
  }
  
  /**
   * supprime une connexion
   * @param agent_id identifiant de l'utilisateur
   * @param session_id identifiant de la session
   * @param data_down date de référence pour la suppression des connexions (format date JJ/MM/AAAA)
   */
  public function deleteConnexionUser($agent_id=-1, $session_id=-1, $date_down=""){
    if ($agent_id!= -1 && $session_id!= -1){
      $strSql = " delete from SIT_AGENT_CONN where AGENT_ID=".$agent_id.
              " and SESSION_ID='".$session_id."'";
      $this->dbConn->executeSql($strSql);
    }
    if ($date_down != ""){
      $date_fin    = $this->dbConn->getDateFormat("DD/MM/YYYY HH:MI", "'".$date_down." 23:59'", true);
      $date_debut  = $this->dbConn->getDateFormat("DD/MM/YYYY HH:MI", "'".$date_down." 00:00'", true);
      $strSql = " delete from SIT_AGENT_CONN where DATE_CONN <=".$date_fin." and DATE_CONN >=".$date_debut;
      $this->dbConn->executeSql($strSql);
    }
  }
  
  /**
   * enregistre une connexion
   * @param agent_id identifiant de l'utilisateur
   * @param session_id identifiant de la session
   * @param cont_id identifiant de l'espace de connexion
   * @param appli_id identifiant de l'application de connexion
   */
  public function insertConnexionUser($agent_id, $session_id, $cont_id, $appli_id){
      $strSql = "insert into SIT_AGENT_CONN(AGENT_ID, SESSION_ID, DATE_CONN, CONT_ID, APPLI_ID) values(".
               $agent_id.",'".$session_id."',".$this->dbConn->getDateCur().",".$cont_id.",".$appli_id.")";
      $this->dbConn->executeSql($strSql);
  }
  
    /**
   * Efface les categories d'espace'
   * @param type_id Identifiant d'une categorie d'espace à supprimer
   */
  function delDataCategorie($type_id)
  {
    $strSql = "delete from SIT_CONT_TYPE where TYPE_ID=".$type_id;
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Ajouter une option d'application
   * @param int $appli_id
   * @param string $alias
   * @param string $intitule
   * @param string $description
   */
  public function addOption($appli_id, $alias, $intitule, $description) {
    $tabValue['OPTION_ID']       = array(ALK_SQL_NUMBER, $this->dbConn->GetNextId("SIT_APPLI_OPTION", "OPTION_ID", "SEQ_SIT_APPLI_OPTION"));
    $tabValue['APPLI_ID']        = array(ALK_SQL_NUMBER, $appli_id);
    $tabValue['OPTION_ALIAS']    = array(ALK_SQL_TEXT, $alias);
    $tabValue['OPTION_INTITULE'] = array(ALK_SQL_TEXT, $intitule);
    $tabValue['OPTION_DESC']     = array(ALK_SQL_TEXT, $description);
  
    $strSql = "insert into SIT_APPLI_OPTION ".$this->getPartInsertSql($tabValue);
    $this->dbConn->ExecuteSql($strSql);
  }
  
  /**
   * Supprimer une option d'application
   * @param int $option_id
   */
  public function removeOption($option_id) {
    $strSql = "DELETE FROM sit_appli_option WHERE option_id = ".$option_id;
    // TODO remove associated ACL...
    trigger_error(__CLASS__."::".__FUNCTION__." - TODO remove associated ACL...", E_USER_WARNING);
    return $this->dbConn->executeSql($strSql);
  }
  
    /**
   * modifie des options aux applications
   * @param identifiant de l'application donnée
   * @param options liste des options à rajouter
   */
  public function updateOptionsToAppli($appli_id, $options){
    $bOK = true;
    foreach ($options as $option_id => $value){
      $strSql = " update SIT_APPLI_OPTION set ".
                " OPTION_ACTIVEE = ". ($value == -1 ? "NULL" : $value).
                ", OPTION_DROITS_AVANCES=".($value == -1 ? 1 : 0).
                " where APPLI_ID=".$appli_id." and OPTION_ID=".$option_id;
      $bOK = $bOK && $this->dbConn->ExecuteSql($strSql);
    }
    return $bOK;
  }
}

?>