<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Module::Espace
Module gestionnaire d'espaces.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkappli.class.php");
require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkintspace.int.php");
require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkintservice.int.php");
require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."pattern/alkintoptions.int.php");

/**
 * @package Alkanet_Module_Espace
 * Module gestionnaire d'espace
 *
 * @class AlkAppliEspace
 * Classe regroupant des fonctionnalités des espaces disponibles
 * pour d'autres applications
 */
class AlkAppliEspace extends AlkAppli implements AlkIntSpace, AlkIntRights, AlkIntService, AlkIntOptions
{
  /** identifiant de l'espace */
  public $cont_id;

  /** tableau mémorisant la liste des applications */
  protected $tabMenuAppli;
  protected $tabMenuSpaceChild;
  protected $tabMenuSpaceBrother;
  protected $tabMenuSpaceOther;

  /**
   * Constructeur par défaut
   *
   * @param oSpace Référence vers l'objet de l'espace en cours
   */
  public function __construct($cont_id)
  {
    parent::__construct(ALK_ATYPE_ID_ESPACE, -1);
    $this->cont_id = $cont_id;
  }   

  /**
   * Destructeur par défaut
   */
  public function __destruct()
  {  
  }

  /**
   * Met à jour le privilège utilisateur en fonction du privilège sur l'espace
   *         - active le bit ALK_PRIV_SPACE_USER si l'utilisateur est invité à l'espace
   *         - active le bit ALK_PRIV_SPACE_ANIM si l'utilisateur est animateur de l'espace
   *        Si le bit ALK_PRIV_SPACE_USER n'est pas activé, le privilège devient ALK_PRIV_SPACE_NONE (aucun droit)
   */ 
  public function verifIfUserIsAnim()
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $user_priv = AlkFactory::getSProperty("user_priv", "0");
    if( ($user_priv & ALK_PRIV_SPACE_ADMIN) == ALK_PRIV_SPACE_ADMIN ) {
      $user_priv = $user_priv | ALK_PRIV_SPACE_ANIM | ALK_PRIV_SPACE_USER;
    } elseif( ($user_priv & ALK_PRIV_SPACE_ANIM) != ALK_PRIV_SPACE_ANIM ) {
      $cont_admin = $this->getAppliProperty("CONT_ADMIN");
      switch( $cont_admin ) {
      case 0: $user_priv = $user_priv | ALK_PRIV_SPACE_USER; break;
      case 1: $user_priv = $user_priv | ALK_PRIV_SPACE_ANIM | ALK_PRIV_SPACE_USER; break;
      }
    }
    if( ($user_priv & ALK_PRIV_SPACE_USER) != ALK_PRIV_SPACE_USER ) {
      $user_priv = ALK_PRIV_SPACE_NONE;
    }
    AlkFactory::setSProperty("user_priv", $user_priv);
  }

  /**
   * Retourne le droit utilisateur sur l'appli en cours
   * @return int
   */
  public function getUserAppliRight()
  {
    $iRight = ALK_APPLI_RIGHT_NONE;
    $user_priv  = AlkFactory::getSProperty("user_priv", "0");

    $iRight = ( ($user_priv & ALK_PRIV_SPACE_VIEWER) == ALK_PRIV_SPACE_VIEWER ||
                ($user_priv & ALK_PRIV_SPACE_USER) == ALK_PRIV_SPACE_USER 
                ? ALK_APPLI_RIGHT_READ 
                : $iRight ); 

    $iRight = ( (ALK_B_SPACE_ADMIN_ANIM && ($user_priv & ALK_PRIV_SPACE_ANIM) == ALK_PRIV_SPACE_ANIM) ||
                (($user_priv & ALK_PRIV_SPACE_ADMIN) == ALK_PRIV_SPACE_ADMIN)
                ? ALK_APPLI_RIGHT_PUBLI 
                : $iRight ); 

    return $iRight;
  }

  /**
   * Retourne l'objet Panel correspondant au iSheet et iSSheet sélectionné
   * 
   * @param oAppliFrom    AlkAppli   Application appelant cette fonction (principe call services par dérivation)
   * 
   * @return AlkHtmlPanel
   */
  public function getPanel(AlkAppli $oAppliFrom=null)
  {
    $oPanel = null;
    switch( $this->iTypeSheet ) {
    case ALK_TYPESHEET_CONSULT: // partie consultation
      switch ( $this->iSheet ){
      case ALK_SHEET_RECHERCHE : // onglet recherche
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformrechercheform.class.php");
        $oPanel = new AlkHtmlFormRechercheForm($this);
        break;
      case ALK_SHEET_ARBORESCENCE : // onglet plan de l'espace
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformsommaireform.class.php");
        $oPanel = new AlkHtmlFormSommaireForm($this);
        break;     
      case ALK_SHEET_TABLE : // onglet tableau de bord
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformtableaubordform.class.php");
        $oPanel = new AlkHtmlFormTableauBordForm($this);
        break;     
      case ALK_SHEET_MODIFIER : // requete ajax mémorisant un espace visible ou non dans le menu pour l'utilisateur connecté
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformsommaireform.class.php");
        $oPanel = new AlkHtmlFormSommaireForm($this);
        $oPanel->doSql();
        break;
      case ALK_SHEET_APPLICATION: // onglet application
        switch( $this->iSSheet ) {
        case ALK_SHEET_USERS_LIST : // requete ajax retournant la liste des utilisateurs sur sélection d'un groupe (prop. appli)
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformappliform.class.php");
          $oPanel = AlkHtmlFormAppliForm::getUsersPanel();
          break;
        }
      }
      break;
      
    case ALK_TYPESHEET_ADMIN: // partie admin
      switch ( $this->iSheet ){
      case ALK_SHEET_FICHE : // onglet propriété de l'espace
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformespace.class.php");
        $oPanel = new AlkHtmlFormEspace($this);
        break;
      case ALK_SHEET_ANIMATEUR: // gestion des animateurs
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformanimform.class.php");
        $oPanel = new AlkHtmlFormAnimForm($this);
        break;

      case ALK_SHEET_APPLICATION: // onglet application
        switch ( $this->iSSheet ){
        case ALK_SHEET_LIST : // liste d'application 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformapplilist.class.php");
          $oPanel = new AlkHtmlFormAppliList($this);
          break;
        case ALK_SHEET_FORM : // propriétés de l'application
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformappliform.class.php");
          $oPanel = new AlkHtmlFormAppliForm($this);
          break;
        case ALK_SHEET_USERS_LIST : // requete ajax retournant la liste des utilisateurs sur sélection d'un groupe (prop. appli)
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformappliform.class.php");
          $iDel = AlkRequest::getToken("idelact", "0");
          if( $iDel == "1" ) {
            $profil_id = AlkRequest::getToken("profil_id", "-1");
            $oQueryAnnuAction = AlkFactory::getQueryAction(ALK_ATYPE_ID_ANNU);
            $tabListUserId = $oQueryAnnuAction->deladdUsersToProfil($profil_id, false);
            $this->setProfilRightToUser($tabListUserId);
          } 
          $oPanel = AlkHtmlFormAppliForm::getUsersPanel();
          if( $iDel == "1" ) {
            echo $oPanel->getHtml();
            exit(); 
          }
          break;
        }
        break;
        
      case ALK_SHEET_USER_LIST : // gestion des invitation à l'espace / via heritage de lsdiff
        $appli_id_lisdif = AlkRequest::getToken("appli_id_lisdif", -1);
        if($appli_id_lisdif!=-1){
          $oAppliLsDif = AlkFactory::getAppli(ALK_ATYPE_ID_LSDIF, $appli_id_lisdif);
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlforminvitationform.class.php");
          $oPanel = new AlkHtmlFormInvitationForm($oAppliLsDif);
        }
        break;      
        
      case ALK_SHEET_UTILISATEUR: // gestion des utilisateurs
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformuserlist.class.php");
        $oPanel = new AlkHtmlFormUserList($this);
        break;

      case ALK_SHEET_PROFIL : // onglet profil
        switch ( $this->iSSheet ){
        case ALK_SHEET_LIST : // liste 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformprofillist.class.php");
          $oAppli = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
          $oPanel = new AlkHtmlFormProfilList($oAppli);
          $oPanel->setContIdAssoc($this->cont_id);
          break;
        case ALK_SHEET_FORM : // propriétés 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."annu/classes/alkhtmlformprofilform.class.php");
          $oAppli = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
          $oPanel = new AlkHtmlFormProfilForm($oAppli);
          $oPanel->setContIdAssoc($this->cont_id);
          break;
        }
        break;

      case ALK_SHEET_STATS: // gestion des statistiques de l'espace'
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformstatlist.class.php");
        $oPanel = new AlkHtmlFormStatList($this);
        break;
        
      case ALK_SHEET_LINK:
       require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlveriflink.class.php");
       $oPanel = new AlkHtmlVerifLink($this);
       break;
       	
      }
      break;
      
    case ALK_TYPESHEET_PROPRIETE: // partie propriété
      switch ( $this->iSheet ){
      case ALK_SHEET_PLANIF : // onglet planification de tâches
        switch ( $this->iSSheet ) {
          case ALK_SHEET_LIST :
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformplaniflist.class.php");
            $oPanel = new AlkHtmlFormPlanifList($this);
          break;
          case ALK_SHEET_FORM :
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformplanifform.class.php");
            $oPanel = new AlkHtmlFormPlanifForm($this);
          break;
        }
        break;
        
       case ALK_SHEET_PARAMETRE:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformparametreform.class.php");
        $oPanel = new AlkHtmlFormParametreForm($this, $this, $this->atype_id);
        break;

      case ALK_SHEET_RUBRIQUE : // gestion des catégories d'espaces
        switch ( $this->iSSheet ){
        case ALK_SHEET_LIST : // liste d'application 
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformcategorielist.class.php");
          $oPanel = new AlkHtmlFormCategorieList($this);          
          break;
        case ALK_SHEET_FORM : // propriétés de l'application
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformcategorieform.class.php");
          $oPanel = new AlkHtmlFormCategorieForm($this);
          break;
        }
        break;
        
      }
      break;

    case ALK_TYPESHEET_POPUP: // partie popup
      switch ( $this->iSheet ) {
      case ALK_SHEET_ARBORESCENCE:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupespacearbo.class.php");
        $oPanel = new AlkHtmlPopupEspaceArbo($this);
        break;
      case ALK_SHEET_USERRIGHT:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupuserright.class.php");
        $oPanel = new AlkHtmlPopupUserRight($this);
        break;
      case ALK_SHEET_PROFILRIGHT:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupprofilright.class.php");
        $oPanel = new AlkHtmlPopupProfilRight($this);
        break;
      case ALK_SHEET_CREATEUSER:
        if(file_exists(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."classes/alkhtmlpopupcreateuser.class.php") && 
           is_file(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."classes/alkhtmlpopupcreateuser.class.php")){
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CONF."classes/alkhtmlpopupcreateuser.class.php");
          $oPanel = new AlkHtmlPopupCreateUserRespire($this);
        } else {
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupcreateuser.class.php");
          $oPanel = new AlkHtmlPopupCreateUser($this);
        }
        break;
      case ALK_SHEET_USER :
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupguestuser.class.php");
        $oPanel = new AlkHtmlPopupGuestUser($this);
        break;
      case ALK_SHEET_USER_MOTIVATION: 
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupaskguestuser.class.php");
        $oPanel = new AlkHtmlPopupAskGuestUser($this);
        break; 
      case ALK_SHEET_CONTACT:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupcontact.class.php");
        $oPanel = new AlkHtmlPopupContact($this);
        break;
        
      case ALK_SHEET_CHANGEPWD:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupchangepwd.class.php");
        $oPanel = new AlkHtmlPopupChangePwd($this);
        break;      
      case ALK_SHEET_FORM:
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlpopupperf.class.php");
        $oPanel = new AlkHtmlPopupPerf($this);
        break;    
      case ALK_SHEET_LABELS://Tags, Mot-clés, Thésaurus
        $params = AlkRequest::getToken("params", "");
        $tabParams = json_decode($params, true);
        $tabInit = $tabParams["_INIT_"];
        unset($tabParams["_INIT_"]);
        
        $oPanel = AlkHtmlFactory::getNewHtmlTag($tabInit["formName"], $tabInit["iMode"], $tabInit["name"], $tabInit["value"], $tabInit["label"], $tabParams, $tabInit["separator"]);
        $oPanel->setBMultiLanguage($tabInit["bMultiLanguage"]);
        break;    
      case ALK_SHEET_CONFIRM_RESA : // fenêtre de confirmation en mode "modal dialog"
        require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformconfirmationform.class.php");
        $oPanel = new AlkHtmlFormConfirmationForm($this);
        break;
      case ALK_SHEET_OPTIONS:
          require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/classes/alkhtmlformoptionsform.class.php");
          $oPanel = new AlkHtmlFormOptionsForm($this);
         break;
      }        
      break;
    }
    return $oPanel;
  }

  /**
   * Ajout d'un espace. Retourne l'identifiant de l'espace créé
   * @param tabFields  tableau associatif des champs nécessaires à la création d'un espace
   * @return int
   */
  public function addSpace($tabFields)
  {
    // à ce niveau, $this correspondant au futur espace parent
    $cont_niveau       = $this->getAppliProperty("CONT_NIVEAU")+1;
    $cont_arbre        = $this->getAppliProperty("CONT_ARBRE");
    $cont_pere_public  = $this->getAppliProperty("CONT_PUBLIC");
    $cont_pere_reserve = $this->getAppliProperty("CONT_RESERVE");
    $cont_pere         = $this->getAppliProperty("CONT_ID");
    
    $cont_appli_defaut = $this->getAppliProperty("CONT_APPLI_DEFAUT");
    $cont_appli_defaut = ( $cont_appli_defaut <= 0 ? -1 : $cont_appli_defaut );
    
    $cont_public  = $tabFields["field"]["CONT_PUBLIC"][1];
    $cont_reserve = 0;
    $typeCrea     =  $tabFields["field"]["TYPE_CREA"][1];
    $iTypeDup     =  ( isset($tabFields["field"]["TYPE_DUP"][1]) ? $tabFields["field"]["TYPE_DUP"][1] : 0 );
    
    unset($tabFields["field"]["TYPE_CREA"]);
    if( isset($tabFields["field"]["TYPE_DUP"]) ) { 
      unset($tabFields["field"]["TYPE_DUP"]);
    }
   
    // récupération de l'espace à dupliquer
    $cont_id_copy=-1;
    if( isset($tabFields["field"]["CTRL_ESPACE"][1])){
      $cont_id_copy = $tabFields["field"]["CTRL_ESPACE"][1];
      unset($tabFields["field"]["CTRL_ESPACE"]);
    }
    
    // récupération du type de duplicat de l'espace
    $iTypeDupEspace = 0;
     if( isset($tabFields["field"]["TYPEDUP_ESPACE"][1])){
      $iTypeDupEspace = $tabFields["field"]["TYPEDUP_ESPACE"][1];
      unset($tabFields["field"]["TYPEDUP_ESPACE"]);
    }
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $user_priv = AlkFactory::getSProperty("user_priv", "0");
    $strUpload = ALK_ROOT_UPLOAD.mb_strtolower(constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_ESPACE))."/";
    
    if( $typeCrea>0 ) {
      // recopie les propriétés communes dans le cas d'un espace clone
      $cont_public = $cont_pere_public;
      $cont_reserve = $cont_pere_reserve;
    }

    $tabFields["field"]["CONT_NIVEAU"] = array(ALK_SQL_NUMBER, $cont_niveau);
    $tabFields["field"]["CONT_ARBRE"] = array(ALK_SQL_TEXT, $cont_arbre);
    $tabFields["field"]["CONT_PERE"] = array(ALK_SQL_NUMBER, $cont_pere);
    $cont_intitule = $tabFields["field"]["CONT_INTITULE"][1]; 
    
    // on ne crée pas d'espace dans le cas dupSpace($typeCrea=4) car addSpace est rappelé avec typeCrea=3
    if($typeCrea < 4 ){
       $cont_id = $this->oQueryAction->addSpace($user_id, $user_priv, $tabFields, $cont_appli_defaut, $strUpload);
       // invite les utilisateurs ayant le privilère ALK_PRIV_SPACE_ADMIN et ALK_PRIV_SPACE_VIEWER
       $this->addUserToSpace(-1, $cont_id, ALK_PRIV_SPACE_ADMIN+ALK_PRIV_SPACE_VIEWER);
      if( $cont_public == 1 ) {
        // association cont_id - dept_id
        $tabDeptId = AlkRequest::_POST("dept", array());
        $this->oQueryAction->delAddAssocContDept($cont_id, $tabDeptId);
      }

      switch( $typeCrea ) {
      case 0: // espace vierge
        // si le conteneur est public
        if( $cont_public == 1 ) {
          // ajoute les agents appartenant à un service de type état
          $this->addUserToSpace(-1, -$cont_id, ALK_PRIV_SPACE_USER);
        }
        break;
        
      case 1: // recopie les applis de l'espace parent + applications des droits pour animateurs
        $this->dupSpaceAppli($cont_pere, $cont_id, $iTypeDup);
        break;
  
      case 2: // recopie les utilisateurs de l'espace parent
        $this->dupSpaceUser($cont_pere, $cont_id);
        break;
  
      case 3: // recopie les applis et les utilisateurs + droits sur les utilisateurs
        $this->dupSpaceUser($cont_pere, $cont_id);
        $this->dupSpaceAppli($cont_pere, $cont_id, $iTypeDup);
        break;
      }
    } else {
      // recopie d'un espace, le positionne en tant que frère de cont_pere 
      $cont_id = $this->dupSpace($cont_id_copy, $cont_pere, $cont_intitule, $tabFields["field"]["CONT_INTITULE_COURT"][1], $iTypeDupEspace);
    }

    // traitement mail
    if( ALK_B_SPACE_SEND_MAIL == true ) {
      $iSendMailAdd = AlkRequest::_POSTint("iSendMail", "0");
      if( $iSendMailAdd == "1" ) {
        $this->sendMailToAdmin($cont_id, $cont_intitule, $cont_public, 1);       
      }
    }

    return $cont_id;
  }

  /**
   * Modification d'un espace
   * @param cont_id    identifiant de l'espace
   * @param agent_id   identifiant de l'utilisateur responsable de la modif
   * @param tabFields  tableau associatif des champs nécessaires à la création d'un espace
   */
  public function updateSpace($cont_id, $tabFields)
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $user_priv = AlkFactory::getSProperty("user_priv", "0");
    $strUpload = ALK_ROOT_UPLOAD.mb_strtolower(constant("ALK_ATYPE_ABREV_".ALK_ATYPE_ID_ESPACE))."/";

    // recupere les infos avant modif
    $cont_rang_old    = $this->getAppliProperty("CONT_RANG");
    $cont_public_old  = $this->getAppliProperty("CONT_PUBLIC");
    $cont_alias_old   = $this->getAppliProperty("CONT_ALIAS");
    $cont_appli_defaut = $this->getAppliProperty("CONT_APPLI_DEFAUT");
    $cont_appli_defaut = ( $cont_appli_defaut <= 0 ? -1 : $cont_appli_defaut );
    $cont_intitule = $tabFields["field"]["CONT_INTITULE"][1];
    
    $bRes = $this->oQueryAction->updateSpace($user_id, $user_priv, $cont_id, $tabFields, $cont_appli_defaut, $strUpload,
                                            $cont_public_old, $cont_rang_old, $cont_alias_old);

    $cont_public = $tabFields["field"]["CONT_PUBLIC"][1];

    if( $cont_public==1 ) {
      // association cont_id - dept_id
      $tabDeptId = AlkRequest::_POST("dept", array());
      $this->oQueryAction->delAddAssocContDept($cont_id, $tabDeptId);
      // ajoute ceux éventuellement oubliés ou appartenant à une nouvelle sélection
      $this->addUserToSpace(-1, -$cont_id, ALK_PRIV_SPACE_USER);
    }
    
    $oAppliAnnu = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
    $oAppliAnnu->syncEspaceListe($cont_id);
  }

  /**
   * Suppression d'un espace
   * @param cont_id  identifiant de l'espace
   */
  public function delSpace()
  {
    $cont_rang    = $this->getAppliProperty("CONT_RANG");
    $cont_logo    = $this->getAppliProperty("CONT_LOGO");
    $cont_logo    = $this->getAppliProperty("CONT_LOGO2");
    $cont_bandeau = $this->getAppliProperty("CONT_BANDEAU");
    $cont_pere    = $this->getAppliProperty("CONT_PERE");
    $strAlias     = $this->getAppliProperty("CONT_ALIAS");

    // suppression des listes associées à l'espace
    $oAppliAnnu = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
    $oAppliAnnu->delListe(-1, $this->cont_id);

    // suppression des applis de l'espace
    $this->delAppli(-1, -1);
    
    $this->oQueryAction->delSpace($this->cont_id, $cont_pere, $cont_rang, $strAlias);

    return $cont_pere;
  }

  /**
   * Déplacement d'un espace
   * @param cont_id       identifiant de l'espace déplacé
   * @param cont_id_dest  identifiant de l'espace nouveau parent
   */
  public function moveSpace($cont_id, $cont_id_dest)
  {
    $this->oQueryAction->moveSpace($cont_id, $cont_id_dest);
  }

  /**
   * Duplication d'un espace en un autre.
   *        Retourne l'identifiant de l'espace dupliqué
   *        Cette méthode se charge d'appelé la recopie des droits
   * @param cont_id_copy  Identifiant de l'espace à copier
   * @param cont_id_dest  Espace parent du nouvel espace copié
   * @param iTypeDup      type de duplication : 0 = application vierge, 
   *                                            1 = structure et paramétrage 
   *                                            2 = structure, paramétrage et données
   * @return int
   */
  public function dupSpace($cont_id_copy, $cont_id_dest, $strContIntitule="", $strContIntituleCourt="", $iTypeDup=0)
  {
    $oSpaceCopy = AlkFactory::getSpace($cont_id_copy);

    $strContIntitule = ($strContIntitule == "" ? "Copie de ".$oSpaceCopy->getAppliProperty("CONT_INTITULE") : $strContIntitule );
    $strContIntituleCourt = ($strContIntituleCourt == "" ? "Copie de ".$oSpaceCopy->getAppliProperty("CONT_INTITULE_COURT") : $strContIntituleCourt );
    
    // recopie appli + util + droits
    $typecrea = 3; 
    $tabFields = array("pk"    => array( ),
                       "field" => array("CONT_ID"             => array(ALK_SQL_NUMBER, -1), 
                                        "CONT_INTITULE_COURT" => array(ALK_SQL_TEXT,   $strContIntituleCourt),
                                        "CONT_INTITULE"       => array(ALK_SQL_TEXT,   $strContIntitule),
                                        "CONT_ALIAS"          => array(ALK_SQL_TEXT,   ""), 
                                        "CONT_PUBLIC"         => array(ALK_SQL_NUMBER, $oSpaceCopy->getAppliProperty("CONT_PUBLIC")),
                                        "CONT_DESC"           => array(ALK_SQL_TEXT,   $oSpaceCopy->getAppliProperty("CONT_DESC")),
                                        "CONT_LOGO"           => array(ALK_SQL_TEXT,   ""), 
                                        "CONT_LOGO2"          => array(ALK_SQL_TEXT,   ""),
                                        "CONT_BANDEAU"        => array(ALK_SQL_TEXT,   ""),
                                        "TYPE_CREA"           => array(ALK_SQL_NONE,   $typecrea),
                                        "TYPE_DUP"            => array(ALK_SQL_NONE,   $iTypeDup),
                                        "CONT_CSS"            => array(ALK_SQL_TEXT,   $oSpaceCopy->getAppliProperty("CONT_CSS")),
                                        "CONT_TYPE"           => array(ALK_SQL_TEXT,   $oSpaceCopy->getAppliProperty("CONT_TYPE"))));

    // création de l'espace copié comme espace fils de cont_id_copy
    $cont_id = $oSpaceCopy->addSpace($tabFields);

    if( $oSpaceCopy->getAppliProperty("CONT_PUBLIC") == "1" ) {
      // recopie les associations cont-dept
      $this->oQueryAction->copyAssocContDept($cont_id_copy, $cont_id);
    }

    // déplacement de l'espace créé
    $this->moveSpace($cont_id, $cont_id_dest);

    $oSpaceCopy = null;
    
    return $cont_id;
  }

  /**
   * Duplique les applications et les droits sur les profils de l'espace courant 
   *        identifié par cont_pere vers l'espace identifié par cont_id
   * @param cont_pere  identifiant de l'espace courant (this)
   * @param cont_id    identifiant de l'espace cible
   * @param iTypeDup      type de duplication : 0 = application vierge, 
   *                                            1 = structure et paramétrage 
   *                                            2 = structure, paramétrage et données
   */
  protected function dupSpaceAppli($cont_pere, $cont_id, $iTypeDup)
  {
    /* création de l'objet Space destinataire */
    $oSpaceDest = AlkFactory::getSpace($cont_id);
    $appli_id_defaut = $this->getAppliProperty("CONT_APPLI_DEFAUT");

    // mémorise l'association entre profil spécifique dupliqué pour ne pas les ajouter n fois si n appli à dupliquer
    $tabProfilSpeAssoc = array();

	  // liste des applis non liées de l'espace
    $dsAppli = $this->oQuery->getDs_listeAppliByEspace($cont_pere, 0, -1, false, false);
    while( $drAppli = $dsAppli->getRowIter() ) {
      $appli_id_src = $drAppli->getValueName("APPLI_ID");
      $appli_intitule = $drAppli->getValueName("APPLI_INTITULE");
      $atype_id = $drAppli->getValueName("ATYPE_ID");
      $appli_rang = $drAppli->getValueName("APPLI_RANG");
      
      $appli_defaut = ($appli_id_defaut==$appli_id_src ? "1" : "0");

      $tabFields = array("pk"    => array(),
                         "field" => array("APPLI_INTITULE" => array(ALK_SQL_TEXT,   $appli_intitule),
                                          "ATYPE_ID"       => array(ALK_SQL_NUMBER, $atype_id),
                                          "APPLI_RANG"     => array(ALK_SQL_NUMBER, $appli_rang),
                                          "APPLI_DEFAUT"   => array(ALK_SQL_NUMBER, $appli_defaut)));

      $appli_id = $oSpaceDest->addAppli($tabFields);
      
      // recopie les droits des profils génériques  et duplique les droits et profils spécifiques pour l'appli donné
      $tabProfilSpeAssoc = $this->oQueryAction->copyAppliRights($appli_id_src, $appli_id, $cont_pere, $cont_id, $tabProfilSpeAssoc);
      
      // Recopie des droits pour la gestion interne de l'application
      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) ) {
        if( $iTypeDup>0 && method_exists($oAppli, "dupAppli") )
          $oAppli->dupAppli($appli_id_src, $iTypeDup);

        if( method_exists($oAppli, "setRightsOnAppli") )
          $oAppli->setRightsOnAppli($atype_id, $appli_id);
      }
    }

    $oSpaceDest = null;
  }

  /**
   * Duplique les utilisateurs et les privilèges de l'espace courant 
   *        identifié par cont_pere vers l'espace identifié par cont_id
   * @param cont_pere  identifiant de l'espace courant (this)
   * @param cont_id    identifiant de l'espace cible
   */
  protected function dupSpaceUser($cont_pere, $cont_id)
  {
    $this->oQueryAction->dupSpaceUser($cont_pere, $cont_id);
  }

  /**
   * Ajout d'une application à un espace. Retourne l'identifiant de l'appli créée
   * @param $tabFields tableau associatif des champs nécessaires à la création d'une appli
   * @return int
   */
  public function addAppli($tabFields)
  {
    // création de l'application
    $appli_id_defaut = $this->getAppliProperty("CONT_APPLI_DEFAUT");
    $atype_id = $tabFields["field"]["ATYPE_ID"][1];
    
    if( isset($tabFields["field"]["CONT_ID"])){
      $this->cont_id = $tabFields["field"]["CONT_ID"][1];
    }
    
    $appli_id = $this->oQueryAction->addAppli($this->cont_id, $tabFields, $appli_id_defaut);
    
    if( !isset($tabFields["field"]["CONT_ID"])){
      $tabFields["field"]["CONT_ID"] = array(ALK_SQL_NUMBER, $this->cont_id);
    }
    
    if( !isset($tabFields["field"]["NOT_CALL_APPLI_ADD"]) ) {
      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addAppli") )
        $oAppli->addAppli($tabFields);
    }
    
    // enregistre les droits sur les profils pour cet appli
    $this->setRightsOnAppli($atype_id, $appli_id);
   
    return $appli_id;
  }

  /**
   * Modification d'une application
   * @param appli_id   Identifiant de l'application
   * @param tabFields  tableau associatif des champs nécessaires à la création d'une appli
   */
  public function updateAppli($appli_id, $tabFields)
  {
    $appli_id_defaut = $this->getAppliProperty("CONT_APPLI_DEFAUT");
    $atype_id = $tabFields["field"]["ATYPE_ID"][1];
    $this->oQueryAction->updateAppli($this->cont_id, $appli_id, $tabFields, $appli_id_defaut);
    
    $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
    if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "updateAppli") )
      $oAppli->updateAppli($tabFields);

    // enregistre les droits sur les profils pour cet appli
    $this->setRightsOnAppli($atype_id, $appli_id);
  }

  /**
   * Suppression d'une application d'un espace
   * @param appli_id  identifiant de l'appli
   * @param atype_id  type d'application
   */
  public function delAppli($appli_id, $atype_id)
  {
    $tabApplis = array();
    if( $appli_id == "-1" ) {
      $dsAppli = $this->oQuery->getDs_listeAppliByEspace($this->cont_id, 0, -1, false, true);
      while( $drAppli = $dsAppli->getRowIter() ) {
        $tabApplis[] = array("appli_id" => $drAppli->getValueName("APPLI_ID"), 
                             "atype_id" => $drAppli->getValueName("ATYPE_ID"));
      }
    } else {
      $tabApplis[] = array("appli_id" => $appli_id, "atype_id" => $atype_id);
    }
    
    $appli_id_defaut = $this->getAppliProperty("CONT_APPLI_DEFAUT");
    foreach($tabApplis as $tabAppli) {
      $atype_id = $tabAppli["atype_id"]; 
      $appli_id = $tabAppli["appli_id"];
      
      // Suppression Initialisation de l'application créée
      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delAppli") )
        $oAppli->delAppli($this->cont_id);

      // suppression de l'application
      $this->oQueryAction->delAppli($this->cont_id, $appli_id, $appli_id_defaut);
    }
  }

  /**
   * Déplacement d'une application d'un espace vers un autre
   *        Cette méthode se charge d'appelé la recopie des droits
   * @param $appli_id  identifiant de l'appli
   * @param $idContTo  identifiant de l'espace recevant l'appli
   */
  public function moveAppli($appli_id, $idContTo) { /** rien à faire */ }

  /**
   * Duplication d'une application en une autre.
   *        Retourne l'identifiant de l'appli dupliquée
   *        Cette méthode se charge d'appelé la recopie des droits
   *        Recopie les applications de cet espace vers cont_id_dest
   * @param $appli_id  identifiant de l'appli
   * @return int
   */
  public function dupAppli($appli_id) { /** rien à faire */ }

  /**
   * Ajout d'un utilisateur à l'annuaire.
   *        Retourne un identifiant de l'utilisateur créé
   * @param user_id   identifiant de l'utilisateur fraichement créé
   * @param tabFields tableau associatif des champs nécessaires à la création d'un utilisateur
   * @return int
   */
  public function addUser($user_id, $tabFields)
  {
    $oQueryAnnu = AlkFactory::getQuery(ALK_ATYPE_ID_ANNU);

    $user_priv    = $tabFields["field"]["AGENT_PRIV"][1];
    $service_id   = $tabFields["field"]["SERVICE_ID"][1];
    $service_etat = $oQueryAnnu->getStrServiceEtat($service_id);

    if( ($user_priv & ALK_PRIV_SPACE_VIEWER) == ALK_PRIV_SPACE_VIEWER ||
        ($user_priv & ALK_PRIV_SPACE_ADMIN)  == ALK_PRIV_SPACE_ADMIN ) {
      $this->addUserToSpace($user_id, -1, $user_priv);
    } 
    else if( $service_etat == "1" ) {
      $this->addUserToSpace($user_id, -1, ALK_PRIV_SPACE_USER);
    }
    
    // appelle addUser pour tous les types d'applis qui implément la méthode addUser()
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");
      if( $atype_id!=ALK_ATYPE_ID_ESPACE && $atype_id!=ALK_ATYPE_ID_ANNU ) {
        $oAppli = AlkFactory::getAppli($atype_id);
        if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addUser") )
          $oAppli->addUser($user_id, $tabFields);
      }
    }
  }

  /**
   * Modification d'un utilisateur
   * @param user_id   identifiant d'un utilisateur
   * @param tabFields tableau associatif des champs nécessaires à la modif d'un utilisateur
   */
  public function updateUser($user_id, $tabFields) 
  { 
    // fixe à nouveau le privilège sur tous les espaces
    if( ($tabFields["agent_priv"] & ALK_PRIV_SPACE_ADMIN) == ALK_PRIV_SPACE_ADMIN ) {
      $this->addUserToSpace($user_id, -1, ALK_PRIV_SPACE_ADMIN);
    }
    
    /** @todo si l'utilisateur appartient à un service de type état, 
     *        il faut l'inviter aux espaces publics ouverts à son dépt s'il ne l'est pas déjà */
    
    
    // appelle addUser pour tous les types d'applis qui implément la méthode updateUser()
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      if( $atype_id!=ALK_ATYPE_ID_ESPACE && $atype_id!=ALK_ATYPE_ID_ANNU ) {
        $oAppli = AlkFactory::getAppli($atype_id);
        if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "updateUser") )
          $oAppli->updateUser($user_id, $tabFields);
      }
    }
  }

  /**
   * Transfère de propriété d'un utilisateur à un autre
   *        Méthode à appeler avant suppression définitive d'un utilisateur
   * @param idUserFrom  identifiant de l'utilisateur perdant la propriété des ses données
   * @param idUserTo    identifiant de l'utilisateur récupérant la propriété des données
   */
  public function replaceUser($idUserFrom, $idUserTo)
  {
    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "replaceUser") )
        $oAppli->replaceUser($idUserFrom, $idUserTo);
    }

    // traitement sur appli spécifiques
    $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
    foreach($tabAtypeID as $atype_id) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "replaceUser") )
        $oAppli->replaceUser($idUserFrom, $idUserTo);
    }

    $this->oQueryAction->replaceUser($idUserFrom, $idUserTo);
  }

  /**
   * Suppression d'un utilisateur pour user_id>=10
   *        Les comptes (user_id<10) spéciaux ne sont pas supprimables
   * @param user_id  identifiant d'un utilisateur
   */
  public function delUser($user_id)
  {
    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "removeUserFromSpace") )
        $oAppli->removeUserFromSpace($user_id, -1);
    }
  
    // traitement sur appli spécifiques
    $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
    foreach($tabAtypeID as $atype_id) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "removeUserFromSpace") )
        $oAppli->removeUserFromSpace($user_id, -1);
    }

    $this->oQueryAction->removeUser($user_id);
  }

  /**
   * Méthode appelée après suppression d'un abonnement
   * @param abonne_id  identifiant de l'abonné, peut contenir une liste d'identifiant
   * @param appli_id   identifiant de l'application de laquelle on souhaite désabonné
   * @param liste_id   identifiant de la liste de laquelle on souhaite désabonné
   */
  public function delAbonnement($abonne_id, $appli_id=-1, $liste_id=-1)
  {
    $atype_id = $this->oQuery->getATypeID($appli_id);
    $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
    if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delAbonnement") ) {
      $oAppli->delAbonnement($abonne_id, $liste_id);
    }
  }

  /**
   * Méthode appelée avant suppression définitive d'un abonne
   * Attention, si un abonné est propriétaire d'une info, il ne faut pas la supprimer
   * Implémenter les actions effectuant le ménage interne à l'application
   * Retourne true si l'abonné reste supprimable, faux si il est propriétaire d'info
   * @param abonne_id  identifiant de l'abonné, peut contenir une liste d'identifiant
   */
  public function delAbonne($abonne_id)
  {
    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delAbonne") )
        $oAppli->delAbonne($abonne_id);
    }
  }

  /**
   * Méthode appelée avant suppression définitive d'une liste d'abonnés
   * @param liste_id  identifiant de la liste, peut contenir une liste d'identifiant
   */
  public function delListe($liste_id, $cont_id=-1)
  {
    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delListe") )
        $oAppli->delListe($liste_id);
    }
  }

  /** 
   * Ajout d'un profil sur l'extranet.
   *        Si l'application gère des droits, il faut y associer le droit NONE
   * @param profil_id  identifiant du profil à supprimer
   * @param cont_id    identifiant de l'espace associé, =0 pour tous les espaces (par défaut)
   */
  public function addProfil($profil_id, $cont_id=0)
  {
    if( $cont_id==0 || $cont_id==$this->cont_id ) {
      // ajout des droits sur les applis de cont_id si >0, de toutes les applis si =0
      $this->oQueryAction->addProfil($profil_id, $cont_id);
  
      $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
      while( $drAppliType = $dsAppliType->getRowIter() ) {
        $atype_id = $drAppliType->getValueName("ATYPE_ID");
  
        $oAppli = AlkFactory::getAppli($atype_id);
        if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addProfil") )
          $oAppli->addProfil($profil_id, $cont_id);
      }
      
      // traitement sur appli spécifiques
      $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
      foreach($tabAtypeID as $atype_id) {
        $oAppli = AlkFactory::getAppli($atype_id);
        if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addProfil") )
          $oAppli->addProfil($profil_id, $cont_id);
      }
    }
  }

  /** 
   * Suppression du profil sur la gestion des droits interne à l'appli
   * @param profil_id  identifiant du profil à supprimer
   */
  public function delProfil($profil_id)
  {
    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delProfil") )
        $oAppli->delProfil($profil_id);
    }
    
    // traitement sur appli spécifiques
    $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
    foreach($tabAtypeID as $atype_id) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delProfil") )
        $oAppli->delProfil($profil_id);
    }
    
    $this->oQueryAction->delProfil($profil_id);
  }
  
  /**
   * Supprime les informations annexes associées à l'application passé en paramètre
   * appel la méthode delAppliCatData() sur toute les applications de l'espace sauf l'application appelante
   * Si cat_id =-1 et data_id=-1, supprime toutes les informations annexes associées à cette application
   * Si cat_id!=-1 et data_id=-1, supprime toutes les informations annexes associées à cette classification cat_id
   * Si data_id!=-1, supprime toutes les informations annexes associées à cet donnée data_id 
   * 
   * @param appli_id_from  identifiant de l'application contenant l'information principale
   * @param cat_id         identifiant de la classification, =-1 par défaut
   * @param data_id        identifiant de l'information, =-1 par défaut
   */
  public function delDataAssoc($appli_id_from, $cat_id=-1, $data_id=-1)
  {
    $oDsAppli = $this->oQuery->getDs_listeAppliByEspace($this->cont_id);
    while ( $oDrAppli = $oDsAppli->getRowIter() ) {
      $atype_id = $oDrAppli->getValueName("ATYPE_ID");
      $appli_id = $oDrAppli->getValueName("APPLI_ID");
      if ( $appli_id_from != $appli_id ) {
        $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
        if ( $oAppli ) {
          $oAppli->delAppliCatData($appli_id_from, $cat_id, $data_id);
        }
      }
    }
  }
  
  /**
   * Application de droits sur les profils pour l'application
   * @note Appelé en config. espace avec gestion des droits sur les profils pour l'appli
   * @param atype_id  type d'application
   * @param appli_id  identifiant de l'application
   */
  public function setRightsOnAppli($atype_id, $appli_id)
  {
    $tabRight = AlkRequest::_POST("listRight", array(), "is_array");

    if( !empty($tabRight) ) { 
      // applique les droits profil pour l'application
      $this->oQueryAction->setRightsOnAppli($appli_id, $tabRight);

      // Meme action pour la gestion interne de l'application
      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "setRightsOnAppli") )
        $oAppli->setRightsOnAppli($atype_id, $appli_id);
    }
    // suppression du cache
    AlkFactory::memCacheFlush();
  }

  /**
   * Applique les droits spécifiques de l'utilisateur aux applis de l'espace
   * @param cont_id       identifiant d'utilisateur
   * @param user_id       identifiant de l'espace
   * @param iRight        non utilisé
   * @param iDefaultRight non utilisé
   */
  public function setUserRightsOnSpace($cont_id, $user_id, $iRight=-1, $iDefaultRight=-1)
  {
    $tabRight = AlkRequest::_POST("listRight", array(), "is_array");
    $tabMem   = AlkRequest::_POST("memRight", array(), "is_array");
   
    $this->oQueryAction->setUserRightsOnSpace($cont_id, $user_id, $tabRight, $tabMem);

    $tabAppliRight = array();
    foreach($tabRight as $i => $strRight) {
      $tabTmp = explode("-", $strRight);
      $tabAppliRight["_".$tabTmp[0]] = array("right" => $tabTmp[1], "default" => $tabMem[$i]);
    }

    // applique les droits sur les applis de l'espace
    $dsAppli = $this->oQuery->getDs_listeAppliByEspace($cont_id);
    while( $drAppli = $dsAppli->getRowIter() ) {
      $appli_id = $drAppli->getValueName("APPLI_ID");
      $atype_id = $drAppli->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "setUserRightsOnSpace") )
        $oAppli->setUserRightsOnSpace($cont_id, $user_id, 
                                      $tabAppliRight["_".$appli_id]["right"], 
                                      $tabAppliRight["_".$appli_id]["default"]);
      // callService("SetDroitToAgent")
    }
    
    // suppression du cache
    AlkFactory::memCacheFlush();
  }

  /**
   * Enregistre les droits du profil sur les applications d'un espace
   * Le profil est contenu dans tabRight
   * @param cont_id       identifiant d'utilisateur
   * @param tabRight      non utilisé
   */
  public function setProfilRightsOnSpace($cont_id, $tabRight=array())
  {
    $tabRight = AlkRequest::_POST("listRight", array(), "is_array");
 
    $this->oQueryAction->setProfilRightsOnSpace($cont_id, $tabRight);

    $tabAppliRight = array();
    foreach($tabRight as $i => $strRight) {
      $tabTmp = explode("-", $strRight);
      $tabAppliRight["_".$tabTmp[0]] = array("_".$tabTmp[1] => $tabTmp[2]);
    }

    // applique les droits sur les applis de l'espace
    $dsAppli = $this->oQuery->getDs_listeAppliByEspace($cont_id);
    while( $drAppli = $dsAppli->getRowIter() ) {
      $appli_id = $drAppli->getValueName("APPLI_ID");
      $atype_id = $drAppli->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "setProfilRightsOnSpace") ) {
        if( array_key_exists("_".$appli_id, $tabAppliRight) ) {
          $oAppli->setProfilRightsOnSpace($cont_id, $tabAppliRight["_".$appli_id]);
          // callService("CopyDroitToProfil")
        }
      }
    }
    
    // suppression du cache
    AlkFactory::memCacheFlush();
  }

  /**
   * Invitation d'un  ou plusieurs utilisateurs à un espace (tous si cont_id=-1, tous les espaces publics si cont_id=-2)
   * @param user_id  identifiant ou liste d'id d'utilisateurs. -1 pour tous les utilisateurs ayant les privilèges priv_id
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces. -2 tous les espaces publics
   * @param priv_id  ensemble des privilèges
   */
  public function addUserToSpace($user_id, $cont_id, $priv_id)
  {
    $paramCont_id = $cont_id;
    $cont_id = ( $cont_id == -1  
                 ? $this->cont_id 
                 : ( $cont_id <= -2
                     ? -$cont_id
                     : $cont_id ));
    $typeAdd = -1;
    if( $user_id=="" && $paramCont_id>0 && $priv_id==ALK_PRIV_SPACE_ANIM ) {
      // doSql FormAnimForm : ajout animateur 
      $typeAdd = 0;
    } elseif( $user_id=="" && $paramCont_id>0 && $priv_id==ALK_PRIV_SPACE_USER ) {
      // doSql formUserList : ajout utilisateur
      $typeAdd = 1;
    } elseif( $user_id=="-1" && $paramCont_id>0 && $priv_id==ALK_PRIV_SPACE_ADMIN+ALK_PRIV_SPACE_VIEWER ) {
      // addSpace : ajout des admin tous espace et admin viewer
      $typeAdd = 2;
    } elseif( $user_id=="-1" && $paramCont_id<=-2 && $priv_id==ALK_PRIV_SPACE_USER ) {
      // addSpace public, ajout des utils appartenant aux services de type etat liés des départements associés 
      // updateSpace qui devient public 
      $typeAdd = 3;
    } elseif( is_numeric($user_id) && $user_id>0 && $paramCont_id==-1 
            && (($priv_id & ALK_PRIV_SPACE_VIEWER) == ALK_PRIV_SPACE_VIEWER ||
                ($priv_id & ALK_PRIV_SPACE_ADMIN)  == ALK_PRIV_SPACE_ADMIN) ) {
      // addUser avec privilège 
      $typeAdd = 4;
    } elseif( is_numeric($user_id) && $user_id>0 && $paramCont_id==-1 && $priv_id==ALK_PRIV_SPACE_USER ) {
      // addUser appartenant à un service de type etat
      $typeAdd = 5;
    }  elseif( is_numeric($user_id) && $user_id>0 && $paramCont_id>0 && 
               (($priv_id & ALK_PRIV_SPACE_USER)==ALK_PRIV_SPACE_USER || 
                ($priv_id & ALK_PRIV_SPACE_ANIM)==ALK_PRIV_SPACE_ANIM || 
                ($priv_id & ALK_PRIV_SPACE_ADMIN)==ALK_PRIV_SPACE_ADMIN )) {
      // addUser à un espace donné avec un privilège donné
      $typeAdd = 6;
    }
    
    $cont_reserve  = 0;
    $acont_reserve = 0;
    $tabUserIdAnimBefore = array();
    if( ALK_B_SPACE_SEND_MAIL == true && $typeAdd == 0 ) {
      // recherche la liste des agent_id animateur avant modif
      $cont_reserve  = AlkRequest::_POSTint("cont_reserve", $cont_reserve);
      $acont_reserve = AlkRequest::_POSTint("acont_reserve", $acont_reserve);

      if( $cont_reserve == $acont_reserve ) {
        $dsUser = $this->oQuery->getDs_listeAnimateurByEspaceForCombo($cont_id);
        while( $drUser = $dsUser->getRowIter() ) {
          $idUser = $drUser->getValueName("ID");
          $tabUserIdAnimBefore["_".$idUser] = true;
        } 
      }     
    }

    $tabListeId = array($user_id);
    if( $typeAdd=="0" || $typeAdd=="1" ) {
      // doSql 0 = FormAnimForm : ajout animateur 
      // doSql 1 = formUserList : ajout utilisateur
      $tabListeAgentId = AlkRequest::_POST("listUsersId", array(), "is_array");
      $tabListeId = array();
      $iCpt = 0;
      $tabListeId[$iCpt] = "";
      foreach($tabListeAgentId as $agent_id) {
        if( $agent_id != "" && $agent_id != "-1" )
          $tabListeId[$iCpt] .= ( $tabListeId[$iCpt] == "" ? "" : "," ).$agent_id;
        if( mb_strlen($tabListeId[$iCpt])>500 ) {
          $iCpt++;
          $tabListeId[$iCpt] = "";
        }
      }
    }

    if( $typeAdd == 0 ) {
      // validation des animateurs : collecte les animateurs généraux et l'utilisateur connecté
      $agent_id = AlkFactory::getSProperty("user_id", "-1");
      $tabListeId[$iCpt] .= ( $tabListeId[$iCpt] == "" ? "" : "," ).$agent_id;

      // $dsAgent prend en compte la notion privé/public et ALK_B_SPACE_ADMIN_ALL_DEPT
      $dsAgent = $this->oQuery->getDsListUserIdAdminAll($cont_id);
      while( $drAgent = $dsAgent->getRowIter() ) {
        $admin_id = $drAgent->getValueName("AGENT_ID");
        $tabListeId[$iCpt] .= ",".$admin_id;
      }
    } 
    elseif( $typeAdd==5 || $typeAdd==6 ) {
      // addUser appartenant à un service de type etat
      // addUser à un espace donné avec un privilège donné
      $iCpt = 0;
      $tabListeId[$iCpt] = $user_id;
      if( $typeAdd==5 ) $cont_id = -2;
    } 
    elseif( $typeAdd==3 ) {
      // addSpace public, ajout des utils appartenant aux services de type etat liés des départements associés 
      // updateSpace qui devient public 
      $dsAgent = $this->oQuery->getDsListUserIdServEtat($cont_id);
      $iCpt = 0;
      $tabListeId[$iCpt] = "";
      while( $drAgent = $dsAgent->getRowIter() ) {
        $admin_id = $drAgent->getValueName("AGENT_ID");
        $tabListeId[$iCpt] .= ( $tabListeId[$iCpt] == "" ? "" : "," ).$admin_id;
        if( mb_strlen($tabListeId[$iCpt])>500 ) {
          $iCpt++;
          $tabListeId[$iCpt] = "";
        }
      }
    } 
    elseif( $typeAdd==4 ) {
      // addUser avec privilège
      $iCpt = 0;
      $tabListeId[$iCpt] = $user_id;
      $cont_id = -1;
    }

    // liste des utilisateurs qui vont perdre l'animation ($typeAdd=0) 
    // et pour lesquels il faut recalculer les droits
    $tabUsers = array();
    foreach($tabListeId as $strListUserId ) {
      $tabUsers = array_merge($tabUsers, $this->oQueryAction->addUserToSpace($typeAdd, $strListUserId, $cont_id, $priv_id, $strListUserId));
    }

    // appel du service sur les applications de l'espace
    if( $cont_id>0 ) {
      // toutes les applis de l'espace
      $dsAppli = $this->oQuery->getDs_listeAppliByEspace($cont_id);
    } else {
      // tous les types d'appli pour 
      $dsAppli = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    }
    while( $drAppli = $dsAppli->getRowIter() ) {
      $appli_id = ( $cont_id>0 ? $drAppli->getValueName("APPLI_ID") : -1 );
      $atype_id = $drAppli->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addUserToSpace") ) {
        foreach($tabListeId as $strListUserId ) {
          if ($strListUserId!=""){
            $oAppli->addUserToSpace($strListUserId, $cont_id, 0);
          }
        }
      }
    }
    
    // appel du service sur l'annuaire
    $oAppliAnnu = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
    if( !is_null($oAppliAnnu) && is_object($oAppliAnnu) && method_exists($oAppliAnnu, "addUserToSpace") ) {
      foreach( $tabListeId as $strListUserId ) {
        if ($strListUserId!="") {
          $oAppliAnnu->addUserToSpace($strListUserId, $cont_id, 0);
        }
      }
    }
    
    if( !empty($tabUsers) ) {
      // recalcul les droits des utilisateurs qui ont perdu l'animation
      $this->setProfilRightToUser(implode(",", $tabUsers));
    }
    
    if( ALK_B_SPACE_SEND_MAIL == true && $typeAdd == 0 ) {
      $cont_intitule = $this->getAppliProperty("CONT_INTITULE");
      $cont_public   = $this->getAppliProperty("CONT_PUBLIC");

      if( $cont_reserve == $acont_reserve ) {
        // traitement ajout-suppression priv animateur
        $this->sendMailToAdmin($cont_id, $cont_intitule, $cont_public, 3, $tabUserIdAnimBefore);
      } else { // modification de la case accès réservé
        if( $cont_reserve==1 ) {
          // avertir les animateurs que les admins principaux n'auront pas la main sur cet espace
          $this->sendMailToAdmin($cont_id, $cont_intitule, $cont_public, 4);
        } else {
          // avertir que les admin principaux ont accès sur l'espace
          $this->sendMailToAdmin($cont_id, $cont_intitule, $cont_public, 5);
        }
      }
    }
  }

  /**
   * Applique les droits d'un utilisateur par rapport à ses profils sur l'ensemble de l'Extranet
   * @param user_id  identifiant de l'utilisateur 
   */
  public function setProfilRightToUser($user_id)
  {
    if( is_array($user_id) ) {
      foreach($user_id as $strListUserId) {
        $this->oQueryAction->setProfilRightToUser($strListUserId);
      }
    } else {
      $this->oQueryAction->setProfilRightToUser($user_id);
    }
    
    $dsAppli = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppli = $dsAppli->getRowIter() ) {
      $atype_id = $drAppli->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "setProfilRightToUser") ) {
        if( is_array($user_id) ) {
          foreach($user_id as $strListUserId) {
            $oAppli->setProfilRightToUser($strListUserId);
          }
        } else {
          $oAppli->setProfilRightToUser($user_id);
        }
      }
    }
    
    // suppression du cache
    AlkFactory::memCacheFlush();
  }
  
  /**
   * Modification du privilège espace pour l'utilisateur à un espace (tous si cont_id=-1)
   * @param user_id  identifiant d'un utilisateur
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces.
   * @param priv_id  identifiant du privilège
   */
  public function updateUserPrivToSpace($user_id, $cont_id, $priv_id)
  {
  }

  /**
   * Suppression de l'accès à un espace (ou tous si cont_id=-1) pour un utilisateur
   * @param user_id  identifiant d'un utilisateur, ="" pour récupérer ceux cochés
   * @param cont_id  identifiant de l'espace. -1 pour tous les espaces.
   */
  public function removeUserFromSpace($user_id, $cont_id)
  {
    $tabListeId = array($user_id);
    if( $user_id == "" ) {
      // reception de espace/formuserlist
      $tabListeAgentId = AlkRequest::_POST("listUserId", array(), "is_array");
      $tabListeId = array();  

      $iCpt = 0;
      $tabListeId[$iCpt] = "";
      foreach($tabListeAgentId as $agent_id) {
        if( $agent_id != "" && $agent_id != "-1" )
          $tabListeId[$iCpt] .= ( $tabListeId[$iCpt] == "" ? "" : "," ).$agent_id;
        if( mb_strlen($tabListeId[$iCpt])>500 ) {
          $iCpt++;
          $tabListeId[$iCpt] = "";
        }
      }
    }

    // liste des applis de l'espace
    $dsAppli = $this->oQuery->getDs_listeAppliByEspace($cont_id);
    while( $drAppli = $dsAppli->getRowIter() ) {
      $appli_id = $drAppli->getValueName("APPLI_ID");
      $atype_id = $drAppli->getValueName("ATYPE_ID");
      for($i=0; $i<count($tabListeId); $i++) {
        /* retire les droits utilisateurs sur l'application */
        $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
        if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "removeUserFromSpace") )
          $oAppli->removeUserFromSpace($tabListeId[$i], $cont_id);
        // callService("DelDroitToAgent")
      }
    }

    // appli annuaire
    $oAppliAnnu = AlkFactory::getAppli(ALK_ATYPE_ID_ANNU);
    if( !is_null($oAppliAnnu) && is_object($oAppliAnnu) && method_exists($oAppliAnnu, "removeUserFromSpace") ) {
      for($i=0; $i<count($tabListeId); $i++) {
        $oAppliAnnu->removeUserFromSpace($tabListeId[$i], $cont_id);
      }
    }

    for($i=0; $i<count($tabListeId); $i++) {
      $this->oQueryAction->removeUserFromSpace($cont_id, $tabListeId[$i]);
    }
  }

  /**
   * Méthode pour propager la création d'un service aux applis ayant des données associées
   * @param service_id   identifiant du service créé
   * @param tabFields    tableau associatif contenant les infos modifiées du service
   */
  public function addService($service_id, $tabFields)
  {
    // pas de traitement au niveau d'un espace
    //$this->oQueryAction->addService($service_id, $tabFields);    

    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addService") )
        $oAppli->addService($service_id, $tabFields);
    }
    
    // traitement sur appli spécifiques
    $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
    foreach($tabAtypeID as $atype_id) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "addService") )
        $oAppli->addService($service_id, $tabFields);
    }
  }
  
  /**
   * Méthode pour propager la mise à jour d'un service aux applis ayant des données associées
   * @param service_id  identifiant du service à modifier
   * @param tabFields   tableau associatif contenant les infos modifiées du service
   */
  public function updateService($service_id, $tabFields)
  {
    // pas de traitement au niveau d'un espace
    //$this->oQueryAction->updateService($service_id, $tabFields);    

    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "updateService") )
        $oAppli->updateService($service_id, $tabFields);
    }
    
    // traitement sur appli spécifiques
    $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
    foreach($tabAtypeID as $atype_id) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "updateService") )
        $oAppli->updateService($service_id, $tabFields);
    }
  }

  /** 
   * Méthode pour propager la suppression d'un service aux applis ayant des données associées
   * @param service_id  identifiant du service à supprimer
   */
  public function delService($service_id)
  {
    // pour ts les types d'applications
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");

      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delService") )
        $oAppli->delService($service_id);
    }
    
    // traitement sur appli spécifiques
    $tabAtypeID = unserialize(ALK_LIST_ATYPE_ID_SPECIAL);
    foreach($tabAtypeID as $atype_id) {
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "delService") )
        $oAppli->delService($service_id);
    }
    
    // pas de traitement au niveau d'un espace
    //$this->oQueryAction->delService($service_id);    
  }

  /**
   * Charge à partir de la base de données, les propriétés de l'espace
   */
  protected function setAppliProperty()
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $dsCont = $this->oQuery->GetDs_ficheEspaceById($this->cont_id, "", false, $user_id);
    if( $oDr = $dsCont->getRowIter() )
      $this->tabProperty = $oDr->getDataRow();
  }

  /**
   * Initialise les tabsheets
   */
  public function setTabSheets()
  {
    $strUrl = ALK_ALKANET;
    $strParam = "";
    $iTypeSheet = AlkRequest::getToken("iTypeSheet");
    $user_id     = AlkFactory::getSProperty("user_id", -1);

    if( $iTypeSheet == ALK_TYPESHEET_POPUP ) 
      return;
    
    // onglets de consultation
    $bVisible = ( defined("ALK_ATYPE_ID_SEARCH") && defined("ALK_B_ATYPE_SEARCH") && ALK_B_ATYPE_SEARCH == true );
    if ( !(defined("ALK_B_SPACE_SHEETSEARCH") && ALK_B_SPACE_SHEETSEARCH==false) ) {
      $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_RECHERCHE, ALK_SHEET_NONE, _t("Effectuer une recherche"), $strUrl, "", "", 
                      ALK_APPLI_RIGHT_READ, "-1", $bVisible);
    }
    
    // onglet tableau de bord // si le template est présent on l'affiche 
    if( !(defined("ALK_B_SPACE_SHEETTABLEBORD") && ALK_B_SPACE_SHEETTABLEBORD==false) ) { 
      $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_TABLE, ALK_SHEET_NONE, _t("Tableau de bord"), $strUrl);
    }
    
    if( !(defined("ALK_B_SPACE_SHEETPLAN") && ALK_B_SPACE_SHEETPLAN==false) ) {
      $this->addSheet(ALK_TYPESHEET_CONSULT, ALK_SHEET_ARBORESCENCE, ALK_SHEET_NONE, _t("Plan de l'espace"), $strUrl);
    }
    
    // onglets d'administration
    if( !(defined("ALK_B_SPACE_SHEETPROP") && ALK_B_SPACE_SHEETPROP==false) ) {
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_FICHE, ALK_SHEET_NONE, _t("Propriétés"), $strUrl, $strParam, 
                      _t("Modifier les propriétés de l'espace"),
                      ALK_APPLI_RIGHT_READ, (ALK_B_SPACE_ANIM_VIEW_STAT_ONLY == true ? ALK_PRIV_SPACE_ADMIN : ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN));
    }
    
    if( ALK_B_SPACE_ANIM_ADD_ANIM && !(defined("ALK_B_SPACE_SHEETANIM") && ALK_B_SPACE_SHEETANIM==false) ) {
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_ANIMATEUR, ALK_SHEET_NONE, _t("Animateurs"), $strUrl, $strParam, 
                      _t("Gérer les animateurs de cet espace"), 
                      ALK_APPLI_RIGHT_READ, (ALK_B_SPACE_ANIM_VIEW_STAT_ONLY == true ? ALK_PRIV_SPACE_ADMIN : ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN));
    }

    if( !(defined("ALK_B_SPACE_SHEETPROFIL") && ALK_B_SPACE_SHEETPROFIL==false) ) {
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_PROFIL, ALK_SHEET_LIST, _t("Profils spécifiques"), $strUrl, $strParam, 
                      _t("Gérer les profils de cet espace"), 
                      ALK_APPLI_RIGHT_READ, (ALK_B_SPACE_ANIM_VIEW_STAT_ONLY == true ? ALK_PRIV_SPACE_ADMIN : ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN));
    }
    
    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_APPLICATION, ALK_SHEET_LIST, _t("Applications"), $strUrl, $strParam, 
                    _t("Gérer les applications de cet espace"), 
                    ALK_APPLI_RIGHT_READ, (ALK_B_SPACE_ANIM_VIEW_STAT_ONLY == true ? ALK_PRIV_SPACE_ADMIN : ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN));

    $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_UTILISATEUR, ALK_SHEET_NONE, _t("Utilisateurs"), $strUrl, $strParam, 
                    _t("Gérer les utilisateurs de cet espace"), 
                    ALK_APPLI_RIGHT_READ, (ALK_B_SPACE_ANIM_VIEW_STAT_ONLY == true ? ALK_PRIV_SPACE_ADMIN : ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN));
    
    // verification de la présence de l'appli lsdiff
    if ( defined("ALK_B_ATYPE_LSDIF") && ALK_B_ATYPE_LSDIF==true ) {
      $dsAppli = $this->oQuery->getDs_listeAppliByEspace($this->cont_id, 0, -1, false, true, ALK_ATYPE_ID_LSDIF);
      $appli_id_lisdif = -1;
      while ( $drAppli = $dsAppli->getRowIter() ) {
        $atype_id = $drAppli->getValueName("ATYPE_ID");
        if($atype_id == ALK_ATYPE_ID_LSDIF){
          $appli_id_lisdif = $drAppli->getValueName("APPLI_ID");
          continue;
        }
      }
      if ( $appli_id_lisdif != -1 ) { 
        $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_USER_LIST, ALK_SHEET_NONE, _t("Invitations"), $strUrl, $strParam."&appli_id_lisdif=".$appli_id_lisdif, 
                    _t("Inviter des utilisateurs à cet espace ou groupe de travail"), 
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
      }
    }       
              
    if ( !(defined("ALK_B_SPACE_SHEETSTAT") && ALK_B_SPACE_SHEETSTAT==false) ) {
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_STATS, ALK_SHEET_NONE, _t("Statistiques"), $strUrl, $strParam,
                      _t("Consulter les statistiques de cet espace"), 
                      ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
    }
    
    if ( (defined("ALK_B_SPACE_SHEET_LINKS") && ALK_B_SPACE_SHEET_LINKS==true) ) {
      $this->addSheet(ALK_TYPESHEET_ADMIN, ALK_SHEET_LINK, ALK_SHEET_NONE, _t("Vérification de liens"), $strUrl, $strParam,
                      _t("Vérifier les liens des différentes applications"), 
                      ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
    }
    
    // onglets de propriétés
    $this->addSheet(ALK_TYPESHEET_PROPRIETE, ALK_SHEET_PLANIF, ALK_SHEET_LIST, _t("Planification"), $strUrl, $strParam,
                    _t("Planifier les tâches des différentes applications"), 
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
    
    // onglets gestion categorie
    $this->addSheet(ALK_TYPESHEET_PROPRIETE, ALK_SHEET_RUBRIQUE, ALK_SHEET_LIST, _t("Catégories"), $strUrl, $strParam,
                      _t("Gérer les catégories des espaces"), 
                      ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
    
    // restriction temporaire au user alkante  
    if ($user_id  == ALK_USER_ID_ADMINALK)           
      $this->addSheet(ALK_TYPESHEET_PROPRIETE, ALK_SHEET_PARAMETRE, ALK_SHEET_LIST, _t("Paramètres"), $strUrl, $strParam,
                    _t("Gérer les constantes des différentes applications"), 
                    ALK_APPLI_RIGHT_READ, ALK_PRIV_SPACE_ANIM+ALK_PRIV_SPACE_ADMIN);
  }

  /**
   * Met à jour les statistiques de consultation
   * @param cont_id   identifiant de l'espace
   * @param appli_id  identifiant de l'application
   */
  public function updateStat($cont_id, $appli_id)
  {
    $bStatConnectSIT   = ( isset($_SESSION["ALK_bStatConnectSIT"])   ? $_SESSION["ALK_bStatConnectSIT"]   : false   );
    $tabStatAccesCont  = ( isset($_SESSION["ALK_tabStatAccesCont"])  ? $_SESSION["ALK_tabStatAccesCont"]  : array() );
    $tabStatAccesAppli = ( isset($_SESSION["ALK_tabStatAccesAppli"]) ? $_SESSION["ALK_tabStatAccesAppli"] : array() );

    $user_id = AlkFactory::getSProperty("user_id", "-1");
    //if( $user_id == ALK_USER_ID_INTERNET /*|| $user_id == ALK_USER_ID_ADMINALK*/ ) {
      // on ne mémorise pas les accès du visiteur internet et celle de l'admin akante
    //  return;
    //}

    if( !array_key_exists("_".$cont_id, $tabStatAccesCont) ) {
      // stat sur espace
      $tabStatAccesCont["_".$cont_id] = true;
      $_SESSION["ALK_tabStatAccesCont"] = $tabStatAccesCont;

      $dsTest = $this->oQuery->GetDs_listeStatsByAgent($user_id, 1, $cont_id);
      if( $dsTest->getCountTotDr() > 0 ) {
        // maj
        $this->oQueryAction->maj_statsByAgent($user_id, 1, $cont_id);
      } else { 
        // ajout
        $this->oQueryAction->add_statsByAgent($user_id, 1, $cont_id);
      }
    }
        
    if( !array_key_exists("_".$appli_id, $tabStatAccesAppli) ) {
      // stat sur appli
      $tabStatAccesAppli["_".$appli_id] = true;
      $_SESSION["ALK_tabStatAccesAppli"] = $tabStatAccesAppli;

      $dsTest = $this->oQuery->GetDs_listeStatsByAgent($user_id, 2, $appli_id);
      if( $dsTest->getCountTotDr() > 0 ) {
        // maj
        $this->oQueryAction->maj_statsByAgent($user_id, 2, $appli_id);
      } else { 
        // ajout
        $this->oQueryAction->add_statsByAgent($user_id, 2, $appli_id);
      }
    }
      
    if( $bStatConnectSIT == false ) {
      // stat sur accès global à Alkanet
      $_SESSION["ALK_bStatConnectSIT"] = true;
      $dsTest = $this->oQuery->GetDs_listeStatsByAgent($user_id, 0, 0);
      if( $dsTest->getCountTotDr() > 0 ) {
        // maj
        $this->oQueryAction->maj_statsByAgent($user_id, 0, 0);
      } else { 
        // ajout
        $this->oQueryAction->add_statsByAgent($user_id, 0, 0);
      }
    }
  }

  /**
   * met à jour la statistique liée à une donnée data_id de l'application appli_id identifiée par action_id
   * @param atype_id    type de l'application
   * @param appli_id    identifiant de l'application
   * @param datatype_id type de l'information
   * @param data_id     identifiant de l'information
   * @param agent_id    identifiant de l'agent
   * @param action_id   identifiant de l'action déclenchée
   */
  public function updateStatAppli($atype_id, $appli_id, $datatype_id, $data_id, $agent_id, $action_id)
  {
    $this->oQueryAction->updateStatAppli($atype_id, $appli_id, $datatype_id, $data_id, $agent_id, $action_id);
  }

  /**
   * met à jour le log lié à l'application appli_id identifiée par action_id
   * A ma différence d'une statAppli, le log est daté.
   * @param atype_id    type de l'application
   * @param appli_id    identifiant de l'application
   * @param agent_id    identifiant de l'agent
   * @param action_id   identifiant de l'action déclenchée
   */
  public function updateLogAppli($atype_id, $appli_id, $agent_id, $action_id)
  {
    $this->oQueryAction->updateStatAppli($atype_id, $appli_id, "null", "null", $agent_id, $action_id);
  }

  /**
   * lance l'import d'un fichier xml et zip présent dans le répertoire ALK_ROOT_PATH.ALK_UPLOAD_FTP_PATH
   * Ce fichier xml contient les données à importer en base. le fichier zip contient les fichiers référencés par le xml.
   * Agent_id est l'identifiant de l'utilisateur qui devient propriétaire de l'information.
   * Retourne la trace d'exécution de l'import
   * @param strXMLFileName  nom du fichier
   * @param agent_id        identifiant de la agent qui devient propriétaire des infos ajoutées
   * @return string
   */
  public function import($agent_id, $strXMLFileName)
  {
    $strLog = "";
    $strPathXsdFileName = ALK_ALKANET_ROOT_PATH.ALK_ROOT_MODULE."espace/schema_import_alkanet.xsd";
    
    $strPathXmlFileName = ALK_ROOT_PATH.ALK_UPLOAD_FTP_PATH.$strXMLFileName;
    $strPathZipFileName = ALK_ROOT_PATH.ALK_UPLOAD_FTP_PATH.str_replace(".xml", ".zip", $strXMLFileName);
    $bZipFile = true;
    
    if( !(file_exists($strPathXmlFileName) && is_file($strPathXmlFileName)) ) {
      $strLog = _f("Erreur : Le fichier source %s n'existe pas.", $strXMLFileName)."\r\n";
      return $strLog; 
    }
    if( !(file_exists($strPathZipFileName) && is_file($strPathZipFileName)) ) {
      $strLog .= _t("Le fichier zip n'a pas été transmis.")."\r\n";
      $bZipFile = false;
    }
    
    // déplacement des fichiers vers le répertoire cache
    if( !@rename($strPathXmlFileName, ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE.$strXMLFileName) ) {
      $strLog .= _f("Erreur : Impossible de déplacer le fichier source %s vers le répertoire de traitement.", $strXMLFileName)."\r\n";
      return $strLog; 
    } else {
      $strLog .= _f("Le fichier xml a été déplacé vers %s.", ALK_ROOT_DIR.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE)."\r\n"; 
    }
    if( $bZipFile ) {
      if( !@rename($strPathZipFileName, ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE.str_replace(".xml", ".zip", $strXMLFileName)) ) {
      $strLog .= _t("Erreur : Impossible de déplacer le fichier zip vers le répertoire de traitement.")."\r\n";
      return $strLog; 
      } else {
        $strLog .= _f("Le fichier zip a été déplacé vers %s.", ALK_ROOT_DIR.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE)."\r\n";
      }
    }

    // nouveaux emplacements des fichiers
    $strPathXmlFileName = ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE.$strXMLFileName;
    $strPathZipFileName = ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE.str_replace(".xml", ".zip", $strXMLFileName);
    $strPathZipExtract  = ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE.str_replace(".xml", "", $strXMLFileName)."/";

    if( $bZipFile ) {
      // décompression du fichier zip dans le répertoire cache
      $strCmd = "unzip -d ".ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE." ".$strPathZipFileName;
      $tabOutput = array();
      exec($strCmd, $tabOutput);
      
      $strLog .= _t("Décompression du fichier zip")."\r\n".implode("\r\n", $tabOutput)."\r\n";
    }
    
    // chargement du fichier xml
    $oXmlImport = new  DOMDocument();
    $oXmlImport->load($strPathXmlFileName);
    /*if( !$oXmlImport->schemaValidate($strPathXsdFileName) ) {
      $strLog .= "Erreur : Le document XML ne respecte pas sa DTD.\r\n";
      return $strLog;
    }*/
    
    // parcours du xml
    // niv 1 : appli
    // niv 2 : 
    $tabPk = array();
    $tabMemAppli = array();
    $tabProfil = array();
    $_POST["listRight"] = array();
    
    // affecte au post, un tableau pour mettre à jour les droits des profils de l'espace pour les applications ajoutées
    // affecte le droit de lecture au profil utilisateur, aucun à tous les autres
    $dsProfil = $this->oQuery->getDsProfilListByAppli("-1", $this->cont_id);
    while( $drProfil = $dsProfil->getRowIter() ) {
      $profil_id = $drProfil->getValueName("PROFIL_ID");
      $droit_id  = ( $profil_id == "1" ? "1" : "0" );
      $tabProfil[] = $profil_id."-".$droit_id; 
    }
    
    $this->tabFieldImport = array();
    
    $oRoot = $oXmlImport->documentElement;
    
    $oDNL = $oRoot->getElementsByTagName("sit_appli");
    foreach($oDNL as $oNodeAppli) {
      
      // lecture des infos de l'appli
      $tabAppliField = array("atype_id"       => "", 
                             "appli_intitule" => "", 
                             "appli_id"       => "");
      $tabCat = array("fdoc_01_rubrique"      => array(),
                      "iedit_datatype_appli"  => array(), 
                      "iedit_classification"  => array(), 
                      "iedit_data"            => array());
      
      foreach($oNodeAppli->childNodes as $oNodeInfoAppli) {
        if( $oNodeInfoAppli->nodeType == XML_ELEMENT_NODE ) {
          if( isset($tabAppliField[$oNodeInfoAppli->nodeName]) ) {
            $tabAppliField[$oNodeInfoAppli->nodeName] = $oNodeInfoAppli->nodeValue;
            //echo "info ". $oNodeInfoAppli->nodeName." = ".$oNodeInfoAppli->nodeValue."<br>";
          } elseif( isset($tabCat[$oNodeInfoAppli->nodeName]) ) {
            //echo "appel avec ".$oNodeInfoAppli->nodeName."<br>";
            $this->getTabFieldsByXmlImport($oNodeInfoAppli->nodeName, $oNodeInfoAppli);
          }
        } 
      }
      
      $tabField = array();
      $i_appli_id                 = $tabAppliField["appli_id"];
      $tabField["ATYPE_ID"]       = array(ALK_SQL_NUMBER, $tabAppliField["atype_id"]);
      $tabField["APPLI_INTITULE"] = array(ALK_SQL_TEXT, $tabAppliField["appli_intitule"]);
      $tabField["APPLI_RANG"]     = array(ALK_SQL_NUMBER, 0);
      $tabField["APPLI_DEFAUT"]   = array(ALK_SQL_NUMBER, 0);
      
      if( $tabAppliField["atype_id"] == ALK_ATYPE_ID_FDOC ) {
        $tabField["NOT_CALL_APPLI_ADD"] = array(ALK_SQL_NONE, 1);
      }
      
      $tabFields = array("pk" => array(), "field" => $tabField);
      
      $appli_id = $this->addAppli($tabFields);
      $tabPk["appli_id"]["_".$i_appli_id] = $appli_id;
      
      $tabMemAppli[] = array("atype_id" => $tabAppliField["atype_id"], "appli_id" => $appli_id); 
    }
  
    $strLog .= _t("Exécution de l'import :")."\r\n";
    $strLog .= $this->oQueryAction->import($this->tabFieldImport, $tabPk, $agent_id, $this->cont_id, $strPathZipExtract);
    
    // réapplique les droits sur les profils
    $_POST["listRight"] = $tabProfil;
    foreach($tabMemAppli as $tabMem ) {
      $this->setRightsOnAppli($tabMem["atype_id"], $tabMem["appli_id"]); 
    }
    
    // nettoyage
    $this->tabFieldImport = array();
    
    if( $bZipFile ) {
      $strCmd = "cd ".$strPathZipExtract." && rm *";
      $tabOutput = array();
      exec($strCmd, $tabOutput);
      $strLog .= _t("Suppression des fichiers joints décompressés.")."\r\n";

      $strCmd = "cd ".ALK_ALKANET_ROOT_PATH.ALK_ROOT_UPLOAD.ALK_UPLOAD_CACHE." && rmdir ".str_replace(".xml", "", $strXMLFileName);
      $tabOutput = array();
      exec($strCmd, $tabOutput);  
      $strLog .= _t("Suppression du répertoire temporaire contenant les fichiers joints.")."\r\n";
    }
    
    @unlink($strPathXmlFileName);
    @unlink($strPathZipFileName);
    $strLog .= _t("Suppression des fichiers xml et zip.")."\r\n";
    $strLog .= _t("Import terminé.")."\r\n";
    
    return $strLog;
  }

  /**
   * lecture récursive d'un noeud xml d'import pour transformer le sous arbre xml 
   * en tableau de données utilisable par la méthode AlkQuery::getPartInsertSql()
   * Met à jour l'attribut $this->tabFieldImport
   * @param tableName  nom de la table
   * @param oNode      noeud du sous arbre xml
   */
  private function getTabFieldsByXmlImport($tableName, $oNode)
  {
    $nbEnr = ( isset($this->tabFieldImport[$tableName]) 
               ? count($this->tabFieldImport[$tableName])
               : 0 );

    if( !isset($this->tabFieldImport[$tableName]["pk"]) ) {
      if( $oNode->hasAttribute("pk") ) {
        $this->tabFieldImport[$tableName]["pk"] = $oNode->getAttribute("pk");
        //echo "clé primaire de ".$tableName." = ".$this->tabFieldImport[$tableName]["pk"]."<br>"; 
      }
    } 

    foreach($oNode->childNodes as $oSubNode) {
      if( $oSubNode->nodeType == XML_ELEMENT_NODE ) {
        if( $oSubNode->hasChildNodes() ) {
          if( $oSubNode->childNodes->length>1 ) {
            //echo "appel avec ".$oSubNode->nodeName."<br>";
            $this->getTabFieldsByXmlImport($oSubNode->nodeName, $oSubNode);
          } else {
            $iType = ( $oSubNode->hasAttribute("type") 
                       ? $oSubNode->getAttribute("type")
                       : 0 );
            //echo "enr n°".$nbEnr." avec ".$tableName.".".$oSubNode->nodeName." = ".$oSubNode->nodeValue." type=".$iType."<br>";
            
            $this->tabFieldImport[$tableName][$nbEnr][$oSubNode->nodeName] = array($iType, $oSubNode->nodeValue);
          }
        }
      }
    }
  }

  /**
   * Retourne le tableau des élts de menu : liste des applications
   *
   * @param tabAppliMenu Tableau contenant les elts de menu pour l'application en cours (non utilisé)
   * @return Retourne un tableau contenant le couple (tableau Appli, tableau sous appli)
   */
  function getTabMenuAppli($tabAppliMenu=array())
  {
    if( is_array($this->tabMenuAppli) && count($this->tabMenuAppli)>0 )
      return array("appli" => $this->tabMenuAppli);

    $strDefaultTarget = "";
    $this->tabMenuAppli = array();
    $this->tabSsMenuAppli = array();

    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $cont_id = $this->cont_id;
    $appli_id = AlkRequest::getToken("appli_id");

    // cas de l'accès réservé
    $strCondProfil = ( ALK_B_SPACE_REMOVE_PRIV_ADMINALL && AlkFactory::isSpaceAnimAll()
                       ? " and c.CONT_RESERVE=0 "
                       : "" );
   
    if( !is_array($tabAppliMenu) ) $tabAppliMenu = array();

    $strParam = "cont_id=".$this->cont_id."&appli_id=";

    $dsAppli = $this->oQuery->getDs_listeAppliAgentByEspace($user_id, $cont_id, $strCondProfil);
    $cpt = 0;
    while( $drAppli = $dsAppli->GetRowIter() ) {
      $cpt++;
      $_appli_id         = $drAppli->getValueName("APPLI_ID");
      $_atype_id         = $drAppli->getValueName("ATYPE_ID");
      $_strAppliTypeIntitule   = $drAppli->getValueName("ATYPE_INTITULE");
      $_strAppliLogo     = $drAppli->getValueName("APPLI_LOGO");
      $_strAppliTypeLogo = $drAppli->getValueName("ATYPE_LOGO");
      $_strUrlAppli      = $drAppli->getValueName("ATYPE_URL");
      $_strTargetAppli   = $drAppli->getValueName("ATYPE_URL_TARGET");
      $_strAppliIntitule = $drAppli->getValueName("APPLI_INTITULE");
      $_atype_compatible = $drAppli->getValueName("ATYPE_COMPATIBLE");
      $_appli_id_liee    = $drAppli->getValueName("APPLI_LIEE");
      $_appli_right_id   = $drAppli->getValueName("RIGHT_ID");

      $bUseParamDeptInTitle = ( $_atype_id == ALK_ATYPE_ID_ASPIC || $_atype_id == ALK_ATYPE_ID_ASPICC || 
                                $_atype_id == ALK_ATYPE_ID_RAA );
      $strNumDept = "";
      if( $bUseParamDeptInTitle ) {
        $strNumDept = trim(mb_substr($_strAppliIntitule, -3));
        $strNumDept = ( is_numeric($strNumDept) || 
                        mb_strtolower(trim($strNumDept))=="2a" || 
                        mb_strtolower(trim($strNumDept))=="2b" 
                        ? $strNumDept 
                        : "0" ); 
      }  

      $this->tabMenuAppli[$cpt] = 
        array("url"  => ( $this->_testUrlJs($_strUrlAppli)
                          ? $this->_getUrlJS($_strUrlAppli, ( $_atype_compatible == "1" 
                                                              ? "token=".AlkRequest::getEncodeParam($strParam.$_appli_id) 
                                                              : ( $bUseParamDeptInTitle
                                                                  ? $strNumDept."','"
                                                                  : "").$strParam.$_appli_id), $_strTargetAppli)
                          : ( $_strUrlAppli=="" 
                              ? ALK_ALKANET."?token=".AlkRequest::getEncodeParam($strParam.$_appli_id)
                              : (strpos($_strUrlAppli, "?") ===false ? $_strUrlAppli."?".$strParam.$_appli_id : $_strUrlAppli."&".$strParam.$_appli_id ))),
              "target"     => ( $_strTargetAppli=="" ? $strDefaultTarget : $_strTargetAppli ),
              "titreCourt" => $this->_tronqueItemMenu($_strAppliIntitule),
              "titre"      => $_strAppliIntitule,
              "logo"       => ( $_strAppliLogo!="" ? $_strAppliLogo : $_strAppliTypeLogo ),
              "id"         => $_appli_id,
              "id_appli_liee" => $_appli_id_liee,
              "atypeid"    => $_atype_id,
              "atypeintitule"  => $_strAppliTypeIntitule,
              "rightid"    => $_appli_right_id,
              "nbFils"     => 0);
      $strLogo = $this->tabMenuAppli[$cpt]["logo"];
      $this->tabMenuAppli[$cpt]["logo"] = ALK_MEDIA_URL."pictos/".$strLogo;
      $this->tabMenuAppli[$cpt]["csslogo"] = str_replace("_", "", substr($strLogo, 0, -4));
    }
    return array("appli" => $this->tabMenuAppli);
  }

  /**
   * Retourne le tableau des élts de type espace pour le menu : 
   *        - liste des espaces fils si iType=1 (par défaut)
   *        - liste des espaces frere si iType=2
   *        - liste des espaces fils et frere si iType=3
   * @param iType  Type de 
   * @return Retourne un tableau contenant le couple (tableau espace fils, tableau espace petit fils)
   */
  public function getTabMenuSpace($iType=1)
  {
    if( ($iType & 1) == 1 && is_array($this->tabMenuSpaceChild) && count($this->tabMenuSpaceChild)>0 ||
        ($iType & 2) == 2 && is_array($this->tabMenuSpaceBrother) && count($this->tabMenuSpaceBrother)>0 ||
        ($iType & 4) == 4 && is_array($this->tabMenuSpaceOther) && count($this->tabMenuSpaceOther)>0 ) {
      return array("child" => ( ($iType & 1) == 1 && is_array($this->tabMenuSpaceChild) && count($this->tabMenuSpaceChild)>0
                                ? $this->tabMenuSpaceChild 
                                : array() ),
                   "brother" => ( ($iType & 2) == 2 && is_array($this->tabMenuSpaceBrother) && count($this->tabMenuSpaceBrother)>0
                                  ? $this->tabMenuSpaceBrother
                                  : array()),
                   "Other"   => ( ($iType & 2) == 2 && is_array($this->tabMenuSpaceOther) && count($this->tabMenuSpaceOther)>0
                                  ? $this->tabMenuSpaceOther
                                  : array()));
    }
    
    $user_id = AlkFactory::getSProperty("user_id", "-1");
    $this->tabMenuSpaceChild = array();
    $this->tabMenuSpaceBrother = array();
    $this->tabMenuSpaceOther = array();

    // cas de l'accès réservé
    $strCondProfil = "";
    if( ALK_B_SPACE_REMOVE_PRIV_ADMINALL==true && AlkFactory::isSpaceAnimAll() ) {
      $strCondProfil = " and c.CONT_RESERVE=0 ";
    }
    
    if( ($iType & 1) == 1 ) {
      // selection des espaces fils accessibles avec détermination de l'appli par defaut
      $dsCont = $this->oQuery->getDs_listeSousEspaceByAgent($user_id, $this->cont_id, $strCondProfil);
      $this->tabMenuSpaceChild = $this->_getTabMenuSpace($dsCont);
    }

    if( ($iType & 2) == 2 ) {
      // selection des espaces freres accessibles avec détermination de l'appli par defaut
      $cont_id_pere = $this->getAppliProperty("CONT_PERE");
      $dsCont = $this->oQuery->getDs_listeSousEspaceByAgent($user_id, $cont_id_pere, $strCondProfil);
      $this->tabMenuSpaceBrother = $this->_getTabMenuSpace($dsCont);
    }
    
    if( ($iType & 4) == 4 ) {
      // sélection des espaces non directement accessibles
      $dsCont = $this->oQuery->getDsListeSousEspaceOrphelinsByAgent($user_id, $this->cont_id, $strCondProfil);
      $this->tabMenuSpaceOther = $this->_getTabMenuSpace($dsCont);
    }
    
    // on retire les espaces autres qui peuvent être dans les frêres
    if( ($iType & 4) == 4 && ($iType & 2) == 2 ) {
      foreach($this->tabMenuSpaceBrother as $tabSpaceBrother) {
        for($i=0; $i<count($this->tabMenuSpaceOther); $i++) {
          if( isset($this->tabMenuSpaceBrother[$i]) && $this->tabMenuSpaceOther[$i]["id"] == $tabSpaceBrother["id"] ) {
            unset($this->tabMenuSpaceBrother[$i]);
            break;
          } 
        }
      }
    }
      
    return array("child" => $this->tabMenuSpaceChild, 
                 "brother" => $this->tabMenuSpaceBrother,
                 "other"   => $this->tabMenuSpaceOther);
  }

  /**
   * Retourne un tableau contenant les éléments du menu concernant les espace
   *        à partir d'un dataSet
   * @param dsCont  dataSet contenant les éléments du menu
   * @return array
   */
  private function _getTabMenuSpace($dsCont)
  {
    $strDefaultTarget = "";
    $tabMenuSpace = array();

    $cpt=0;
    while( $drCont = $dsCont->GetRowIter() ) {
      $cpt++;
      $_cont_id              = $drCont->getValueName("CONT_ID");
      $_iAppliDefaut         = $drCont->getValueName("DEFAUT");
      $_appli_id             = $drCont->getValueName("APPLI_ID");
      $_strUrlAppli          = $drCont->getValueName("ATYPE_URL");
      $_strTargetAppli       = $drCont->getValueName("ATYPE_URL_TARGET");
      $_strContIntitule      = $drCont->getValueName("CONT_INTITULE");
      $_strContIntituleCourt = $drCont->getValueName("CONT_INTITULE_COURT");
      $_iContPublic          = $drCont->getValueName("CONT_PUBLIC");
      $_appli_id = ( $_appli_id == "0" ? "-1" : $_appli_id );
        
      if( $_iAppliDefaut == 1 ) {
        // l'appli par défaut est spécifiée
        $strToken = AlkRequest::getEncodeParam("cont_id=".$_cont_id."&appli_id=".$_appli_id);
        if( $this->_testUrlJs($_strUrlAppli) ) {
          // url javascript
          $strUrl = $this->_getUrlJS($_strUrlAppli, "token=".$strToken, "'".$_strTargetAppli."'");
          $strTarget = $_strTargetAppli;
        } elseif( $_strUrlAppli == "" ) {
          // url alkanet par défaut
          $strUrl = ALK_ALKANET."?token=".$strToken;
          $strTarget = $strDefaultTarget;
        } else {
          // url spécifique
          $strUrl = $_strUrlAppli;
          $strTarget = $_strTargetAppli;
        }
      } else {
        // par d'appli par défaut spécifiée : 
        // la page par défaut est la page de recherche ou la page mon espace si l'appli synd est installée
        $strToken = "cont_id=".$_cont_id."&appli_id=".( defined("ALK_B_ATYPE_SYND") && ALK_B_ATYPE_SYND==true ? "-2" : "-1" );
        $strUrl = ALK_ALKANET."?token=". AlkRequest::getEncodeParam($strToken);
        $strTarget = $strDefaultTarget;
      }
        
      $tabMenuSpace[$cpt] =
        array("id"          => $_cont_id,
              "url"         => $strUrl,
              "target"      => $strTarget,
              "title"       => $_strContIntitule,
              "shortTitle"  => $this->_tronqueItemMenu($_strContIntituleCourt),
              "public"      => $_iContPublic,
              "logo"        =>  ALK_MEDIA_URL."pictos/spacetype_".$_iContPublic.".gif",
              "nbChild"     => 0);
    }
    
    return $tabMenuSpace;
  }

  /**
   * Test si l'url est un appel a une fonction javascript
   *
   * @param strUrl url a tester
   * @return Retourne vrai si l'url est un appel a une fonction javascript, faux sinon
   */
  private function _testUrlJs($strUrl)
  {
    if( $strUrl == "" )
      return false;
    if( mb_strtolower(substr($strUrl, 0, 10)) == "javascript" )
      return true;
    return false;
  }


  /**
   * ajoute les paramètres php à la fonction javascript
   *
   * @param strNomJS     Nom de la fonction javascript
   * @param strToken     token ou paramètre passé à l'url de la fonction
   * @param strWindName  Dans le cas d'une ouverture popup, nom de la fenêtre à ouvrir
   * @return La chaine de caractères contenant l'appel complet de la fonction javascript
   */
  private function _getUrlJS($strNomJS, $strParam="", $strWindName="_blank")
  {
    return $strNomJS."('".$strParam."', '".$strWindName."')";
  }

  /**
   * Tronque l'intitulé du menu sur 2 lignes de 15 caractères
   *
   * @param strItemMenu Intitulé du menu
   * @return Retourne l'intitulé correctement formaté
   */
  private function _tronqueItemMenu($strItemMenu)
  {
    // effectue des cesures tous les 15 caractères
    $strItemMenu = wordwrap($strItemMenu, 32, "<br/>", 1);

    // coupe la chaine avant le second <br>
    $tabStrMenu = explode("<br/>", $strItemMenu);
    $strRes = $tabStrMenu[0];
    return $strRes;
  }

  /**
   * Tronque l'intitulé du sous-menu sur 1 ligne de 32 caractères
   *
   * @param strItemMenu Intitulé du sous-menu
   * @return Retourne l'intitulé correctement formaté
   */
  private function _tronqueItemSsMenu($strItemMenu)
  {
    // effectue des cesures tous les 15 caractères
    $strItemMenu = wordwrap($strItemMenu, 32, "<br/>", 1);
    
    // coupe la chaine avant le second <br>
    $tabStrMenu = explode("<br/>", $strItemMenu);
    $strRes = $tabStrMenu[0];
    return $strRes;
  }

  /**
   * Retourne la liste des liens a afficher. Classes de style utilisees : menuPath, menuPathBD
   *
   * @param tabLink           Tableau associatif contenant les liens-texte de navigation
   * @param strDefaultTarget  target par défaut sur les applis alkanet
   * @return Retourne une chaine HTML contenant les liens de navigation
   */
  public function getTabNav($tabLink, $strDefaultTarget="")
  {
    $user_id = AlkFactory::getSProperty("user_id", "-1");

    $strNav = "";
    $strSepar = " > ";
  
    // ajoute les espaces successifs
    $strListeCont = $this->getAppliProperty("CONT_ARBRE");
    $strListeCont = mb_substr($strListeCont,1,mb_strlen($strListeCont)-2);
    $strListeCont = mb_ereg_replace("-", ",", $strListeCont);
      
    $tabRes = array();

    if ($strListeCont != ""){
      $dsCont = $this->oQuery->getDs_listeEspaceNavByAgent($user_id, $strListeCont, 0);
      while( $drCont = $dsCont->GetRowIter() ) {
        $iDroit = $drCont->getValueName("DROIT");
        $icont_id = $drCont->getValueName("CONT_ID");
        $icont_intitule = $drCont->getValueName("CONT_INTITULE");
        $iAppliDefaut = $drCont->getValueName("DEFAUT");
        $idAppliDefaut = $drCont->getValueName("CONT_APPLI_DEFAUT");
        $strUrlAppli = $drCont->getValueName("ATYPE_URL");
        $strTargetAppli = $drCont->getValueName("ATYPE_URL_TARGET");
        $icont_appli_defaut = ( $iAppliDefaut==0 ? 0 : $idAppliDefaut );
        $strTargetAppli = ( $strTargetAppli!="" ? $strTargetAppli : $strDefaultTarget );

        $strUrl = ( $strUrlAppli == "" 
                    ? ALK_ALKANET."?token="
                    : (strpos($strUrlAppli, "?")==false ? $strUrlAppli."?" : $strUrlAppli."&"  ));
        $strParam = "cont_id=".$icont_id."&appli_id=";
        $bUseToken = ( $strUrlAppli == "" );

        if( $iDroit == 1 ) {
          if( $icont_appli_defaut==0 ) {
            // écran de recherche par défaut
            $strUrl .= ( $bUseToken 
                         ? AlkRequest::getEncodeParam($strParam."-1")
                         : $strParam."-1" );
          } else {
            // iSheet pris par défaut à la récupération
            $strToken = ( $bUseToken
                          ? AlkRequest::getEncodeParam($strParam.$icont_appli_defaut)
                          : $strParam.$icont_appli_defaut );
            if( $this->_testUrlJs($strUrlAppli) ) {
              $strUrl = $oSpace->_getUrlJS($strUrlAppli, "token=".$strToken, "");
            } else {
              $strUrl .= $strToken;
            }
          }

          $tabRes[] = array("titre" => $icont_intitule, "url" => $strUrl, "target" => $strTargetAppli, "type" => "espace");
        }
      }
    }

    // ajoute le tableau lié à l'application courante
    if( is_array($tabLink) ) {
      foreach($tabLink as $t) {
        $strTarget = ( array_key_exists("target", $t)==true ? $t["target"] : "");
        $tabRes[] = array("titre" => $t["titre"], "url" => $t["url"], "target" => $strTarget, "type" => "appli");
      }
    }

    return $tabRes;
  }
 
  /**
   * Envoi de mail aux admin concernant la création/modification d'un espace
   * @param cont_id             Identifiant de l'espace
   * @param cont_intitule       Intitulé de l'espace
   * @param cont_public         =0 espace privé, =1 espace public, =2 espace privé visible
   * @param iMode               Mode de traitement en cours (1=ajout, 2=modif, 3=ajout/suppr anim, 4=espace devient non réservé, 5=espace devient réservé)
   * @param tabUserIdAnimBefore tableau dont les clés correspondantes aux idUsers animateur avant modif, uniquement pour iMode=3
   */
  private function sendMailToAdmin($cont_id, $cont_intitule, $cont_public, $iMode, $tabUserIdAnimBefore=array()) 
  {
    $strNameFrom = AlkFactory::getSProperty("user_name", ALK_APP_TITLE);
    $strMailFrom = AlkFactory::getSProperty("user_mail", ALK_MAIL_ADMIN_MAIL);

    $dsUser = $this->oQuery->getDs_listeAnimateurByEspaceForCombo($cont_id);
    while( $drUser = $dsUser->getRowIter() ) {
      $idUser    = $drUser->getValueName("ID");
      $strNameTo = $drUser->getValueName("PRENOM_NOM");
      $strMailTo = $drUser->getValueName("MAIL");
      $strLocale = $drUser->getValueName("AGENT_LG");

      setMailAlkLocale($strLocale);

      $tabTypeEspace = array(_t("privé"), _t("public"), _t("privé (visible avec possibilité d'invitation)"));
       
      $strDate = date("d/m/Y", AlkFactory::getLocalDate()); 
      $strTime = date("H:i", AlkFactory::getLocalDate());
      
      $tabAssoc = array();
      $tabAssoc["msgTitle"] = _t("Espace de travail");
      $tabAssoc["msg"] = "";
      $mail_cle = "";
    
      switch( $iMode ) {
      case 1: // création de l'espace classique
        $mail_cle = "ESPACE_ADD_SPACE";
        $tabAssoc["msg"] = _f("Vous êtes animateur d'un nouvel espace de travail %s nommé '%s', créé le %s à %s.", 
                              $tabTypeEspace[$cont_public], $cont_intitule, $strDate, $strTime)."<br/>";
        
        // informations annexe pour la personnalisation
        $tabAssoc["cont_public"]   = $tabTypeEspace[$cont_public];
        $tabAssoc["cont_intitule"] = $cont_intitule;
        break;
    
      case 2 : // modification de l'espace
        // pas de message envoyé
        break;
  
      case 3 : // ajout- retrait du privilège animateur d'espace
        // rien à faire ici
        break;
        
      case 4 : // modification, l'espace devient réservé (implémentation partielle et non testé)
        /*$tabAssoc["msgcontext"] = _f("L'accès à l'espace de travail %s nommé '%s' est devenu réservé depuis le %s à %s.",
                                     $tabTypeEspace[$cont_public], $cont_intitule, $strDate, $strTime);
        $tabAssoc["msgcontext2"] = _t("Les administrateurs généraux n'auront accès qu'à l'administration de cet espace.").
          "<br/>".
          _t("Les animateurs, non administrateur général, peuvent rendre l'accès en décochant" .
            " la case à cocher 'Accès réservé' sur l'administration des propriétés de cet espace.").
          "<br/>";*/
        break;
    
      case 5 : // modification, l'espace devient classique (implémentation partielle et non testé)
        /*$tabAssoc["msgcontext"] = _f("L'accès à l'espace de travail %s nommé '%s' n'est plus réservé depuis le %s à %s.",
                                     $tabTypeEspace[$cont_public], $cont_intitule, $strDate, $strTime);
        $tabAssoc["msgcontext2"] = _t("Les administrateurs généraux ont de nouveau accès à cet espace en consultation.")."<br/>";*/
        break;    
      }
  
      if( $mail_cle != "" ) {
        
        $bSend = true;
        /*if( $iMode==3 ) {
          if( !array_key_exists("_".$idUser, $tabUserIdAnimBefore) ) {
            $tabAssoc["msgcontext"] = _f("Vous êtes devenu animateur de l'espace de travail %s nommé '%s' le %s à %s.",
                                        $tabTypeEspace[$cont_public], $cont_intitule, $strDate, $strTime);
            $tabAssoc["msgcontext2"] = _t("Désormais, vous possédez le privilège suffisant pour configurer et animer cet espace de travail.")."<br/>";
          } else {
            unset($tabUserIdAnimBefore["_".$idUser]);
            $bSend = false; 
          }
        }*/
        
        if( $bSend && $strMailTo != "" ) { 
          $oAlkMail = AlkFactory::getMail();
          $oAlkMail->setFrom($strNameFrom, $strMailFrom);
          $oAlkMail->addTo($strNameTo, $strMailTo);
          
          AlkMailing::SendMailType($oAlkMail, $mail_cle, $tabAssoc);
        }
      }
      
    }
    setCurrentAlkLocale();
    
    /** @todo : envoyer un mail à ceux encore présents dans $tabUserIdAnimBefore correspondant aux supprimés ? */
  }

  /**
   * Permet à une application d'appeler une methode spécifique sur l'ensemble des applications de l'espace
   * @param atype_id_from   AtypeId de l'application appelante
   * @param strMethod       Nom de la méthode à appeler sur les applications de l'espace courant
   * @param [2..n]          Arguments passés à la méthode appelée
   * @return mixed (y compris void)
   */
  public function callService($atype_id_from, $strMethod)
  {
    $tabParams = func_get_args();
    array_shift($tabParams);// supprime l'argument atype_id_from
    array_shift($tabParams);// supprime l'argument strMethod
    
    // Vérification que la méthode a été déclarée dans une interface AlkIntCallService
    if ( (interface_exists("AlkIntCallService".constant("ALK_ATYPE_ABREV_".$atype_id_from))
          && !method_exists("AlkIntCallService".constant("ALK_ATYPE_ABREV_".$atype_id_from), $strMethod) ) 
      || (!interface_exists("AlkIntCallService".constant("ALK_ATYPE_ABREV_".$atype_id_from))
          && !method_exists("AlkIntCallService", $strMethod) ) 
    ){
      $this->triggerError(__CLASS__."::".__FUNCTION__."<br/>" .
                          "La méthode ".$strMethod." doit être déclarée dans l'interface AlkIntCallService " .
                          "ou dans une interface dérivée portant le nom AlkIntCallService".constant("ALK_ATYPE_ABREV_".$atype_id_from), 
                          E_USER_ERROR);
    }
    
    
    $tabRes = array();
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");
      $atype_intitule = $drAppliType->getValueName("ATYPE_INTITULE");

      $oAppli = AlkFactory::getAppli($atype_id, -1);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, $strMethod) ){
        // Vérification que l'application hérite bien d'une interface AlkIntCallService
        if ( !($oAppli instanceof AlkIntCallService) ){
          $this->triggerError(__CLASS__."::".__FUNCTION__."<br/>" .
                              "L'application ".get_class($oAppli)." doit implémenter l'interface AlkIntCallService" .
                              " ou une de ses dérivées pour être appelée en callService sur le service ".$strMethod, 
                              E_USER_ERROR);
        }
        //Appel de la méthode
        $return = call_user_func_array(array($oAppli, $strMethod), $tabParams);
        if ( isset($return) && !empty($return) ){
          $tabRes[$atype_intitule] = $return;
        }
      }
    }
    return $tabRes;
  }
  
   /**
   * vérifie les liens gérés par l'application et retourne le résultat sous forme de tableau
   * @param $atype_id
   * @return array
   */
  public function verifLinks($atype_id=-1)
  {
    $tabRes = array();
    
    // on ne renvoie que le tableau correspondant à cet atype_id
    if ($atype_id != -1){
      $oAppli = AlkFactory::getAppli($atype_id);
      if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "verifLinks") )
        $tabRes = $oAppli->verifLinks();
    }else {
      $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
      while( $drAppliType = $dsAppliType->getRowIter() ) {
        $atype_id = $drAppliType->getValueName("ATYPE_ID");
        $oAppli = AlkFactory::getAppli($atype_id);
        if( !is_null($oAppli) && is_object($oAppli) && method_exists($oAppli, "verifLinks") )
          $tabRes[$atype_id] = $oAppli->verifLinks();
      }
   }
 
      
    return $tabRes;  
  }
  
  /**
   * Retourne le tableau des intervals utilisés dans la mise en place de tâches planifiées
   * @param cron_interval   string type d'interval souhaité ("jourssemaine", "joursmois", "mois")
   * @param bTous           booléen à true pour ajouter la clause "Tous" à la fin du tableau, false sinon (défaut false)
   * @return array
   */
  public function getTabCronInterval($cron_interval="jourssemaine", $bTous=false)
  {
    $tabInterval = array();
    
    $i = -1;
    switch ( $cron_interval ) {
      case "jourssemaine" :
        $tabInterval = array("Lundi" => pow(2, ++$i), "Mardi" => pow(2, ++$i), "Mercredi" => pow(2, ++$i), "Jeudi" => pow(2, ++$i),
                             "Vendredi" => pow(2, ++$i), "Samedi" => pow(2, ++$i), "Dimanche" => pow(2, ++$i),
        );
      break;
      case "joursmois" :
        $tabInterval = array("01" => pow(2, ++$i), "02" => pow(2, ++$i), "03" => pow(2, ++$i), "04" => pow(2, ++$i), "05" => pow(2, ++$i), "06" => pow(2, ++$i),
                             "07" => pow(2, ++$i), "08" => pow(2, ++$i), "09" => pow(2, ++$i), "10" => pow(2, ++$i), "11" => pow(2, ++$i), "12" => pow(2, ++$i),
                             "13" => pow(2, ++$i), "14" => pow(2, ++$i), "15" => pow(2, ++$i), "16" => pow(2, ++$i), "17" => pow(2, ++$i), "18" => pow(2, ++$i),
                             "19" => pow(2, ++$i), "20" => pow(2, ++$i), "21" => pow(2, ++$i), "22" => pow(2, ++$i), "23" => pow(2, ++$i), "24" => pow(2, ++$i),
                             "25" => pow(2, ++$i), "26" => pow(2, ++$i), "27" => pow(2, ++$i), "28" => pow(2, ++$i), "29" => pow(2, ++$i), "30" => pow(2, ++$i),
                             "31" => pow(2, ++$i),
        );
      break;
      case "mois" :
        $tabInterval = array("Janvier" => pow(2, ++$i), "Février" => pow(2, ++$i), "Mars" => pow(2, ++$i), "Avril" => pow(2, ++$i), "Mai" => pow(2, ++$i), "Juin" => pow(2, ++$i),
                             "Juillet" => pow(2, ++$i), "Août" => pow(2, ++$i), "Septembre" => pow(2, ++$i), "Octobre" => pow(2, ++$i), "Novembre" => pow(2, ++$i), "Décembre" => pow(2, ++$i),
        );
      break;
    }
    
    if ( $bTous && !empty($tabInterval) ) {
      $tabInterval["Tous"] = pow(2, ++$i) - 1;
    }
    
    return $tabInterval;
  }
  
  /**
   * Scrute les tâches planifiées à exécuter à l'instant t
   * @param time    timestamp heure de lancement du démon
   */
  public function cron($time=null, $cont_id=0, $appli_id=-1, $action_id=-1)
  {
    $cron_interval = ( defined("ALK_CRON_INTERVAL") ? ALK_CRON_INTERVAL : 15 );
    
    $tabJoursSemaine = $this->getTabCronInterval("jourssemaine");
    $tabDays = array(1 => "Monday", 2 => "Tuesday", 3 => "Wednesday", 4 => "Thursday", 5 => "Friday", 6 => "Saturday", 7 => "Sunday");
    
    $dbConn = AlkFactory::getDbConn();
    $oQueryEspace = AlkFactory::getQuery(ALK_ATYPE_ID_ESPACE);
    $oQueryActionEspace = AlkFactory::getQueryAction(ALK_ATYPE_ID_ESPACE);
    
    $oDsTask = $oQueryEspace->getDs_listeCronTask();
    while ( $oDrTask = $oDsTask->getRowIter() ) {
      $task_id           = $oDrTask->getValueName("TASK_ID");
      $cont_id           = $oDrTask->getValueName("CONT_ID");
      $atype_id          = $oDrTask->getValueName("ATYPE_ID");
      $appli_id          = $oDrTask->getValueName("APPLI_ID");
      $action_id         = $oDrTask->getValueName("ACTION_ID");
      $action_intitule   = $oDrTask->getValueName("ACTION_INTITULE");
      $task_typetask     = $oDrTask->getValueName("TASK_TYPETASK");
      $task_date         = $oDrTask->getValueName("TASK_DATE");
      $task_jourssemaine = $oDrTask->getValueName("TASK_JOURSSEMAINE");
      $task_joursmois    = $oDrTask->getValueName("TASK_JOURSMOIS");
      $task_mois         = $oDrTask->getValueName("TASK_MOIS");
      $task_typetime     = $oDrTask->getValueName("TASK_TYPETIME");
      $task_time         = $oDrTask->getValueName("TASK_TIME");
      $task_datelastexec = $oDrTask->getValueName("TASK_DATELASTEXEC");
      $task_encours      = $oDrTask->getValueName("TASK_ENCOURS");
      
      echo $this->formatCronLog("Analyse tâche planifiée numéro ".$task_id." ...");
      echo $this->formatCronLog("Action : ".$action_intitule);
      
      // tâche planifiée à traiter
      if ( !$task_encours ) {
        $iLastExecDay = 0;
        $iCurrentDay  = (int)date("N", $time); // 1 (pour Lundi) à 7 (pour Dimanche)
        
        $taskTimeLastExec = 0;
        $taskTimeExec     = 0;
        
        // timestamp de la dernière exécution de la tâche planifiée
        if ( $task_datelastexec ) {
          $tabDateTimeLastExec  = explode(" ", $task_datelastexec);
          $tabDateLastExec      = explode("/", $tabDateTimeLastExec[0]);
          $tabTimeLastExec      = explode(":", $tabDateTimeLastExec[1]);
          $taskTimeLastExec     = strtotime($tabDateLastExec[2]."-".$tabDateLastExec[1]."-".$tabDateLastExec[0]." ".$tabTimeLastExec[0].":".$tabTimeLastExec[1]);
          $iLastExecDay         = (int)date("N", $taskTimeLastExec);
        }
        
        // récupère le timestamp de la prochaine exécution
        switch ( $task_typetask ) {
          case 0 :  // périodique
            $tabNextTimeExecByDay = array();
            $j = 1;
            foreach ( $tabJoursSemaine as $jour => $value ) {   // parcourt les jours de la semaine
              if ( ($task_jourssemaine & $value) == $value ) {  // le jour est prévu dans la définition de la tâche planifiée
                switch ( $task_typetime ) {
                  case 0 :  // fréquence
                    if ( $j == $iCurrentDay ) {
                      if ( $iLastExecDay != $iCurrentDay ) {  // première exécution de la tâche planifiée dans la journée courante
                        $tabNextTimeExecByDay[$j] = strtotime(date("Y-m-d H:i", $time));  // affecte le timestamp du moment de l'exécution du démon sans les secondes
                      } else {
                        // timestamp calculé pour l'exécution de la tâche planifiée (= dernière exécution + heure de fréquence)
                        $tabTime = explode(":", $task_time);
                        $timeHour = $tabTime[0];
                        $timeMin = $tabTime[1];
                        $tabNextTimeExecByDay[$j] = strtotime("+".$timeHour." hours ".$timeMin." minutes", $taskTimeLastExec);
                        // resynchronise le timestamp calculé au cas où celui-ci serait antérieur à la précédente exécution du démon (démon non exécuté pendant un certain temps par exemple)
                        // sinon la tâche reste bloquée pour la suite de la journée courante
                        if ( $tabNextTimeExecByDay[$j] <= (strtotime("-".$cron_interval." minutes", $time)) ) {
                          $tabNextTimeExecByDay[$j] = strtotime(date("Y-m-d H:i", $time));  // affecte le timestamp du moment de l'exécution du démon sans les secondes
                        }
                      }
                    } else {
                      $tabNextTimeExecByDay[$j] = strtotime("next ".$tabDays[$j]);
                    }
                  break;
                  case 1 :  // fixe
                    if ( $j == $iCurrentDay ) {
                      $taskTimeExecToday = strtotime(date("Y-m-d", $time)." ".$task_time);  // timestamp du moment de l'exécution de la tâche planifiée dans la journée courante
                      if ( strtotime("-".$cron_interval." minutes", $time) >= $taskTimeExecToday ) {  // le moment de l'exécution de la tâche planifiée est dépassé
                        $tabNextTimeExecByDay[$j] = strtotime(date("Y-m-d", strtotime("next ".$tabDays[$j]))." ".$task_time); // affecte le timestamp du jour de la semaine prochaine
                      } else {
                        $tabNextTimeExecByDay[$j] = $taskTimeExecToday;
                      }
                    } else {
                      $tabNextTimeExecByDay[$j] = strtotime(date("Y-m-d", strtotime("next ".$tabDays[$j]))." ".$task_time); // affecte le timestamp du jour $j
                    }
                  break;
                }
              }
              
              $j++;
            } // end foreach
            if ( array_key_exists($iCurrentDay, $tabNextTimeExecByDay) ) {
              $taskTimeExec = $tabNextTimeExecByDay[$iCurrentDay];
            } else {
              $timeExecFirstDay = 0;
              foreach ( $tabNextTimeExecByDay as $iDay => $taskNextTimeExec ) {
                if ( $timeExecFirstDay == 0 ) {
                  $timeExecFirstDay = $tabNextTimeExecByDay[$iDay];
            }
                if ( $iDay > $iCurrentDay ) {
                  $taskTimeExec = $tabNextTimeExecByDay[$iDay];
                }
              }
              if ( $taskTimeExec == 0 ) {
                $taskTimeExec = $timeExecFirstDay;
              }
            }
          break;
          
          case 1 :  // ponctuelle
            // timestamp prévu pour l'exécution de la tâche planifiée
            $tabDateExec = explode("/", $task_date);
            $taskTimeExec = strtotime($tabDateExec[2]."-".$tabDateExec[1]."-".$tabDateExec[0]." ".$task_time);
          break;
        }
        
        // Logs
        echo $this->formatCronLog("Type de tâche (0 = Périodique, 1 = Ponctuelle) : ".$task_typetask);
        if ( $task_typetask == 0 ) {
          echo $this->formatCronLog("Type d'heure (0 = fréquence, 1 = fixe ) : ".$task_typetime);
        }
        echo $this->formatCronLog("Date et heure de la dernière exécution : ".( $taskTimeLastExec != 0 ? date("d/m/Y H:i:s", $taskTimeLastExec) : "" ));
        echo $this->formatCronLog("Date et heure de la prochaine exécution : ".date("d/m/Y H:i:s", $taskTimeExec));
        
        /*
         * si l'heure d'exécution prévue (ponctuelle) ou calculée (périodique) se situe entre la dernière exécution et l'exécution courante du démon
         * et que l'heure de la dernière exécution ne se situe pas dans cet interval, 
         * on exécute la tâche planifiée
         */
        if ( (strtotime("-".$cron_interval." minutes", $time) < $taskTimeExec) && ($taskTimeExec <= $time)
          && !((strtotime("-".$cron_interval." minutes", $time) < $taskTimeLastExec) && ($taskTimeLastExec <= $time))
        ) {
          // place un verrou sur la tâche
          $dbConn->executeSql("update SIT_CRON_TASK set TASK_ENCOURS=1 where TASK_ID=".$task_id);
          
          // Logs
          echo $this->formatCronLog("Début d'exécution de la tâche planifiée ...");
          
          // execute la tâche
          $oAppli = AlkFactory::getAppli($atype_id, $appli_id);
          $oAppli->cron($time, $cont_id, $appli_id, $action_id);
          
          // dévérouille la tâche et met à jour la date de dernière exécution (correspond au moment du lancement du démon et non de la tâche => nécessaire pour les calculs de fréquence)
          $dbConn->executeSql("update SIT_CRON_TASK set TASK_ENCOURS=0, TASK_DATELASTEXEC=".$dbConn->getDateFormat("DD/MM/YYYY HH:MI", "'".date("d/m/Y H:i", $time)."'")." where TASK_ID=".$task_id);
          
          // Logs
          echo $this->formatCronLog("Fin d'exécution de la tâche planifiée");
        }
        
      }
      
      // tâche planifiée en cours d'exécution
      else {
        echo $this->formatCronLog("Tâche planifiée en cours d'exécution");
      }
      
      echo $this->formatCronLog("**************************************************************");
    } // fin while : boucle sur chaque tâche planifiée
  }
  
  /**
   * Retourne la liste de tous les abonnements
   * @return array
   */
  public function getListAbonnements()
  {
    $tabRes = array();
    $user_id = AlkFactory::getSProperty("user_id", -1);
    $dsAppliType = $this->oQuery->getDs_listeTypeAppliByListId("", "", "1", true);
    while( $drAppliType = $dsAppliType->getRowIter() ) {
      $atype_id = $drAppliType->getValueName("ATYPE_ID");
      $oQuery = AlkFactory::getQuery($atype_id);
      if( !is_null($oQuery) && is_object($oQuery) && method_exists($oQuery, "getListAbonnements") ) {
        $tabRes[$atype_id] = $oQuery->getListAbonnements($user_id);
      }
    }
    
    return $tabRes;
  }
  
  /**
   * Enregristre les élements de connexions d'un utilisateur
   * @param agent_id identifiant de l'uitilisateur connecté
   * @param session_id identifiant de la session courante
   * @param cont_id identifiant de l'espace de connexion
   * @param appli_id identifiant de l'application de connexion
   * @param page_contid identifiant de l'espace de connexion (renseigné s'il s'agit d'une page de type collab)
   * @param page_appliid identifiant de l'application de connexion (renseigné s'il s'agit d'une page de type collab)
   */
  public function updateConnnexionUser($agent_id, $session_id, $cont_id, $appli_id, $page_contid=-1, $page_appliid=-1){
    
    if ($page_appliid != -1 && $page_appliid!= ""){
      $appli_id = $page_appliid;
    }
    if ($page_contid != -1 && $page_contid!= ""){
      $cont_id = $page_contid;
    }
    
    // suppression de la ligne en base si elle existe déjà
    $this->oQueryAction->deleteConnexionUser($agent_id, $session_id);
    
    // insertion de la ligne de connexion
    $this->oQueryAction->insertConnexionUser($agent_id, $session_id, $cont_id, $appli_id);
  }
  
  public function deleteConnnexionUser($date_up){
    $this->oQueryAction->deleteConnexionUser(-1, -1, $date_up);
  }
  
  /**
   * @see AlkIntOptions::getDsOptions
   * @param int $appli_id
   */
  public function getDsOptions($appli_id) {
    return $this->oQuery->getDsOptions($appli_id);
  }
  
  /**
   * @see AlkIntOptions::addOption
   * @param int $appli_id
   * @param string $alias
   * @param string $intitule
   * @param string $description
   */
  public function addOption($appli_id, $alias, $intitule, $description) {
    $this->oQueryAction->addOption($appli_id, $alias, $intitule, $description);
  }
  
  /**
   * @see AlkIntOptions::removeOption
   * @param int $option_id
   */
  public function removeOption($option_id) {
    $this->oQueryAction->removeOption($option_id);
  }
  
  public function getRessourceInfo($ress_id, $resstype_id){
    $tabData = array();
    $dsInfoOption = $this->oQuery->getDsOptionById($ress_id);
    while($drInfo = $dsInfoOption->getRowIter()){
      $tabData[$ress_id]["nom"]        = $drInfo->getValueName("OPTION_INTITULE");
      $tabData[$ress_id]["description"]= $drInfo->getValueName("OPTION_DESC");
      $tabData[$ress_id]["alias"]      = $drInfo->getValueName("OPTION_ALIAS");
    }
    return $tabData;
  }

  /**
   * Retourne le libellé de la page d'aide associée à l'application
   * @return string
   */
  public function getHelp(){
    return _t("Espaces de travail");
  }

}
?>