<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Form
Module fournissant les classes d'affichage Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

require_once(ALK_ALKANET_ROOT_PATH.ALK_ROOT_CLASSE."form/alkhtmllist.class.php");

/**
 * @package Alkanet_Class_Form
 * @class AlkHtmlTreeTable
 * 
 * Classe affichant un arbre de données sous un format d'affichage d'arbre par défaut
 *        mais également sous des formats définis par l'utilisateur (définition d'un template pour l'arbre).
 *        Chaque noeud de l'arbre (AlkTreeData) est caractérisé par son noeud parent et
 *        optionnellement) par une catégorie d'affichage. 
 *        A chaque catégorie d'affichage (AlkTreeCategorie), on peut associer différentes propriétés 
 *        d'affichage :
 *          <ul><li>une icone</li>
 *          <li>un template</li><ul>
 *        Si aucun template n'est défini pour l'arbre ou ses noeuds, l'affichage sera celui d'un arbre Javascript
 *        Sinon, l'arbre ou ses noeuds suivront l'affichage déterminé par leur template
 */
class AlkHtmlTreeTable extends AlkHtmlBlock
{
  public $tabProperties;
  public $tabCategorieName;
  public $tabCategorie;
  public $tabData;
  public $oRoot;
  public $bOpenAll;
  public $opened_id;
  
  /**
   *  Constructeur
   * @param oForm   Formulaire propriétaire
   * @param name    Nom de l'arbre
   * @param strTitle    titre du bloc
   * @param strDesc     descriptif du bloc
   * @param iWidthLabel largeur dédiée au label du controle (=180 par défaut)
   * @param iWidthCtrl  largeur dédiée au controle de saisie (=390 par défaut)
   * @param iFirstNiv   niveau de la racine, =0 par défaut, peut valoir -1 pour éviter une marge à gauche   
   */
  public function __construct($oForm, $name, $strTitle="", $strDesc="", $iWidthLabel="180", $iWidthCtrl="390", $iFirstNiv=0)
  {
    // réduit la taille du tableau intérieur de 20 pour éviter l'affichage exécessif des scrollbars
    parent::__construct($oForm, $name, $strTitle, $strDesc, $iWidthLabel, $iWidthCtrl-20);
    
    if ( !is_null($this->oForm) )
      $this->oForm->addPanel($this);
    $this->tabProperties = array("iFirstNiv" => $iFirstNiv);
    $this->tabCategorie = array();
    $this->tabCategorieName = array();
    $this->tabData = array();
    $this->oRoot = new AlkTreeData($this, "0");
    $this->bOpenAll = true;
    $this->opened_id = "-1";
  }
  
  /**
   *  Ajoute une propriété à l'arbre qui sera exploitée lors de l'affichage
   * @param name    Nom de la propriété
   * @param value   Valeur de la propriété
   * @param bForce  Pour écraser la valeur d'une propriété de même nom déjà définie (true par défaut)
   */
  public function addProperty($name, $value, $bForce=true)
  {
    if ( $bForce || !array_key_exists($name, $this->tabProperties) ){
      $this->tabProperties[$name] = $value;
    }
  }
  
  /**
   *  Retourne la valeur d'une propriété d'affichage définie par la méthode addProperty
   * @param name        Nom de la propriété
   * @param strDefault  Valeur par défaut attendu en retour
   * @return mixed    Valeur de la propriété d'affichage
   */
  public function getPropertyValue($name, $strDefault="")
  {
    if ( array_key_exists($name, $this->tabProperties) ){
      return $this->tabProperties[$name];
    }
    return $strDefault;
  }
  
  
  /**
   *  Crée une catégorie d'affichage et en retourne une référence
   * @param name      Nom de la catégorie
   * @param icone     Image représentant les éléments de cette catégorie
   * @param template  Fichier template pour représenter les éléments de cette catégorie
   * @return & AlkTreeCategorie
   */
  public function &addCategorie($name, $icone="", $template="")
  {
    $oCategorie = new AlkTreeCategorie($name, $icone, $template);
    $this->addCategorieByObj($oCategorie);
    return $oCategorie;
  }
  
  /**
   *  Ajoute une instance catégorie d'affichage
   * @param & oCategorie   Catégorie d'affichage
   */
  protected function addCategorieByObj(&$oCategorie)
  {
    if ( is_null($oCategorie) ) return;
    if ( in_array($oCategorie->name, $this->tabCategorieName) ) return;
    $this->tabCategorie[] =& $oCategorie;
    $this->tabCategorieName[] = $oCategorie->name;
  }
  
  
  
  /**
   *  Crée un "noeud de données" de parent donné et en retourne une référence 
   * @param oParent     AlkTreeData ou null, Noeud parent du nouveau noeud
   * @param oCategorie  Catégorie de représentation du noeud
   * @param node_id     Identifiant du noeud
   * @param tabData     Tableau de donnée du noeud
   * @param tabActions  Tableau des actions du noeud
   * @return & AlkTreeData
   */
  public function &addNode($oParent, $oCategorie=null, $node_id="", $tabData=array(), $tabActions=array(), $strIcone="")
  {
    $this->addCategorieByObj($oCategorie);    
    if ( is_null($oParent) ){
      if ( $node_id=="" ) $node_id = $this->oRoot->getNextNodeId();
      $oData = new AlkTreeData($this, $node_id, $this->oRoot, $oCategorie, $tabData, $tabActions, $strIcone);
    }
    else if ( is_object($oParent) && $oParent->isTypeOf("AlkTreeData") && $oParent->isInTree($this) ){
      if ( $node_id=="" ) $node_id = $oParent->getNextNodeId();
      $oData = new AlkTreeData($this, $node_id, $oParent, $oCategorie, $tabData, $tabActions, $strIcone);
    }
    else 
      return self::$oNull;
    $this->tabData[] = $oData;
    return $oData;
  }
  
  /**
   *  Définit les propriétés par défaut d'un arbre de type Treeview (@see /lib/lib_arbre.js)
   */
  protected function setDefaultProperties()
  {
    $this->addProperty("iTabLg",           "'".$this->iWidthCtrl."'", false);
    $this->addProperty("classLinkFolder",  "''", false);
    $this->addProperty("classTextFolder",  "''", false);
    $this->addProperty("classLinkDoc",     "''", false);
    $this->addProperty("classTextDoc",     "''", false);
    $this->addProperty("imgNodeOpen",      "'".ALK_MEDIA_URL."images/icon_dossier_close.gif'", false);
    $this->addProperty("imgNodeClose",     "'".ALK_MEDIA_URL."images/icon_dossier_cant_open.gif'", false);
    $this->addProperty("imgNodeLineClose", "'".ALK_MEDIA_URL."images/icon_arbo_open.gif'");
    $this->addProperty("imgNode",          "'".ALK_MEDIA_URL."images/icon_dossier_cant_open.gif'", false);
    $this->addProperty("imgIconC",         "'".ALK_MEDIA_URL."images/icon_arbo_open.gif'", false);
    $this->addProperty("imgIconO",         "'".ALK_MEDIA_URL."images/icon_arbo_close.gif'", false);
    $this->addProperty("imgIconCOC",       "'".ALK_MEDIA_URL."images/icon_arbo_cant_open.gif'", false);
  }
  
  /**
   *  Retourne le contenu html de l'arbre de données
   *        Si aucun template n'est défini pour l'arbre ou ses noeuds, l'affichage sera celui 
   *        d'un arbre Javascript (@see /lib/lib_arbre.js)
   *        Sinon, l'arbre ou ses noeuds suivront l'affichage déterminé par leur template
   * @return string html
   */
  public function getHtml()
  {
    $strHtml = "";
    $nbTemplate = 0;
    foreach ($this->tabCategorie as $oCategorie){
      if ( $oCategorie->template!="" ){
        $nbTemplate++;
      }
    }
    
    // Template sur l'arbre définit
    if ( $this->strTemplateFile!="" ){
      foreach ($this->tabProperties as $property=>$value){
        $this->oTemplate->assign($property, $value);
      }
      $contents = $this->oRoot->getHtml();
      if ( $contents!="" ){
        $this->oTemplate->assign("contents", $contents);
        $strHtml .= $this->oTemplate->fetch($this->strTemplateFile);
      }
    }
    
    // Aucun template : arbre javascript
    else if ( !($nbTemplate==count($this->tabCategorie) && $nbTemplate>0) ){
      $this->addScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_layer.js");
      $this->addScriptJs(ALK_ALKANET_ROOT_URL.ALK_ROOT_LIB."lib_arbre.js");
      $strJs = "var obj_".$this->guid." = new Treeview(document, 'layer".$this->guid."', 1, '".$this->guid."');";
      $this->setDefaultProperties();
      foreach ($this->tabProperties as $property=>$value){
        $strJs .= "obj_".$this->guid.".".$property." = ".$value.";";
      }
      $this->addJs($strJs);

      if ($this->title!=""){
        $strHtml .= "<table summary='treeTable' class='".$this->cssBlock."' id='".$this->guid."'>" .
          "<thead>" .
          "<tr><td class='rowDimension' style='width:".($this->iWidthCtrl+20)."px;font-size:0px;' height='0'>".
          "<div style='width:".$this->iWidthCtrl."px; height:0px;font-size:0px;'></div></td></tr>".
          "<tr><td class='main'>".$this->title."</td></tr>" .
          "</thead>" .
          "</table>";
      }
      $strHtml .= "<div id='layer".$this->guid."'".
         " style='overflow:auto; text-align:left;".
         ( $this->iWidthLabel!="" 
           ? "height:".$this->iWidthLabel."px;" 
           : "" ).
         "width:".($this->iWidthCtrl+20)."px;'></div>";

      $strHtml .= $this->oRoot->getHtml();
      if ( $this->bOpenAll )
        $this->addOnload("obj_".$this->guid.".OpenAll();");
      else if ( $this->opened_id!=-1 ){
        $this->addOnload("obj_".$this->guid.".OpenNode('".$this->opened_id."');");
      }
      $this->addOnload("obj_".$this->guid.".Draw();");
      
    }
    
    // Pas de template sur l'arbre défini mais noeuds en templates : Donne les propriétés de l'arbre aux noeuds
    else {
      $this->oTemplate = AlkFactory::getTemplate();
      foreach ($this->tabProperties as $property=>$value){
        $this->oTemplate->assign($property, $value);
      }
      $strHtml .= $this->oRoot->getHtml();
    }
    return $strHtml;
  }
  

}
/**
 * @file alkhtmltreetable.class.php
 * @Class AlkTreeData
 * @package Alkanet_Class_Form 
 *  Noeud d'un arbre AlkHtmlTreeTable caractérisé par son noeud parent et
 *        optionnellement) par une catégorie d'affichage. 
 */
class AlkTreeData extends AlkHtmlPanel 
{
  public $oTree;
  
  public $node_id;
  
  public $node_niveau;
  
  public $tabData;
  
  public $oParent;
  
  public $oCategorie;
  
  public $tabChildren;
  
  public $tabActions; 
  
  public $bSelected;
  
  public $icone;
  
  /**
   *  Constructeur
   * @param AlkHtmlTreeTable oTree  Arbre possesseur du noeud
   * @param node_id     Identifiant du noeud
   * @param oParent     Parent du noeud
   * @param oCategorie  Catégorie du noeud
   * @param tabData     Tableau des données du noeud
   * @param tabAction   Tableau des actions du noeud
   */
  public function __construct(AlkHtmlTreeTable $oTree, $node_id, $oParent=null, $oCategorie=null, $tabData=array(), $tabActions=array(), $icone="")
  {
    $this->oTree = $oTree;
    $this->node_id      = $node_id;
    $this->tabChildren  = array();
    $this->tabData      = $tabData;
    $this->tabActions   = $tabActions;
    $this->icone        = $icone;
    $this->oCategorie   = $oCategorie;
    if (!is_null($this->oCategorie)){
      $this->oCategorie->addData($this);
    } 
    $this->oParent = $oParent;
    $this->node_niveau = $this->oTree->tabProperties["iFirstNiv"];
    if (!is_null($this->oParent)){
      $this->node_niveau = $this->oParent->node_niveau+1;
      $this->oParent->addChildren($this);
    }
    $this->bSelected = false;
  }
  
  /**
   *  Détermine si un noeud est possédé par l'arbre donné
   * @param AlkHtmlTreeTable oTree  Arbre à évaluer en possesseur du noeud
   * @param boolean   L'arbre du noeud est bien celui passé en paramètre
   */
  public function isInTree(AlkHtmlTreeTable $oTree)
  {
    return $this->oTree->equals($oTree);
  }
  
  /**
   *  Ajoute une donnée au tableau des données
   * @param oData (mixed) Donnée à ajouter
   * @param index         Index de la donnée ("" par défaut)
   */
  public function addData($oData, $index="")
  {
    if ($index==""){
      $this->tabData[] = $oData;
    }
    else if ($index=="node_id"){
      $this->node_id = $oData;
    }
    else {
      $this->tabData[$index] = $oData;
    }
  }
  
  /**
   *  Ajoute une action au tableau des actions
   * @param index       Index de l'action
   * @param strAction   Valeur de l'action
   */
  public function addAction($index, $strAction)
  {
    $this->tabActions[$index] = $strAction;
  }
  
  
  /**
   *  Ajoute un fils au noeud courant
   * @param oChild   AlkTreeData   Objet fils à ajouter
   */
  public function addChildren(AlkTreeData $oChild)
  {
    $this->tabChildren[] = $oChild;
  }  
  
  /**
   *  Retire un fils au noeud courant
   * @param oChild   AlkTreeData   Objet fils à retirer
   */
  public function removeChildren(AlkTreeData $oChild)
  {
    $tmp = $this->tabChildren;
    foreach ($this->tabChildren as $iChild=>$oChildren){
      if ( $oChildren->equals($oChild) ){
        unset($tmp[$iChild]);
      }
    }
    $this->tabChildren = $tmp;
  }  
  
  /**
   *  Sélectionne le noeud courant
   */
  public function selectNode()
  {
    $this->bSelected = true;
  }
  /**
   *  Désélectionne le noeud courant
   */
  public function unselectNode()
  {
    $this->bSelected = false;
  }
  /**
   *  Retourne le contenu html du noeud
   *        Si la catégorie est stricte, on n'affiche le noeud que s'il possède des fils
   *        Si aucun template n'est défini sur la catégorie, on crée un noeud dans un  
   *        arbre Javascript (@see /lib/lib_arbre.js)
   *        Sinon, l'affichage est celui déterminé par le template de catégorie
   * @return string html
   */
  public function getHtml()
  {
    $strHtml = "";
    if ( !is_null($this->oCategorie) && $this->oCategorie->isStrict() ){
      if ( empty($this->tabChildren) ){
        if ( !is_null($this->oParent) )  
          $this->oParent->removeChildren($this);
        return "";
      }
    }
    if ( !is_null($this->oParent) ){
      if ( is_null($this->oCategorie) || $this->oCategorie->template==""){
        $strData = "";
        foreach ($this->tabData as $oData){
          $strGlue = ($strData=="" ? "" : " ");
          if (is_object($oData) && $oData->isTypeOf(self::ALK_CLASS_HTMLPANEL)){
            $strData .= $strGlue.$oData->getHtml();
          }
          else {
            $strData .= $strGlue.$oData;          
          }
        }
        $strJs = 
          " \n var oNode = obj_".$this->oTree->getGUID().".AddFolder(\"_".$this->node_id."\"" .
            ", \"_".(is_null($this->oParent) ? "-1" : $this->oParent->node_id)."\"" .
            ", \"" .($this->bSelected ? "<b>" : "").$strData .($this->bSelected ? "</b>" : "")."\"".
            ", ".$this->node_niveau.
            ", ".(array_key_exists("iDroitLink", $this->tabActions) ? $this->tabActions["iDroitLink"] : "0").
            ", ".(array_key_exists("iDroitAction", $this->tabActions) ? $this->tabActions["iDroitAction"] : "0").
            ", \"".(array_key_exists(ALK_TREEACTION_URL, $this->tabActions) ? $this->tabActions[ALK_TREEACTION_URL] : "")."\"".
            ", \"\"".
            ");";
        if ( !is_null($this->oCategorie) && $this->oCategorie->getProperty("icone")!="" ){
          $strJs .= 
          " \n oNode.SetIcone('".($this->icone!="" ? $this->icone : $this->oCategorie->getProperty("icone"))."');";
        }
        $this->addJs($strJs);
      }
      else if ( !is_null($this->oCategorie) ){
        $this->oTemplate =& AlkFactory::getTemplate();
        $strChildren = ""; 
        foreach ($this->tabChildren as $oChild){
          $strChildren .= $oChild->getHtml();
        } 
        $this->oTemplate->assign_by_ref("oNodeObj", $this);
        $this->oTemplate->assign("oNode", $this->tabData);
        $this->oTemplate->assign("strChildren", $strChildren);
        $strHtml .= $this->oTemplate->fetch($this->oCategorie->template); 
           
      }
    }
    if ( !isset($strChildren) ){
      foreach ($this->tabChildren as $oChild){
        $strHtml .= $oChild->getHtml();
      }
    }
    return $strHtml;
  }
  
  /**
   *  Retourne le prochain identifiant d'un fils de l'arbre
   * @return string
   */
  public function getNextNodeId()
  {
    return ($this->node_id!="" ? $this->node_id."_" : "").count($this->tabChildren);
  }
}

$iActions = 0;
define("ALK_TREEACTION_URL", ++$iActions);
define("ALK_TREEACTION_RADIO", ++$iActions);
define("ALK_TREEACTION_CHECK", ++$iActions);
define("ALK_TREEACTION_CUT", ++$iActions);
define("ALK_TREEACTION_COPY", ++$iActions);
define("ALK_TREEACTION_PASTE", ++$iActions);

define("ALK_TREEACTIONPOS_ALL", ++$iActions);
define("ALK_TREEACTIONPOS_BEFORE", ++$iActions);
define("ALK_TREEACTIONPOS_AFTER", ++$iActions);

/**
 * @file alkhtmltreetable.class.php
 * @Class AlkTreeCategorie
 * @package Alkanet_Class_Form 
 *  Catégorie d'affichage et de gestion de noeuds d'un arbre AlkHtmlTreeTable
 */
class AlkTreeCategorie extends AlkObject
{
  public $bStrict;
  
  public $tabData;
  
  public $name;
  
  public $icone;
  
  public $template;
  
  public $tabActions;
  
  /**
   *  Constructeur
   * @param name      Nom de la catégorie
   * @param icone     Icone représentant la catégorie ("" par défaut)
   * @param template  Template représentant la catégorie ("" par défaut)
   */
  public function __construct($name, $icone="", $template="")
  {
    $this->name = $name;
    $this->icone = $icone;
    $this->template = $template;
    
    $this->bStrict = false;
    $this->tabActions = array(ALK_TREEACTIONPOS_ALL=>array(), ALK_TREEACTIONPOS_BEFORE=>array(), ALK_TREEACTIONPOS_AFTER=>array(), );
    $this->tabData = array();
  }
  /**
   *  Ajoute un noeud dans la catégorie
   * @param oData AlkTreeData   noeud à ajouter
   */
  public function addData(AlkTreeData $oData)
  {
    $this->tabData[] = $oData;
  }
  /**
   *  Ajoute une action pour les noeuds dans la catégorie
   * @param iAction     Type d'action (ALK_TREEACTION_xxx)
   * @param nameAction  Nom de l'action
   * @param actionPos   Position de l'action vis à vis du noeud (ALK_TREEACTIONPOS _ALL=le noeud, _BEFORE=Avant le noeud, _AFTER=Après le noeud)
   */
  public function addAction($iAction, $nameAction, $actionPos=ALK_TREEACTIONPOS_ALL)
  {
    $this->tabActions[$actionPos][$iAction] = $nameAction;
  }
  
  /**
   *  Retourne le prochain identifiant d'un noeud de cette catégorie
   * @return string
   */
  public function getNextNodeId()
  {
    return ($this->name!="" ? $this->name."_" : "").count($this->tabData);
  }
  
  /**
   *  Indique si la catégorie est stricte (noeuds fils obligatoires)
   * @return boolean
   */
  public function isStrict()
  {
    return $this->bStrict;
  }
  
  /**
   *  Caractérise la catégorie en catégorie stricte (noeuds fils obligatoires)
   * @param bStrict boolean
   */
  public function setIsStrict($bStrict)
  {
    $this->bStrict = $bStrict;
  }
}
?>