<?php
/*licence/ 

Module écrit, supporté par la société Alkante SAS <alkante@alkante.com>

Nom du module : Alkanet::Class::Form
Module fournissant les classes d'affichage Alkanet.
Ce module appartient au framework Alkanet.

Ce logiciel est régi par la licence CeCILL-C soumise au droit français et
respectant les principes de diffusion des logiciels libres. Vous pouvez
utiliser, modifier et/ou redistribuer ce programme sous les conditions
de la licence CeCILL-C telle que diffusée par le CEA, le CNRS et l'INRIA
sur le site http://www.cecill.info.

En contrepartie de l'accessibilité au code source et des droits de copie,
de modification et de redistribution accordés par cette licence, il n'est
offert aux utilisateurs qu'une garantie limitée. Pour les mêmes raisons,
seule une responsabilité restreinte pèse sur l'auteur du programme, le
titulaire des droits patrimoniaux et les concédants successifs.

A cet égard l'attention de l'utilisateur est attirée sur les risques
associés au chargement, à l'utilisation, à la modification et/ou au
développement et à la reproduction du logiciel par l'utilisateur étant
donné sa spécificité de logiciel libre, qui peut le rendre complexe à
manipuler et qui le réserve donc à des développeurs et des professionnels
avertis possédant des connaissances informatiques approfondies. Les
utilisateurs sont donc invités à charger et tester l'adéquation du
logiciel à leurs besoins dans des conditions permettant d'assurer la
sécurité de leurs systèmes et ou de leurs données et, plus généralement,
à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.

Le fait que vous puissiez accéder à cet en-tête signifie que vous avez
pris connaissance de la licence CeCILL-C, et que vous en avez accepté les
termes.

/licence*/

/**
 * @package Alkanet_Class_Form
 * 
 * @class AlkFormData
 * @brief classe de description d'une donnée de formulaire
 */
class AlkFormData extends AlkObject
{  
  /** nom de l'attribut */
  public $oForm;
  
  /** nom de l'attribut */
  public $name;
  
  /** nom de l'attribut utilisé dans les requests ou dataset */
  public $nameForEval;

  /** valeur de l'attribut */
  public $value;

  /** valeur par défaut */
  public $defaultValue;

  /** valeur extraite de la base */
  public $dbValue;

  /** valeur reçue par request */
  public $requestValue;

  /** vrai si le champ est multilingue */
  public $bMultiLanguage;

  /** type de données SQL associé 0=varchar ou text, 1=int ou number ou float, 2=date */
  public $typeSql;

  /** nom de la fonction appelée pour vérifier le format de la donnée postée */
  protected $fctRequest;

  /** Indique si cette donnée sera présente dans le formulaire */
  protected $isVisible;

  /** Indique si cette donnée sera un champ caché dans le formulaire */
  protected $isHidden;
  
  /** Tableau des controles dont on doit additionner les valeurs pour l'enregistrement */
  protected $tabAdditionalCtrls;
  
  /** nom de l'attribut */
  public $iMode;
  
  /** nom de l'attribut */
  public $label;

  /**
   *  Constructeur par défaut.
   *
   * @param oForm           Formulaire propriétaire de l'attribut
   * @param name            Nom de l'attribut
   * @param value           Valeur de la donnée
   * @param typeSql         Type du valeur SQL
   * @param fctRequest      Nom de la fonction appelée pour vérifier le format de la donnée postée
   * @param bMultiLanguage  Vrai si le champ est multilingue, faux sinon (par défaut)
   * @param default_value   Valeur par défaut (si null c'est value)
   */
  function __construct(AlkHtmlForm $oForm, $name, $value, $typeSql="0", $fctRequest="DefaultTest", $bMultiLanguage=false, $default_value=null)
  {
    parent::__construct();

    $this->oForm = $oForm;
    $this->oForm->addData($this);
    
    $this->name = $name;
    $this->nameForEval = mb_ereg_replace("\[.*\]", "", $this->name);
    $this->value = $value;
    $this->defaultValue = (is_null($default_value) ? $value : $default_value);
    $this->typeSql = $typeSql;
    $this->fctRequest = $fctRequest;
    $this->iMode = 0;
    $this->label = "";
        
    $this->bMultiLanguage = $bMultiLanguage;
    if( $this->bMultiLanguage == true ) {     
      foreach($this->tabLangue as $key => $tabLg) {
        $this->value[$key] = "";
      }
      $this->defaultValue = $this->value;   
    }
    $this->isVisible = true;
    $this->isHidden  = false;
    
    $this->tabAdditionalCtrls = array();
  }
  
  /**
   *  Initialise la donnée en fonction du mode passé
   * @param 
   */
  public function initData($initMode, $oDr, $reqMethod)
  {
    switch ( $initMode ){
      case ALK_INIT_MODE_DATAROW :
        $this->GetData($oDr);
      break;
      
      case ALK_INIT_MODE_REQUEST :
        $this->GetRequest($reqMethod);
      break;
      
      default : break;
    }
  }

  /**
   *  Charqe les données à partir du dataRow.
   *        Prend en charge le multilingue
   *
   * @param oDrData  Référence vers un dataRow
   */
  function GetData(&$oDrData)
  {
    if ( is_null($oDrData) ) return;
    if( $this->bMultiLanguage == true ) {
      foreach($this->tabLangue as $key => $tabLg) {
        $this->value[$key] = $oDrData->GetValueName(strtoupper($this->nameForEval).$tabLg["bdd"], $this->value[$key]);
        $this->dbValue[$key] = $this->value[$key];
      }
    } else {
      $this->value = $oDrData->GetValueName(strtoupper($this->nameForEval), $this->value);
      $this->dbValue = $this->value;
    }
  }

  /**
   *  Charqe les données à partir du dataRow.
   *        Prend en charge le multilingue
   *
   * @param reqMethod Type de request (REQ_POST, REQ_GET, REQ_POST_GET (par défaut), REQ_GET_POST)
   */
  function GetRequest($reqMethod=REQ_POST_GET)
  {
    if( $this->bMultiLanguage == true ) {
      foreach($this->tabLangue as $key => $tabLg) {
        $this->value[$key] = $this->_GetRequest($reqMethod, $this->nameForEval.$tabLg["bdd"], $key);
        $this->requestValue[$key] = $this->value[$key];
      }
    } else {
      $this->value = $this->_GetRequest($reqMethod, $this->nameForEval);
      $this->requestValue = $this->value;
    }    
  }

  /**
   *  Se charge de récupérer la valeur postée de la bonne manière
   *
   * @param reqMethod   Type de request (REQ_POST, REQ_GET, REQ_POST_GET, REQ_GET_POST)
   * @param strVarName
   * @param idLg        Identifiant de la langue courante (-1 par défaut correspondant bMultiLanguage=false)
   * 
   * @return value
   */
  function _GetRequest($reqMethod, $strVarName, $idLg="-1")
  {    
    $strValue = $this->value;
    if( $idLg != "-1" )
      $strValue = $this->value[$idLg];
   
    $strFunction = "";
    switch ( $reqMethod ){
      case REQ_GET :
        $strFunction = "_GET";
      break;
      case REQ_POST :
        $strFunction = "_POST";
      break;
      default :
        $strFunction = "_REQUEST";
      break;
    }
    if ( $strFunction=="" ) 
      return $strValue;
    
    $strFctNotArray = $strFunction; 
    $bArray = false;
    if ( $this->fctRequest==ALK_VERIF_ARRAY 
      || $this->fctRequest==ALK_VERIF_ARRAYINT 
      || $this->fctRequest==ALK_VERIF_ARRAYCHECK 
    ){
      $bArray = true;
      $strFunction .= "array";
    }
      
    if ( $this->fctRequest==ALK_VERIF_ARRAY ){
      $strFunction .= "value";
    }
      
    if ( $this->fctRequest==ALK_VERIF_NUMERIC || $this->fctRequest==ALK_VERIF_ARRAYINT ){
      $strFunction .= "int";
      $strFctNotArray .= "int";
    }
      
    if ( $this->fctRequest==ALK_VERIF_CHECK || $this->fctRequest==ALK_VERIF_ARRAYCHECK ){
      $strFunction .= "check";
      $strFctNotArray .= "check";
    }
      
    if ( $this->fctRequest==ALK_VERIF_DATE ){
      $strFunction .= "date";
      $strFctNotArray .= "date";
    }

    if ( !$bArray ){
      $strRes = call_user_func(array("alkrequest", $strFunction), $strVarName, $strValue);
    }
    else {
      $strIndex = preg_replace("!".$strVarName."\[([^[]+)\]!", "$1", $this->name);
      if ( $strIndex!="" ){
        $strRes = call_user_func(array("alkrequest", $strFunction), $strVarName, $strIndex, $strValue);
      }
      else {
        $strRes = call_user_func(array("alkrequest", $strFctNotArray), $strVarName, $strValue);
      }
    }
    return $strRes;
  }

 
  /**
   *  Retourne la valeur de l'attribut bMultiLanguage
   * @return boolean
   */
  public function getBMultiLanguage()
  {
    return $this->bMultiLanguage; 
  }
  
  /**
   *  Affecte l'attribut bMultiLanguage
   * 
   * @return boolean : affectation réussie
   */
  public function setBMultiLanguage($bMultiLanguage)
  {
    $this->bMultiLanguage = $bMultiLanguage;
    return true;
  }

 
  /**
   *  Retourne la valeur de l'attribut isVisible
   * @return boolean
   */
  public function getIsVisible()
  {
    return $this->isVisible; 
  }
  
  /**
   *  Affecte l'attribut isVisible
   * 
   * @return boolean : affectation réussie
   */
  public function setIsVisible($isVisible)
  {
    $this->isVisible = $isVisible;
    return true;
  }
 
  /**
   *  Retourne la valeur de l'attribut isHidden
   * @return boolean
   */
  public function getIsHidden()
  {
    return $this->isHidden; 
  }
  
  /**
   *  Affecte l'attribut isHidden
   * 
   * @return boolean : affectation réussie
   */
  public function setIsHidden($isHidden)
  {
    $this->isHidden = $isHidden;
    return true;
  }
  
  /**
   *  Retourne la valeur traduite dans la langue courante si donnée en multilingue
   * @return mixed : value ou value[ALK_LG]
   */
  public function getValueLanguage()
  {
    if ( !$this->bMultiLanguage ) return $this->value;
    return $this->value[ALK_LG];
  }
  
  /**
   *  Enregistre un autre controle comme devant fournir sa valeur à l'enregistrement en base
   * @param oOtherData  AlkFormData    Autre donnée
   * @param separator   mixed          Séparateur entre les valeurs (chaine vide par défaut)
   * @param operation   string         Opération PHP à effectuer entre les valeurs parmi {"concat"(default), "add", "sub", "div", "times", "replace"}
   * @return boolean : ajout effectué
   */
  public function addAdditionnalCtrl(AlkFormData $oOtherData, $separator="", $verification="", $operation="concat")
  {
    if ( $this === $oOtherData ) return false;
    $this->tabAdditionalCtrls[] = array(
                                    "data"          => $oOtherData,
                                    "separator"     => $separator,
                                    "verification"  => $verification,
                                    "operation"     => $operation,
                                  );
    return true;
  }
  
  /**
   *  Calcule la valeur du controle en tenant compte des controles additionnels
   * @param strKey          Clé multilingue
   * @return mixed    Valeur du controle tenant compte des controles additionnels
   */
  public function getValueForQuery($strKey="")
  {
    if ( $this->bMultiLanguage ){
      if ( $strKey=="" ) $strKey = ALK_LG;
    }
    $value = ($strKey >=0 && $this->bMultiLanguage ? $this->value[$strKey] : $this->value);
    foreach ( $this->tabAdditionalCtrls as $tabCtrl ){
      $value = $this->getOperationAdditionnal($tabCtrl, $value, $strKey);
    }
    return $value;
  }
  
  /**
   *  Calcule une nouvelle valeur à partir de la définition d'un controle ajouté 
   * @param tabAddingCtrl   Définition du controle ajouté
   * @param value           Valeur de départ
   * @param strKey          Clé multilingue
   * @return mixed    Valeur calculée à partir d'un controle additionnel
   */
  protected function getOperationAdditionnal($tabAddingCtrl, $value, $strKey=""){
    $oData = $tabAddingCtrl["data"];
    $separator = $tabAddingCtrl["separator"];
    $verification = $tabAddingCtrl["verification"];
    $operation = $tabAddingCtrl["operation"];
    $data_value = $oData->getValueForQuery($strKey);
    if ( $verification!="" ){
      if ( @eval("return !(".$verification.");") ){
        return $value;
      }
    }    
    
    switch ( $operation ){
      case "replace" :
        $value = $data_value;
      break;
      case "concat" :
        $value = $value . $separator . $data_value;
      break;
      case "add" :
        if ( $separator=="" ) $separator = 0;
        $value = $value + $data_value + $separator;
      break;
      case "sub" :
        if ( $separator=="" ) $separator = 0;
        $value = $value - $data_value - $separator;
      break;
      case "div" :
        if ( $separator=="" ) $separator = 1;
        $separator = max(1, (float)$separator);
        $value = $value / $data_value / $separator;
      break;
      case "times" :
        if ( $separator=="" ) $separator = 1;
        $value = $value * $data_value * $separator;
      break;
    }

    return $value;
  }
  
  /**
   * Permet de verifier les droits d'un champ (administré par l'application 'droits') sur un mode de travail du formulaire pour un role donné
   * @param table_prefix (string) Préfixe des tables de gestion des droits
   * @param field_table  (string) Nom de la table du champ dans la table [table_prefix]_CHAMP
   * @param role_id      (int)    Identifiant du role pour lequel on cherche les droits
   * @param iMode        (int<=-1) Mode de travail pour lequel on demande les droits
   * @param bToReadMode  (boolean<=true)si true alors le controle est transformé en mode lecture et un champ hidden sera ajouté au formulaire avec la valeur enregistrée
   *                              si false, le controle passe en hidden (setIsHidden(true)) 
   */
  public function setFieldRight($table_prefix, $field_table, $role_id, $iMode=-1, $bToReadMode=true)
  {
    if ( is_null($role_id) ) return;
    if ( $iMode==-1 )
      $iMode = $this->oForm->getProperty("iMode", ALK_FORM_MODE_READ);
    $dsVisibility = AlkQuery::getFieldVisibility($table_prefix, $field_table, $this->nameForEval, $role_id, pow(2, $iMode));
    if ( $drVisibility = $dsVisibility->getRowIter() ){
      $bVisible = ($drVisibility->getValueName("B_VISIBLE")==1);
      $this->label = $drVisibility->getValueName("CHAMP_LABEL");
    }
    else {
      $this->triggerError(__CLASS__."::".__FUNCTION__." : cette méthode ne devrait pas être utilisée".
          " car aucun champ de nom '".$this->nameForEval."' et de table '".$field_table."'".
          " n'est défini dans la table '".$table_prefix."_CHAMP'", E_USER_ERROR);
    }
    if ( !$bVisible ){
      if ( $bToReadMode ){
        $this->iMode = 1;
      }
      else {
        $this->setIsHidden(true);
      }
    }
  }

  /**
   * Vérifie la visibilité de ce controle
   * @return boolean true=visible
   */
  public function checkCtrlVisibility()
  {
    return $this->getIsVisible();
  }
  
  /**
   * Définit l'état d'un controle en terme de lecture/ecriture et de propriété hidden
   * @param AlkHtmlCtrl oCtrl Le controle à paramétrer
   */
  public function checkCtrlState($oCtrl)
  {
    $oCtrl->setMode($this->iMode);
    $oCtrl->setWriteHiddenOnReadOnly($this->iMode==1);
    $oCtrl->setHidden($this->getIsHidden());
  }
  
  /**
   * Après utilisation de la méthode setFieldRight, retourne le label défini pour ce champ dans la gestion des droits
   * @return string (="" si la méthode setFieldRight n'a pas été au préalable utilisée)
   */
  public function getLabel()
  {
    return $this->label;
  }
  
  /**
   * retourne le label défini pour ce champ dans la gestion des droits : recherche sur la base si le label est non déjà récupéré
   * @return string
   */
  public function getLabelFromDatabase($table_prefix, $field_table, $bError=true)
  {
    if ( $this->label!="" )
      return $this->label;
    $dsVisibility = AlkQuery::getFieldVisibility($table_prefix, $field_table, $this->nameForEval);
    if ( $drVisibility = $dsVisibility->getRowIter() ){
      $this->label = $drVisibility->getValueName("CHAMP_LABEL");
    }
    else if ( $bError ){
      $this->triggerError(__CLASS__."::".__FUNCTION__." : cette méthode ne devrait pas être utilisée".
          " car aucun champ de nom '".$this->nameForEval."' et de table '".$field_table."'".
          " n'est défini dans la table '".$table_prefix."_CHAMP'", E_USER_WARNING);
    }
    return $this->label;
  }
}
?>