##############################################################################
#
# Copyright (c) 2002 Ingeniweb SARL
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE
#
##############################################################################

"""
$$
"""
__author__  = ''
__docformat__ = 'restructuredtext'

# Python imports
import os
import Globals
from Acquisition import aq_base
from StringIO import StringIO
from types import StringType, UnicodeType
from new import instancemethod

# Zope imports
from Globals import InitializeClass
from AccessControl import ClassSecurityInfo
from OFS.Image import File
from OFS.SimpleItem import SimpleItem
from ZPublisher.Iterators import filestream_iterator
#from Shared.DC.ZRDB.TM import TM

# CMF imports
from Products.CMFCore.utils import getToolByName

# Archetypes imports
from Products.Archetypes.Storage import StorageLayer
from Products.Archetypes.interfaces.field import IObjectField
from Products.Archetypes.interfaces.base import IBaseUnit
try:
    from Products.Archetypes.Field import Image # Changes since AT1.3.4
except:
    from OFS.Image import Image

# Products imports
from Products.FileSystemStorage.rdf import RDFWriter
from FileUtils import copy_file, move_file, rm_file

class VirtualData:

    __allow_access_to_unprotected_subobjects__ = 1

    def __init__(self, name, instance, path):
        self.name = name
        self.instance = instance
        self.path = path
        self.__name__ = name
        
    def getData(self):
        """Returns data stored on filesystem"""
        
        # Use StringIO for large files
        virtual_file = StringIO()
        value = ''
        
        # Make sure file exists. If not returns empty string
        if not os.path.exists(self.path) or os.path.getsize(self.path) == 0:
            return ''

        # Copy all data in one pass
        try:
            copy_file(self.path, virtual_file)
            virtual_file.seek(0)
            value = virtual_file.getvalue()
        finally:
            virtual_file.close()
        return value

    def __str__(self):
        return str(self.getData())
    
    def __len__(self):
        return len(str(self))
    
    def __getattr__(self, key):
        if key == 'data':
            return self.getData()
    
    read = __str__
    
InitializeClass(VirtualData)

class VirtualBinary(VirtualData):

    def __init__(self, name, instance, path, filename, mimetype, size):
        VirtualData.__init__(self, name, instance, path)
        self.filename = filename
        self.content_type = mimetype
        self.size = size
        
    def __len__(self):
        return self.size
    
    def absolute_url(self):
        url = '%(instance_url)s/fss_get/%(name)s' % {
            'instance_url': self.instance.absolute_url(),
            'name': self.name,
            }
            
        return url
    
    def download(self, request):
        response = request.RESPONSE
        response.setHeader('Content-Disposition',
                           'attachment; filename="%s"' % self.filename)
        response.setHeader('Content-Type', self.content_type)
        response.setHeader('Content-Length', self.size)
        return filestream_iterator(self.path, mode='rb')
    
    def get_size(self):
        return len(self)

    def getContentType(self):
        return self.content_type
        
    def evalCmd(self, cmd_name):
        """Eval command on storage"""
        
        if cmd_name not in dir(self):
            raise AttributeError, 'Unknown attribute: %s' % cmd_name
        
        request = self.REQUEST
        response = request.RESPONSE
        kwargs = dict(request.form)
        kwargs['REQUEST'] = request
        kwargs['RESPONSE'] = response
        cmd = getattr(self, cmd_name)
        return cmd(**kwargs)

InitializeClass(VirtualBinary)

class VirtualFile(VirtualBinary, File):
    
    __allow_access_to_unprotected_subobjects__ = 1
    
    def __init__(self, name, instance, path, filename, mimetype, size):
        VirtualBinary.__init__(self, name, instance, path, filename, mimetype, size)
        
    def __getattr__(self, key):
        if key == 'data':
            return self.getData()
        return File.__getattr__(self, key)
    
InitializeClass(VirtualFile)

class VirtualImage(VirtualBinary, Image):
    
    __allow_access_to_unprotected_subobjects__ = 1
    
    def __init__(self, name, instance, path, filename, mimetype, size, width, height):
        VirtualBinary.__init__(self, name, instance, path, filename, mimetype, size)
        self.width = width
        self.height = height
        
    def __getattr__(self, key):
        if key == 'data':
            return self.getData()
        return Image.__getattr__(self, key)
    
InitializeClass(VirtualImage)


# ####
# Storage info
# ####

class FSSInfo(SimpleItem):
    """FileSystemStorageInfo Base class. Used for string data"""
    
    security = ClassSecurityInfo()
    
    def __init__(self, uid):
        self.update(uid)
    
    security.declarePrivate('update')
    def update(self, uid):
        self.uid = uid
    
    security.declarePrivate('getUID')
    def getUID(self):
        return self.uid
    
    security.declarePrivate('setUID')
    def setUID(self, uid):
        self.uid = uid
    
    security.declarePrivate('getValue')
    def getValue(self, name, instance, path):
        return str(VirtualData(name, instance, path))

    security.declarePrivate('getRDFFieldProperties')
    def getRDFFieldProperties(self, name, instance):
        """Returns RDF field properties list.
        
        Each property is defined in a dictionnary {'id': ...,  'value': ...}
        
        @param name: name of the field
        @param instance: Content using this storage"""
        
        props = (
            {'id': 'dc:title', 'value': instance.title_or_id()},
            {'id': 'dc:description', 'value': instance.Description()},
            {'id': 'dc:language', 'value': instance.Language()},
            {'id': 'dc:creator', 'value': instance.Creator()},
            {'id': 'dc:date', 'value': instance.modified()},
            {'id': 'dc:format', 'value': getattr(self, 'mimetype', 'text/plain')},
            )
        return props
    
    security.declarePrivate('getProperties')
    def getProperties(self):
        """Returns info attributes in a dictionnary"""
        
        props = {}
        props['uid'] = self.uid
        return props
    
    security.declarePrivate('getRDF')
    def getRDF(self, name, instance):
        """Returns RDF dictionnary to inject into RDFWriter
        
        @param name: name of the field
        @param instance: Content using this storage"""
        
        rdf_args = {}
        
        # Get charset
        ptool = getToolByName(instance, 'portal_properties')
        rdf_args['charset'] = ptool.site_properties.default_charset
        
        # Get namespaces
        rdf_args['namespaces'] = (
            {'id': 'xmlns:rdf', 'value': 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'},
            {'id': 'xmlns:dc', 'value': 'http://purl.org/dc/elements/1.1/'},
            {'id': 'xmlns:fss', 'value': 'http://namespace.ingeniweb.com/fss'},) 
        
        # Get field url
        utool = getToolByName(instance, 'portal_url')
        portal_path = utool.getPortalObject().getPhysicalPath()
        portal_path_len = len(portal_path)
        rel_path = '/'.join(instance.getPhysicalPath()[portal_path_len:])
        fss_path = '%s/fss_get/%s' % (rel_path, name)
        rdf_args['field_url'] = fss_path
        
        # Get field properties
        rdf_args['field_props'] = self.getRDFFieldProperties(name, instance)
        return rdf_args
        
InitializeClass(FSSInfo)

class FSSFileInfo(FSSInfo):
    """FileSystemStorageInfo File class. Used for file data"""
    
    security = ClassSecurityInfo()
    
    def __init__(self, uid, title, size, mimetype):
        self.update(uid, title, size, mimetype)
    
    security.declarePrivate('update')
    def update(self, uid, title, size, mimetype):
        FSSInfo.update(self, uid)
        self.title = title
        self.size = size
        self.mimetype = mimetype

    security.declarePrivate('getTitle')
    def getTitle(self):
        return self.title
    
    security.declarePrivate('setTitle')
    def setTitle(self, title):
        self.title = title
    
    security.declarePrivate('getSize')
    def getSize(self):
        return self.size
    
    security.declarePrivate('setSize')
    def setSize(self, size):
        self.size = size
    
    security.declarePrivate('getMimetype')
    def getMimetype(self):
        return self.mimetype
    
    security.declarePrivate('setMimetype')
    def setMimetype(self, mimetype):
        self.mimetype = mimetype
        
    security.declarePrivate('getValue')
    def getValue(self, name, instance, path):
        return VirtualFile(name, instance, path, self.title, self.mimetype, self.size)

    security.declarePrivate('getProperties')
    def getProperties(self):
        """Returns info attributes in a dictionnary"""
        
        props = FSSInfo.getProperties(self)
        props['mimetype'] = self.mimetype
        props['title'] = self.title
        props['size'] = self.size
        return props

    security.declarePrivate('getRDFFieldProperties')
    def getRDFFieldProperties(self, name, instance):
        """Returns RDF field properties list.
        
        Each property is defined in a dictionnary {'id': ...,  'value': ...}
        
        @param name: name of the field
        @param instance: Content using this storage"""
        
        props = FSSInfo.getRDFFieldProperties(self, name, instance) + (
            {'id': 'fss:filename', 'value': self.title},
            {'id': 'fss:size', 'value': self.size},
            )
        return props

InitializeClass(FSSFileInfo)

class FSSImageInfo(FSSFileInfo):
    """FileSystemStorageInfo Image class. Used for image data"""
    
    security = ClassSecurityInfo()
    
    def __init__(self, uid, title, size, mimetype, width, height):
        self.update(uid, title, size, mimetype, width, height)
    
    security.declarePrivate('update')
    def update(self, uid, title, size, mimetype, width, height):
        FSSFileInfo.update(self, uid, title, size, mimetype)
        self.width = width
        self.height = height

    security.declarePrivate('getWidth')
    def getWidth(self):
        return self.width
    
    security.declarePrivate('setWidth')
    def setWidth(self, width):
        self.width = width
        
    security.declarePrivate('getHeight')
    def getHeight(self):
        return self.height
    
    security.declarePrivate('setHeight')
    def setHeight(self, height):
        self.height = height
        
    security.declarePrivate('getValue')
    def getValue(self, name, instance, path):
        return VirtualImage(name, instance, path, self.title, self.mimetype, self.size, self.width, self.height)

    security.declarePrivate('getProperties')
    def getProperties(self):
        """Returns info attributes in a dictionnary"""
        
        props = FSSFileInfo.getProperties(self)
        props['width'] = self.width
        props['height'] = self.height
        return props

    security.declarePrivate('getRDFFieldProperties')
    def getRDFFieldProperties(self, name, instance):
        """Returns RDF field properties list.
        
        Each property is defined in a dictionnary {'id': ...,  'value': ...}
        
        @param name: name of the field
        @param instance: Content using this storage"""
        
        props = FSSFileInfo.getRDFFieldProperties(self, name, instance) + (
            {'id': 'fss:width', 'value': self.width},
            {'id': 'fss:height', 'value': self.height},
            )
        return props

InitializeClass(FSSImageInfo)

# Keep it for compatibility
class FileSystemStorageInfo(FSSFileInfo):
      pass
      
InitializeClass(FileSystemStorageInfo)


# ####
# Storage
# ####

class FileSystemStorage(StorageLayer):
    
    __implements__ = StorageLayer.__implements__
    
    security = ClassSecurityInfo()
    
    security.declarePrivate('getFSSInfoVarname')
    def getFSSInfoVarname(self, name):
        """ """
        
        return '%s_filesystemstorage_info' % name
    
    security.declarePrivate('getFSSInfo')
    def getFSSInfo(self, name, instance, **kwargs):
        """Get fss info"""
        
        info_varname = self.getFSSInfoVarname(name)
        return getattr(aq_base(instance), info_varname, None)
    
    
    security.declarePrivate('delFSSInfo')
    def delFSSInfo(self, name, instance, **kwargs):
        """Delete fss info attribute"""
        
        info_varname = self.getFSSInfoVarname(name)
        delattr(aq_base(instance), info_varname)
    
    security.declarePrivate('setFSSInfo')
    def setFSSInfo(self, name, instance, value, **kwargs):
        """Set new value in fss info"""
        
        info_varname = self.getFSSInfoVarname(name)
        field = self.getField(name, instance, **kwargs)

        # Check types
        uid = instance.UID()
        if isinstance(value, File) or isinstance(value, Image):
            size = value.get_size()
            mimetype = kwargs.get('mimetype', getattr(value, 'content_type', 'application/octet-stream'))
            title = kwargs.get('filename', getattr(value, 'filename', getattr(value, 'title', name)))
            
            if isinstance(value, Image):
                width = value.width
                height = value.height
                if type(title) not in (StringType, UnicodeType):
                    # Make sure title is a string (Fix for image thumbs)
                    title = name
                    
                # Like Image field
                info = FSSImageInfo(uid, title, size, mimetype, width, height)
            else:
                # Like File field
                info = FSSFileInfo(uid, title, size, mimetype)
        else:
            # Other 
            info = FSSInfo(uid)
        
        # Make sure, we have deleted old info
        if hasattr(aq_base(instance), info_varname):
            delattr(instance, info_varname)
      
        # Store in attributes
        setattr(instance, info_varname, info)
    
    security.declarePrivate('getField')
    def getField(self, name, instance, **kwargs):
        """Get field"""
        
        return kwargs.get('field', instance.getField(name))
    
    security.declarePrivate('getBackupPath')
    def getBackupPath(self, name, instance, **kwargs):
        """
        """
        
        path = default_path = os.path.join(Globals.INSTANCE_HOME, 'var')
        
        try:
            fss_tool = getToolByName(instance, 'portal_fss')
            path = fss_tool.getBackupPath()
        except:
            # No tool installed
            path = default_path
        
        return path
    
    security.declarePrivate('getBackupFilename')
    def getBackupFilename(self, name, instance, **kwargs):
        """
        This is the same as filename but it uses ".bak" extension.
        """
        
        return '%s.bak' % self.getFilename(name, instance, **kwargs)
    
    security.declarePrivate('getBackupFilePath')
    def getBackupFilePath(self, name, instance, **kwargs):
        """Get backup path of file
        """
        
        path = self.getBackupPath(name, instance, **kwargs)
        filename = self.getBackupFilename(name, instance, **kwargs)
        
        return os.path.join(path, filename)

    security.declarePrivate('getPath')
    def getPath(self, name, instance, **kwargs):
        """
        """
        
        path = default_path = os.path.join(Globals.INSTANCE_HOME, 'var')
        
        try:
            fss_tool = getToolByName(instance, 'portal_fss')
            path = fss_tool.getStoragePath()
        except:
            # No tool installed
            path = default_path
        
        return path
    
    security.declarePrivate('getFilename')
    def getFilename(self, name, instance, **kwargs):
        """
        """
        
        uid = kwargs.get('uid', instance.UID())
        return '%s_%s' % (uid, name)

    security.declarePrivate('getFilePath')
    def getFilePath(self, name, instance, **kwargs):
        """Get path of file
        """
        
        path = self.getPath(name, instance, **kwargs)
        filename = self.getFilename(name, instance, **kwargs)
        return os.path.join(path, filename)
    
    security.declarePrivate('getRDFFilename')
    def getRDFFilename(self, name, instance, **kwargs):
        """Returns RDF filename
        """
        
        uid = kwargs.get('uid', instance.UID())
        return '%s_%s.rdf' % (uid, name)
    
    
    security.declarePrivate('getRDFFilePath')
    def getRDFFilePath(self, name, instance, **kwargs):
        """Get path of RDF file
        """
        
        path = self.getPath(name, instance, **kwargs)
        filename = self.getRDFFilename(name, instance, **kwargs)
        return os.path.join(path, filename)
    
    security.declarePrivate('get')
    def get(self, name, instance, **kwargs):
        """ """
        
        path = self.getFilePath(name, instance, **kwargs)
        backup_path = self.getBackupFilePath(name, instance, **kwargs)
        info = self.getFSSInfo(name, instance)
        
        if not os.path.exists(path) and os.path.exists(backup_path):
            # Maybe it's an undo
            # Move backup file
            move_file(backup_path, path)
            
            # Create RDF file
            ftool = getToolByName(instance, 'portal_fss')
            if ftool.isRDFEnabled():
                rdf_script = ftool.getRDFScript()
                self.updateRDF(name, instance, rdf_script)
            
        if info is None:
            return ''
        
        value = info.getValue(name, instance, path)

        return value
        
    security.declarePrivate('set')
    def set(self, name, instance, value, **kwargs):
        """ """
        
        # Ignore initialize process
        initializing = kwargs.get('_initializing_', False)
        if initializing:
            return
        
        field = self.getField(name, instance, **kwargs)
        path = self.getFilePath(name, instance)
        
        # Remove acquisition wrappers
        value = aq_base(value)

        # Wrap value
        if IObjectField.isImplementedBy(value):
            new_value = value.getRaw(self.instance)
        if IBaseUnit.isImplementedBy(value):
            new_value = value.getRaw()
        elif isinstance(value, File):
            new_value = value.data
        else:
            new_value = str(value)
        
        # Empty files are not allowed
        #if len(new_value) == 0:
        #    return
        
        # Copy data to filesystem
        virtual_file = StringIO(new_value)
        try:
            copy_file(virtual_file, path)
        finally:
            virtual_file.close()
            
        # Create File System Storage Info
        self.setFSSInfo(name, instance, value, **kwargs)
        
        # Create RDF file
        ftool = getToolByName(instance, 'portal_fss')
        if ftool.isRDFEnabled():
            rdf_script = ftool.getRDFScript()
            self.updateRDF(name, instance, rdf_script)

    security.declarePrivate('unset')
    def unset(self, name, instance, **kwargs):
        """Delete value"""
        
        path = self.getFilePath(name, instance, **kwargs)

        if os.path.exists(path):
            # Move to backup
            backup_path = self.getBackupFilePath(name, instance, **kwargs)
            move_file(path, backup_path)
            
            # Remove RDF
            rdf_path = self.getRDFFilePath(name, instance, **kwargs)
            if os.path.exists(rdf_path):
                os.remove(rdf_path)
            
            if not kwargs.get('del_obj', False):
                # Delete field so delete fss attribute
                self.delFSSInfo(name, instance, **kwargs)

    security.declarePrivate('initializeInstance')
    def initializeInstance(self, instance, item=None, container=None):
        """Initialize new object"""
        pass
    
    security.declarePrivate('cleanupInstance')
    def cleanupInstance(self, instance, item=None, container=None):
        """Delete object"""
        pass
        
    security.declarePrivate('initializeField')
    def initializeField(self, instance, field):
        """Initialize field of object"""

        name = field.getName()
        path = self.getFilePath(name, instance)
        info = self.getFSSInfo(name, instance)
        uid = instance.UID()
        old_uid = uid
        
        if info is not None:
            old_uid = info.getUID()
            
            if uid == old_uid:
                # Update RDF file
                ftool = getToolByName(instance, 'portal_fss')
                if ftool.isRDFEnabled():
                    rdf_script = ftool.getRDFScript()
                    self.updateRDF(name, instance, rdf_script)
        
        if uid != old_uid:
            # Copy/Paste
            old_path = self.getFilePath(name, instance, uid=old_uid)
            
            if os.path.exists(old_path):
                copy_file(old_path, path)
      
            # Change UID in info
            info.setUID(uid)
            
    security.declarePrivate('cleanupField')
    def cleanupField(self, instance, field, **kwargs):
        """Delete field of object"""
        
        is_deleted = not getattr(instance, '_v_cp_refs', False)
        
        if is_deleted:
            name = field.getName()
            kwargs['del_obj'] = True
            self.unset(name, instance, **kwargs)
    
    security.declarePrivate('updateRDF')        
    def updateRDF(self, name, instance, rdf_script='', **kwargs):
        """Update RDF file.
        
        Use this method if FSSInfo is mentionned
        
        @param name: Name of the field
        @param instance: Content using this storage
        @param rdf_script: Script used to generate rdf args
        """

        # Generate RDF
        info = self.getFSSInfo(name, instance)
        default_rdf = info.getRDF(name, instance)
        rdf_args = None
        if rdf_script:
            func = getattr(instance, rdf_script, None)
            if func is not None:
                rdf_args = func(name=name, instance=instance, properties=info.getProperties(), default_rdf=default_rdf)
        
        if rdf_args is None:
            rdf_args = default_rdf
        writer = RDFWriter(**rdf_args)
        rdf_text = writer.getRDF()
        
        # Copy RDF to filesystem
        rdf_path = self.getRDFFilePath(name, instance)
        rdf_file = StringIO(rdf_text)
        try:
            copy_file(rdf_file, rdf_path)
        finally:
            rdf_file.close()
        