<?php
# XMLFile -- Initial version June 1, 2000
# May 29, 2002
#  Fixed to work with later versions of PHP that have deprecated the call-time
#  pass by reference syntax.  This may break versions of PHP older than 3, so
#  if you are using this on those versions, go with an earlier version.
#  Also added some more comments on how to use it to create XML files.  Some
#  people (a surprising number of them, in fact) were having difficulty
#  understanding the recursive nature of XML files, and that each tag has
#  subtags, each of which can have subtags, each of which....
# July 12, 2001
#  Fixed an oops in the find_subtags_by_name function.
# July 11, 2001
#  Incorporated Kevin Howe's read_xml_string function.
#  Incorporated Mike Konkov's find_subtags_by_name function (with some changes).
#  Fixed an errant documentation comment (open instead of fopen -- oops).
#
# September 29, 2000
# by Chris Monson -- e408345b17be3ce90059d01d96be0599@orangatango.com This PHP module is licensed under the GNU LGPL (www.gnu.org)
# Please become familiar with this license before sending this thing all over
# the place!  I would like to have any changes submitted back to me, and this
# comment must be included with any distribution of this component.
#
# The purpose of this module was to provide a simple interface for dealing with
# small to medium sized XML files.  It is probably not suitable for very large
# files since it reads the entire file into a structure in memory.  For very
# large files, the XML parsing functions should be used more or less directly
# so that pieces of the file can be dealt with as they are read in.
#
# The basic idea is this: Read the XML file into an internal tree structure
# that is easy to traverse.  This module also allows you to create such a
# structure in memory and then write it out to disk.  The output is formatted
# in a nice, readable way, using whitespace to delimit tag containment, etc.
# It makes it very easy to write nice-looking XML files.
#
# I have included some usage comments.  They are almost certainly incomplete.
# If you get stumped, first use the source, then email me.
#
#------------------------------------------------------------------------------
### USAGE ###
#------------------------------------------------------------------------------
# Reading an XML file:
#------------------------------------------------------------------------------
#
# $xml = new XMLFile();
# $fh = fopen ('myxmlfile.xml', 'r');
# $xml->read_file_handle ($fh);
# close ($fh);
#
# Now the tags can be accessed via the root node of $xml:
#
# $root = &$xml->roottag;
# $tagname = $root->name;
# $tagdata = $root->cdata;
#
# Note that a tag is of the following form:
# <NAME attribute = value>CDATA</NAME>
# Each tag contains an attributes array, which is associative by nature.  In
# other words, you would access the value of "attribute" as follows:
#
# $value = $root->attributes['attribute'];
#
# Also, each tag has a 'tags' proprerty, which is an ordered array (integer
# indices, not associative!) which has the tags that were contained within
# this tag in their order of appearance.  The reason that this is not
# associative is that there can be multiple tags of the same name.  There is
# nothing in the XML spec (barring a DTD) that declares the uniqueness of tag
# names.  For example:
#
# <OUTER>
#     <INNER>CDATA</INNER>
#     <INNER name = "hello"/>
# </OUTER>
#
# In the above example, the outer tag would have a tags array that has two
# entries, each of which has a tag name of "INNER".  The one with CDATA wrapped
# inside would be at index 0, and the other would be at index 1.
#
# Once you have finished with the XMLFile object, you need to call the cleanup
# method.  If you don't, you will get a memory leak, since PHP is reference
# counted and each element in the tree refers to its parent.  'cleanup' simply
# traverses the tree and disconnects the parent pointers.  The PHP cleans up
# everything else.
#
# $xml->cleanup();
#
# Note that you can change the elements, delete tags, and do other things
# to the tree once it has been read from the file.  After it has been changed,
# you can write it back out and the file will reflect your changes.
#------------------------------------------------------------------------------
# Writing a new file:
#
# $xml = new XMLFile();
# $xml->create_root(); # necessary -- no root is created until requested
# $xml->roottag->name = 'ROOT';
# $xml->roottag->add_subtag ('INNER', array());
# $innertag = &$xml->roottag->curtag;
# $innertag->add_subtag ('REALLYINNER', array());
# # Or, you can do this:
# $xml->roottag->curtag->add_subtag ('INNER2', array());
# # The point is that each tag can have subtags.  The most recently added
# # subtag is always the curtag of its parent.
# $xml->roottag->add_subtag ('INNER', array ('name' => 'value'));
# $xml->roottag->curtag->cdata = "Hello!"; # curtag is the most recent addition
# $fh = fopen ('myxmlfile.xml', 'w');
# $xml->write_file_handle ($fh);
# close ($fh);
#
# The file will look like this: (no space between ? and >)
#
# <?xml version = "1.0" encoding = "UTF-8" ? >
# <ROOT>
#     <INNER>
#           <REALLYINNER/>
#           <INNER2/>
#     </INNER>
#     <INNER name = "value">Hello!</INNER>
# </ROOT>
#
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------
#

class XMLTag
{
    var $cdata;
    var $attributes;
    var $name;
    var $tags;
    var $parent;

    var $curtag;

    function XMLTag (&$parent) {
        $this->_init();
        if (is_object ($parent)) {
            $this->parent =& $parent;
        }
    }

    function _init() {
        $this->attributes = array();
        $this->cdata = '';
        $this->name = '';
        $this->tags = array();
	unset ($this->parent);
	unset ($this->curtag);
    }

    function &add_subtag($name, $attributes = 0) {
        $tag =& new XMLTag ($this);
        $tag->set_name ($name);
        if  (is_array ($attributes)) {
            $tag->set_attributes ($attributes);
        }
	// print count ($this->tags) .'  ';
        $this->tags[] =& $tag;
        $this->curtag =& $tag;
	//print $this->name ."<br>\n";
	//print_r ($this);
	//print "  In ".$this->name." (id='".
	//      $this->get_attribute('id')."') ".
	//      $tag->name." added : ".count($this->tags)."\n";
	return $this->curtag ;
    }

    function index_of_subtag_by_name ($name) {
	$name = strtolower ($name);
        $ret = -1;
        for ( $i = 0; $i < $this->num_subtags(); $i++ ) {
            if ( $this->tags[$i]->name == $name ) {
                $ret = $i; break;
            }
        }
        return $ret;
    }

    function index_of_subtag_by_attr ($attr, $value) {
	$attr = strtolower ($attr);
        $ret = -1;
        for ( $i = 0; $i < $this->num_subtags(); $i++ ) {
            if ( $this->tags[$i]->get_attribute ($attr) == $value ) {
                $ret = $i; break;
            }
        }
        return $ret;
    }

    function find_subtags_by_name ($name) {
	$name = strtolower ($name);
        $found = false;
        for ( $i = 0; $i < $this->num_subtags(); $i++ ) {
            if ( $this->tags[$i]->name == $name ) {
                $found = true;
                $array2return[] =& $this->tags[$i];
            }
        }
        if($found) { return $array2return; }
        else { return false; }
    }

    function &find_nth_subtag_by_name ($name, $number) {
	$a = $this->find_subtags_by_name ($name);
	if ( $a and count ($a) > $number ) {
	    $ret = &$a[$number];
	} else { $ret = false; }
	return $ret;
    }

    function &find_first_subtag_by_name ($name) {
	return $this->find_nth_subtag_by_name ($name, 0);
    }

    function &find_tag_by_path ($path) {
	$pelems = explode ('.', strtolower ($path));
	$tag = &$this;
	foreach ( $pelems as $el ) {
	    $sels = explode ('[', $el);
	    if ( $sels[1] ) {
		$nb = substr ($sels[1], 0, -1);
		$el = $sels[0];
	    } else {
		$nb = 0;
	    }
	    $tag = &$tag->find_nth_subtag_by_name ($el, $nb);
	    if ( ! $tag ) { break; }
	}
	return $tag;
    }

    function clear_subtags() {
        // Traverse the structure, removing the parent pointers
        for ( $i = 0 ; $i < count ($this->tags) ; $i++ ) {
            $this->tags[$i]->clear_subtags();
            unset ($this->tags[$i]->parent);
        }
        // Clear the tags array
        $this->tags = array();
        $this->curtag = &$this;
    }

    function remove_subtag ($index) {
        if (is_object($this->tags[$index])) {
	    $this->tags[$index]->clear_subtags();
            unset ($this->tags[$index]->parent);
            unset ($this->tags[$index]);
        }
    }

    function num_subtags() {
	if ( $this->tags[0]->name == '_text_' ) { return 0; }
        return count ($this->tags);
    }

    function add_attribute ($name, $val) {
	$name = strtolower ($name);
        $this->attributes[$name] = $val;
    }

    function clear_attributes() {
        $this->attributes = array();
    }

    function set_name ($name) {
        $this->name = strtolower ($name);
    }

    function set_attributes ($attributes) {
        $this->attributes = array();
	if ( is_array($attributes) ) {
	    foreach ( $attributes as $k => $val) {
		$this->add_attribute ($k, $val);
	    }
	}
    }

    function set_an_attribute ($ident, $value) {
	$ident = strtolower ($ident);
	$this->attributes[$ident] = $value;
    }

    function get_attribute ($ident) {
	$ident = strtolower ($ident);
	return $this->attributes[$ident];
    }

    function add_cdata ($data) {
//	print "==".$data."##\n";
	if ($data != '' && $data != "\n") {
            $text =& $this->add_subtag ('_text_', array());
	    $text->cdata = $data;
//	    $this->cdata = $data;
//	    print "===".$text->cdata."###<br>\n";
	}
    }

    function set_cdata_from_text ($text) {
	$this->clear_cdata();
	$pelems = explode ("\n", $text);
	$n = count ($pelems) - 1;
	for ( $i = 0 ; $i <= $n ; $i++ ) {
	    $this->add_cdata (trim ($pelems[$i]));
	    if ( $i != $n ) {
		$this->add_subtag ('br', array());
	    }
	}
    }

    function get_cdata () {
        // Traverse the structure, concatening text pieces with <CR> for
	// <br/> tag
	$txt = ''; $prec = 0 ;
        for ( $i = 0 ; $i < count ($this->tags) ; $i++ ) {
            if ( $this->tags[$i]->name == '_text_' ) {
//		if ( $prec == 0 ) {
		    $txt .= $this->tags[$i]->cdata;
//		} else {
//		    $txt .= ' '.$this->tags[$i]->cdata;
//		}
		$prec = 1;
	    } elseif ( $this->tags[$i]->name == 'br' ) {
		$txt .= "\n"; $prec = 1 ;
	    } else { $prec = 0; }
            unset ($this->tags[$i]->parent);
	}
//	$txt = str_replace ("\n ", "\n", $txt);
	return $txt;
    }

   function clear_cdata() {
	if ( $this->tags[0]->name == '_text_' ) {
	    $this->clear_subtags();
	}
        $this->cdata = "";
    }

    function write_file_handle ($fh, $prepend_str = '') {
	// write only valids nodes with information
	if ( ! isset ($this->attributes) || ! isset ($this->tags) ) {
	    return;
	}
//	if ( count ($this->attributes) == 0 &&
//	     count ($this->tags) == 0 &&
//	     $this->cdata == '' ) {  return; }
        // build the attribute string
        $attrs = array();
        $attr_str = '';
        foreach ($this->attributes as $key => $val) {
            $attrs[] = "$key=\"$val\"";
        }
        if ($attrs) {
            $attr_str = join (" ", $attrs);
        }
        // Write out the start element
	$name = $this->name;
        $tagstr = "$prepend_str<{$name}";
        if ($attr_str) {
            $tagstr .= " $attr_str";
        }

        // If there are subtags and no data (only whitespace),
        // then go ahead and add a carriage
        // return.  Otherwise the tag should be of this form:
        // <tag>val</tag>
        // If there are no subtags and no data, then the tag should be
        // closed: <tag attrib = "val"/>
        $numtags = $this->num_subtags();
        $trimmeddata = trim ($this->get_cdata());
        if ( $numtags && ($trimmeddata == "") ) {
            $tagstr .= ">\n";
        } elseif ( !$numtags && ($trimmeddata == "") ) {
            $tagstr .= "/>\n";
        } else {
            $tagstr .= ">";
        }
        fwrite ($fh, $tagstr);
        // Write out the data if it is not purely whitespace
        if ( $trimmeddata != "" ) {
	    $trimmeddata = str_replace ('&', '&amp;', $trimmeddata);
	    $trimmeddata = str_replace ("\n", "<br />\n", $trimmeddata);
            fwrite ($fh, $trimmeddata);
        } else {
	    // Write out each subtag
	    foreach ( $this->tags as $t ) {
		$t->write_file_handle ($fh, "$prepend_str  ");
	    }
	}
        // Write out the end element if necessary
        if ($numtags || ($trimmeddata != "")) {
            $tagstr = "</{$name}>\n";
            if ($numtags) {
                $tagstr = "$prepend_str$tagstr";
            }
            fwrite ($fh, $tagstr);
        }
    }

    function show ($prepend_str='', $format='html') {
	if ( $format == 'html') {
	    $sp = '&nbsp;'; $lt = '&lt;'; $gt = '&gt;'; $br = "<br>\n";
	} else {
	    $sp = ' '; $lt = '<'; $gt = '>'; $br = "\n";
	}

        // build the attribute string
        $attrs = array();
        $attr_str = '';
        foreach ($this->attributes as $key => $val) {
            $attrs[] = "$key=\"$val\"";
        }
        if ($attrs) {
            $attr_str = join (" ", $attrs);
        }
        // Write out the start element
	$name = $this->name;
        $tagstr = "$prepend_str$lt$name";
        if ($attr_str) {
            $tagstr .= " $attr_str";
        }

        // If there are subtags and no data (only whitespace),
        // then go ahead and add a carriage
        // return.  Otherwise the tag should be of this form:
        // <tag>val</tag>
        // If there are no subtags and no data, then the tag should be
        // closed: <tag attrib = "val"/>
        $numtags = $this->num_subtags();
        $trimmeddata = trim ($this->get_cdata());
        if ( $trimmeddata == "" ) {
	    if ( !$numtags ) {
		$tagstr .= "/$gt$br";
	    } else {
		$tagstr .= "$gt$br";
	    }
	} else {
	    $tagstr .= "$gt";
	}
        print $tagstr;
        // Write out the data if it is not purely whitespace
        if ( $trimmeddata != "" ) {
	    $trimmeddata = str_replace ('&', '&amp;', $trimmeddata);
	    $trimmeddata = str_replace ("\n", "<br />\n", $trimmeddata);
            print $trimmeddata;
        } else {
	    // Write out each subtag
	    foreach ( $this->tags as $t ) {
		$t->show ("$prepend_str$sp$sp", $format);
	    }
	}
        // Write out the end element if necessary
        if ( $trimmeddata != "" ) {
            $tagstr = "$lt/{$name}$gt$br";
            print $tagstr;
	} else {
            if ($numtags) {
                $tagstr = "$prepend_str$lt/{$name}$gt$br\n";
                print $tagstr;
	    }
        }
    }
}

class XMLFile
{
    var $parser;
    var $roottag;
    var $curtag;
    var $encoding = 'iso-8859-1';

   function XMLFile() {
        $this->init();
    }

    // Until there is a suitable destructor mechanism, this needs to be
    // called when the file is no longer needed.  This calls the clear_subtags
    // method of the root node, which eliminates all circular references
    // in the xml tree.
    function cleanup() {
        if (is_object ($this->roottag)) {
            $this->roottag->clear_subtags();
        }
    }

    function init() {
        $this->roottag = 0;
        $this->curtag = $this->roottag;
    }

    function create_root() {
	$null = 0;
        $this->roottag =& new XMLTag ($null);
        $this->curtag =& $this->roottag;
    }

    // read_xml_string
    // Same as read_file_handle, but you pass it a string.  Note that
    // depending on the size of the XML, this could be rather memory
    // intensive.
    // Contributed July 06, 2001 by Kevin Howe
    function read_xml_string ($str, $encoding='') {
        $this->init();
	if ( $encoding ) { $this->encoding = $encoding; }
        $this->parser = xml_parser_create ($this->encoding);
        xml_set_object ($this->parser, $this);
        xml_set_element_handler ($this->parser, "_tag_open", "_tag_close");
        xml_set_character_data_handler ($this->parser, "_cdata");
        xml_parse ($this->parser, $str);
        xml_parser_free ($this->parser);
    }

    function read_file ($file, $encoding='') {
	$fh = fopen ($file, 'r');
	$this->read_file_handle ($fh, $encoding);
	fclose ($fh);
    }

    function read_file_handle ($fh, $encoding='') {
        $this->init();
	if ( $encoding ) { $this->encoding = $encoding; }
        $this->parser = xml_parser_create ($this->encoding);
        xml_set_object ($this->parser, $this);
        xml_set_element_handler ($this->parser, "_tag_open", "_tag_close");
        xml_set_character_data_handler ($this->parser, "_cdata");

        while ($data = fread ($fh, 4096)) {
            if (!xml_parse ($this->parser, $data, feof ($fh))) {
                die(sprintf("XML error: %s at line %d",
                    xml_error_string(xml_get_error_code($this->parser)),
                    xml_get_current_line_number($this->parser)));
            }
        }
        xml_parser_free ($this->parser);
    }

    function write_file ($file, $write_header=1) {
	$fh = fopen ($file, 'w');
	$this->write_file_handle ($fh, $write_header);
	fclose ($fh);
    }

    function show ($format='html') {
	$this->roottag->show('', $format);
    }

    /** Return the value contained in the element at the given path */
    function getValueByPath($path) {
        $pelems = explode ('@', $path);
        if ( $pelems[0] ) {
            $cur =& $this->roottag->find_tag_by_path($pelems[0]);
        } else { $cur =& $this->roottag; }
        if ( $cur) {
            if ( $pelems[1] ) {
                $ret = $cur->get_attribute($pelems[1]);
            } else {
                $ret = $cur->get_cdata();
            }
        } else { $ret = ''; }
        return stripslashes ($ret);
    }
        

    function write_file_handle ($fh, $write_header=1) {
        if ($write_header) {
            fwrite ($fh,
	            "<?xml version='1.0' encoding='$this->encoding'?>\n");
        }
        // Start at the root and write out all of the tags
        $this->roottag->write_file_handle ($fh);
    }

    // Parser callback functions

    function _tag_open ($parser, $tag, $attributes) {
        //print "tag_open: $parser, $tag, $attributes<br>\n";
        // If the current tag is not set, then we are at the root
        if ( !$this->curtag) {
	    //print "create root\n";
	    $null=0;
            $this->roottag =& new XMLTag ($null);
            $this->roottag->set_name ($tag);
            $this->roottag->set_attributes ($attributes);
	    $this->curtag = &$this->roottag;
        }
        else { // otherwise, add it to the tag list and move curtag
 	    //print "add subtag $tag to ".$this->curtag->name."\n";
            $this->curtag =& $this->curtag->add_subtag ($tag, $attributes);
	    //print "new curtag: ".$this->curtag->name."\n";
        }
    }

    function _tag_close ($parser, $tag) {
        // Move the current pointer up a level
	$prec = $this->curtag->name;
        $this->curtag =& $this->curtag->parent;
	//print "End-Tag $prec, new curtag: ".$this->curtag->name."\n";
    }

    function _cdata ($parser, $data) {
//	$data = str_replace ('{br}', "\n", $data);
        $data = trim ($data);
        $this->curtag->add_cdata ($data);
    }
}
/*
$null = null;
$root = new XMLTag ($null);
$root->set_name ('root');
$root->set_attributes (array ('id'=>'root'));
$cur =& $root->add_subtag ('prem', array ('id'=>'prem'));
// print "Current: $cur->name - parent: {$cur->parent->name}\n";
$cur =& $root->add_subtag ('sec', array ('id'=>'sec'));
// print "Current: $cur->name - parent: {$cur->parent->name}\n";
$cur =& $cur->add_subtag ('three', array ('id'=>'three'));
// print "Current: $cur->name - parent: {$cur->parent->name}\n";
print $cur->parent->num_subtags().'\n';
$root->show ('', 'html');
print "<br>\n";
$cur =& $root->find_tag_by_path ('sec.three');
print $cur->name.' '.$cur->get_attribute ('id').'<br>\n';
 */
/*
$xml = new XMLFile();
$xml->read_file ('../../extra/xml-data/lettres/20041228_voeux.xml');
//print $xml->roottag->name."\n";
$xml->show('html');
print $xml->getValueByPath("@id")."<br>";
print $xml->getValueByPath("dest@carnet");
*/
?>