<?php
/*
 * Package:	xml/collection
 * Version:	0.01
 * Date:	20020907
 * Author:	Daniel Deveaux
 * Licence:	GPL (see the file 'COPYING' and 'GNU_LICENSE in distribution)
 *
 *  This package implements a simple tuples array that is saved as a XML
 *  file. At creation, the xml/collection is connected with an xml file and
 *  an object view is loaded in memory. Methods allow to modifiy, display in
 *  html and PDF formats and save the collection.
 *  This implementation loads all the collection in memory, it is suitable
 *  for small collections as little address-books; for larger collections
 *  it is preferable to use true data bases.
 */
    require_once "xml/xmlfile.php";
    require_once 'Text/MiniWiki.php';

/** convert an ansi date image to standard image yyyy/mm/dd
 */
function ansi_to_date ($ansid, $mode='fr') {
    if ( $mode == 'fr' ) {
        $ret = implode ('/', array (substr ($ansid, 6), substr ($ansid, 4, 2),
                                    substr ($ansid, 0, 4)));
    } else {
        $ret = implode ('/', array (substr ($ansid, 0, 4), substr ($ansid, 4, 2),
                                    substr ($ansid, 6)));
    }
    return $ret;
}

/** convert an ansi hour image to standard image nnhss
 */
function ansi_to_hour ($ansih) {
    $ret = implode ('h', array (substr ($ansih, 0, 2), substr ($ansih, 2)));
    return $ret;
}

/** Remove all character other than a-zA-Z0-9_ */
function normalizeName($text) {
    $text = strtolower($text);
    $text = preg_replace ('/[]/', 'a', $text);
    $text = preg_replace ('/[]/', 'e', $text);
    $text = preg_replace ('/[]/', 'i', $text);
    $text = preg_replace ('/[]/', 'o', $text);
    $text = preg_replace ('/[]/', 'u', $text);
    $text = preg_replace ('/[]/', 'A', $text);
    $text = preg_replace ('/[]/', 'E', $text);
    $text = preg_replace ('/[]/', 'I', $text);
    $text = preg_replace ('/[]/', 'O', $text);
    $text = preg_replace ('/[]/', 'U', $text);
    $text = preg_replace("/[^a-zA-Z0-9_]/", "_", $text);
    $text = preg_replace("/_+/", '_', $text);
    $text = trim($text, '_');
    return $text;
}

/** XML Collection template
 *  This class stores a collection template and provides methods to:
 *   - control elements structure,
 *   - generate a form for data capture
 *   - generate arrays header and footer for HTML and PDF displais
 */
class XmlColTemplate {
    /** instance variables						*/
    var $name = '';	// current element name
    var $index;		// ordered array of items names
    var $idcons = '';	// guide string to construct element ID
    var $labels;	// table of labels for data capture form
    var $alabels;	// table of labels for displayed arrays
    var $chps;		// table des descripteurs de format
    var $needed;	// table des champs requis
    var $ronly;		// table des champs read only
    var $hidden;	// table des champs cachs
    var $extref;	// table des champs rfrence
    var $curdir;        // directory that contains the actual collection
    var $mode;		// selected user mode
    var $fields_number;	// number of tags in an element

    /** template constructor
     */
    function &XmlColTemplate(&$noderef, $curpath, $mode='') {
	$this->ref = &$noderef;
	$this->name = $noderef->name;
	$this->mode = $mode;
	$this->curdir = $curpath;
	// "element name: '$this->name'<br>\n";
	$this->idcons = $noderef->attributes['idcons'];
	// print "id constructor: '$this->idcons'<br>\n";
	$this->fields_number = count($noderef->tags);
	for ($i = 0 ; $i < $this->fields_number ; $i++) {
	    $chpn = $noderef->tags[$i]->name;
	    $this->index[$i] = $chpn;
	    $this->extref[$chpn] = $noderef->tags[$i]->get_attribute('ref');
	    $desc = $noderef->tags[$i]->get_cdata();
	    $labs = explode('/', $desc);
	    $lib = $labs[0];
	    if ( $lib{0} == '*') {
		$this->needed[$chpn] = true;
		$this->labels[$chpn] = substr($lib, 1);
	    } else {
		$this->needed[$chpn] = false;
		$this->labels[$chpn] = $lib;
	    }
	    $this->ronly[$chpn] = false; $this->hidden[$chpn] = false;
	    if ( $labs[3] ) {
		if ( $mode != '') {
		    $pos = strpos($labs[3], $mode);
		    if ( $pos === false ) { $this->ronly[$chpn] = true; }
		} else { $this->ronly[$chpn] = true; }
	    } else {
		if ( $mode ) { $this->ronly[$chpn] = true; }
	    }
	    if ( $labs[4] && $this->ronly[$chpn] ) {
		if ( $mode != '') {
		    $pos = strpos ($labs[4], $mode);
		    if ( $pos === false ) { $this->hidden[$chpn] = true; }
		} else { $this->hidden[$chpn] = true; }
	    } else {
		if ( $mode && $this->ronly[$chpn]) {
		    $this->hidden[$chpn] = true;
		}
	    }
//	    print (" ====&gt; ".$this->labels[$chpn]." --&gt; ".
//	           $this->ronly[$chpn]" -- ".$this->hidden[$chpn]. "<br/>\n");
	    $lib = $this->labels[$chpn];
	    $this->labels[$chpn] = str_replace('_', '&nbsp;', $lib);
	    if ($labs[1]) { $this->alabels[$chpn] = $labs[1]; }
	    else { $this->alabels[$chpn] = $this->labels[$chpn]; }
	    $this->chps[$chpn] = $labs[2];
//	    print "&nbsp;&nbsp;". $this->index[$i] ." : ".
//	         $this->labels[$chpn] ." - ". $this->alabels[$chpn] ." - ".
//	         $this->chps[$chpn] ;
//	    if ($this->needed[$chpn]) { print" - requis"; }
//	    if ($this->ronly[$chpn]) { print" - readonly"; }
//	    print "<br>\n";
	}
    }

    /** Get the index of a specified field
     */
    function get_field_index($fname) {
        for ($i = 0; $i < $this->fields_number; $i++) {
            if ( $this->index[$i] == $fname ) { break; }
        }
        return $i;
    }
    
    /** Compose the head of the HTML presentation table
     */
    function thead($width="100%") {
	$txt = "<p>&nbsp;</p>\n";
	$txt .= "<table border='1' align='center' width='".$width."'>\n";
	$txt .= " <tr>\n";
	foreach ($this->index as $key) {
	    $txt .= "  <th>". $this->alabels[$key] ."</th>\n";
	}
	$txt .= " </tr>\n";
	return $txt;
    }

    /** Compose the HTML form for data input
     */
    function form($file='', $elt=null, $vals=null, $pre='') {
	$txt = '';
	if ( $pre ) {
	    $txt .= "<input type='hidden' name='pre' value='$pre'>\n";
	}
	$txt .= "<input type='hidden' name='cfile' value='".$file."'>\n";
	if ($elt != null) {
	    $txt .= "&nbsp;lment &nbsp;<input readonly type='text' ".
	            "name='key' value='". $elt->key .
		    "' style='background-color: #ddd; width: 15em;'>\n";
	}
	$txt .= "  <table style='background-color: #ddd;'>\n";
//	if ($elt == null) {
//	    $txt .= "   <tr>\n    <td width='40%' class='left'>".
//	            "Cl de l'lment ('$this->idcons')</td>\n";
//	    $txt .= "    <td width='60%' class='left'>";
//	    $txt .= "<input type='text' size='20' name='newkey'>";
//	    $txt .= "</td>\n   </tr>\n";
//	}
	$i = 0;
	foreach ($this->index as $key) {
	    $name = ( $pre ) ? $pre."item$i" : "item$i";
            $valitem = '';
            if ( $vals && isset($vals["item0"]) ) {
                $valitem = $vals["item$i"];
            } else {
                if ( $elt ) { $valitem = $elt->valueAt($key) ; }
            }
	    if ( !$this->hidden[$key] ) {
		$txt .= "   <tr>\n    <td width='40%' class='left'>".
			$this->labels[$key];
		if ( $this->needed[$key] ) {
		    $txt .= " <samp style='color: red;'>*</samp>";
		}
		$txt .= "</td>\n    <td width='60%' class='left'>";
		$pos = strpos ($this->chps[$key], ',');
		if ( $this->ronly[$key] ){
		    $ro = "readonly style='background-color: #999;' ";
		} else {
		    $ro = '';
		}
	    }
	    if ( $this->hidden[$key] ){
		if ($elt != null) {
		   $txt .= "<input type='hidden' name='$name' ".
	                   "value='$valitem'>\n";
		}
            } elseif ( strncmp ($this->chps[$key], "sel,", 4) == 0) {
		$values = explode (",", $this->chps[$key]);
		$txt .= "<select $ro name='$name' style='width: 15em'>\n";
		foreach ($values as $val) {
		    if ( $val == "sel" ) { continue; }
		    $txt .= "  <option value='$val'";
		    if ( $elt != null || $valitem ) {
			$a = $valitem;
			if ( $a == $val ) {
			    $txt .= " selected";
			}
		    }
		    $txt .= ">$val</option>\n";
		}
		$txt .= "</select>\n";
	    } elseif ( !($pos === false) ) { // textarea
		list ($c, $l) = explode(',', $this->chps[$key]);
		$txt .= "<textarea $ro cols='$c' rows='$l' name='$name'>";
		if ($elt != null || $valitem ) {
		   $txt .= $valitem;
		}
		$txt .= "</textarea>";
            } elseif ($this->chps[$key] == 'on') {
	        $txt .= "<input $ro type='checkbox' name='$name'";
		if ( $elt != null || $valitem ) {
		    $txt .= $valitem ? " checked" : "";
		}
		$txt .= " value='oui'>";
	    } elseif ( $this->extref[$key] ) {
		$collec = new XmlCollection("$this->curdir/".
		                             $this->extref[$key].".xml");
		$optls = $collec->sorted_ids();
		$txt .= "<select $ro name='$name' style='width: 15em'>\n".
		        "  <option value=''";
		if ( $elt == null || $valitem ) {
		    $txt .= " selected";
		}
	        $txt .= ">&lt;--choisir--&gt;</option>\n";
		foreach ($optls as $val) {
		    $txt .= "  <option value='$val'";
		    if ( $elt != null ) {
			$a = $elt->attributesAt($key);
			if ( $a['ref'] == $val ) {
			    $txt .= " selected";
			}
		    }
		    $txt .= ">$val</option>\n";
		}
		$txt .= "</select>\n";
//		        "&nbsp; et &nbsp;".
//			"<input type='button' name='aditem$i' ".
//			"value='Nouveau' onclick=\"collHandle(".
//			"'../intra/$file','".($num-1)."','mod')\">\n";
	    } else {
	        $txt .= "<input $ro type='text' size='". $this->chps[$key] .
		        "' name='$name'";
		if ($elt != null || $valitem ) {
		    $txt .= " value=\"$valitem\"";
		}
		$txt .= ">";
	    }
	    if ( !$this->hidden[$key] ) {
		$txt .= "</td>\n   </tr>\n";
	    }
	    $i++;
	}
	$txt .= "  </table>\n";
	return $txt;
    }

}

/** XML Collection element
 *  This class stores a collection element and provides methods to:
 *   - get and set items values,
 *   - modify and display the element
 *   - generate arrays line for HTML and PDF displais
 */
class XmlColElement {
    /** instance variables						*/
    var $ref;			// reference of current element
    var $template;		// associated template
    var $key = '';		// current element name
    var $timestamp = '';        // extenal time stamp to compare to current date

    /** Element constructor
     */
    function &XmlColElement(&$noderef, &$template, $newkey='',
                             $items=0) {
	$this->ref =& $noderef;
	$this->template = $template;
	if ( $newkey ) {
	    // print "New items size: ". count ($items) ."<br>\n";
	    $this->key = $newkey;
	    // print "Element '". $this->key ." ". $this->ref->name ."':<br>\n";
	    $this->create ($items);
	} else {
	    $this->key = $noderef->get_attribute('id');
	    //print "Element '". $this->key ."':<br>\n";
	    //for ($i = 0 ; $i < count ($noderef->tags) ; $i++) {
	    //	$chpn = $noderef->tags[$i]->name;
	    //	print "&nbsp;&nbsp;$chpn: ". $this->valueAt ($chpn) ."<br>\n";
	    //}
	}
        $tstamp = $this->template->curdir.'/timestamp';
        if ( file_exists($tstamp) ) {
            $this->timestamp = date("Ymd", filemtime($tstamp));
        }
    }

    /** Create a new element from its items array
     */
    function create ($items) {
	// print count ($items) . "<br>\n";
	for ($i = 0 ; $i < count($this->template->index) ; $i++) {
	    $key = $this->template->index[$i];
	    $new_value = $items[$i];
	    $new =& $this->ref->add_subtag($key);
	    if ( $this->template->extref[$key] ) {
		$new->set_an_attribute('ref', $new_value);
	    } else {
		if ( $this->template->chps[$key] == 'on' &&
		     $new_value != 'oui' )      { $new_value = ''; }
		if ( $this->template->chps[$key] == 'curdate' ) {
		    // print ($key);
		    $new_value = strftime ('%Y%m%d');
		}
		if ( $this->template->chps[$key] == 'timestamp' ) {
		    // print ($key);
		    $new_value = time();
		}
		$new->set_cdata_from_text($new_value);
	    }
	    //print "&nbsp;&nbsp;add ".$new->name." value: ".
	    //       $new->cdata ."<br>\n";
	}
	// for ($i = 0 ; $i < count ($this->ref->tags) ; $i++) {
	//     $chpn = $this->ref->tags[$i]->name;
	//     print "&nbsp;&nbsp;$chpn: ". $this->valueAt ($chpn) ."<br>\n";
	//}
    }

    /** Update the curent from an items array
     */
    function update($items) {
	for ($i = 0 ; $i < count($this->template->index) ; $i++) {
	    $key = $this->template->index[$i];
	    $new_value = $items[$i];
	    $cur = &$this->ref->find_first_subtag_by_name($key);
	    if ( $cur ) {
		//$cur->clear_cdata();
		if ( $this->template->extref[$key] ) {
		    $cur->set_an_attribute('ref', $new_value);
		} else {
		    if ( $this->template->chps[$key] == 'on' &&
			 $new_value != 'oui' )      { $new_value = ''; }
		    if ( $this->template->chps[$key] == 'curdate' ) {
		    	$new_value = strftime('%Y%m%d');
		    }
		    if ( $this->template->chps[$key] == 'timestamp' ) {
		    	$new_value = time();
		    }
		    $cur->set_cdata_from_text($new_value);
		}
	    } else {
		$new =& $this->ref->add_subtag($key);
		if ( $this->template->extref[$key] ) {
		    $new->set_an_attribute('ref', $new_value);
		} else {
		    if ( $this->template->chps[$key] == 'on' &&
			 $new_value != 'oui' )      { $new_value = ''; }
		    if ( $this->template->chps[$key] == 'curdate' ) {
		    	$new_value = strftime('%Y%m%d');
		    }
		    if ( $this->template->chps[$key] == 'timestamp' ) {
		    	$new_value = time();
		    }
		    $new->set_cdata_from_text($new_value);
		}
	    }
	}
    }

    /** Show the current element in a HTML table
     */
    function show($hidden=array()) {
	$wiki =& new MiniWiki();
        $in_table = false; 
        $txt = '';
        $style = "";
	for ($i = 0 ; $i < count($this->ref->tags) ; $i++) {
	    $chpn = $this->ref->tags[$i]->name;
            $key = $this->template->index[$i];
            if ( !in_array($key, $hidden)) {
                $value = $this->valueAt($chpn);
                $n = preg_match('/\d+,\d+/', $this->template->chps[$key]);
                if ( $n == 0 ) {         // the current field is not a textarea
                    if ( ! $in_table ) {
                        $in_table = true;
                        $txt .= "<table border='0' width='100%'>\n".
                               "<colgroup><col width='35%'/><col width='65%'/>".
                               "</colgroup>\n";
                    }
                    $txt .= " <tr>\n  <td>".
                            $this->template->labels[$chpn]." :</td>\n";
                    if ( $this->template->extref[$key] ) {
                        $value = $this->ref->tags[$i]->get_attribute('ref');
                    }
                    $val = $wiki->transform($value);
                    $txt .= "  <td style='$style'>".$val."</td>\n";
                } else {                     // the current field is a textarea
                    if ( $in_table ) {
                        $txt .= "</table>\n<hr/>\n";
                        $in_table = false;
                    }
                    $txt .= "<p><b>".$this->template->labels[$chpn]."</b>:<br/>\n";
                    $val = $wiki->transform($value);
                    $txt .= "<div class='text' style='$style'>".$val."</div>\n";
                }
            }
	}
        if ( $in_table ) {
            $txt .= " </tr>\n</table>\n";
        }
	return $txt;
    }

    /** Get the value of a field
     */
    function valueAt($key) {
	$cur = $this->ref->find_first_subtag_by_name($key);
        // print "$cur --- $key --- "; //.$cur->get_cdata()."<br/>\n";
        $ret = ($cur) ? $cur->get_cdata(): '';
	return $ret;
    }

    /** Get the attribute array of a field
     */
    function attributesAt($key) {
	$cur = $this->ref->find_first_subtag_by_name($key);
	return $cur->attributes;
    }

    /** Show the element as an HTML array line
     */
    function asHTMLarrayLine() {
	$txt = " <tr>\n";
	foreach ($this->template->index as $key) {
	    $value = $this->valueAt($key);
	    if ( $value == '' ) { $value = "&nbsp;"; }
	    $txt .= "  <td>$value</td>\n";
	}
	$txt .= " </tr>\n";
	return $txt;
    }

        
    /** Compose a presentation line subsituting fields in the template
     */
    function substTemplateLine($tline) {
	$wiki =& new MiniWiki();
        $dtb = '';
	foreach ($this->template->index as $key) {
            $value = $this->valueAt($key);
            if ( $key == 'date' ) {
                $dtb = $value;
                $value = ansi_to_date($value);
            } elseif ( $key == 'heure' ) {
                $hb = $value;
                $value = ansi_to_hour($value); 
            } elseif ( $key == 'tstamp' ) { 
                $value = ansi_to_date(date('Ymd', (int)$value)); 
            } elseif ( $key == 'url' ) { 
                ; 
            } else {
                ;
                $value = $wiki->transform($value);
            }               
            $tline = str_replace("*$key*", $value, $tline);
	}
	$tline = str_replace('*elt-key*', $this->key, $tline);
	$tline = str_replace('*elt-dtb*', $dtb, $tline);
	$tline = str_replace('*elt-hb*', $hb, $tline);
        if ($this->timestamp && $dtb && $dtb >= $this->timestamp) {
            $tline = str_replace('*elt-tstamp*', ' class="new"', $tline);
        } else {
            $tline = str_replace('*elt-tstamp*', '', $tline);
        }
	return $tline;
    }
}

/** XML collection interface
 *  This class is the only one public class of the package
 */
class XmlCollection {

    /** instance variables						*/
    var $xml;			// XMLfile object reference
    var $name = '';		// name of the collection
    var $path = '';		// path of the XML storage file
    var $mode = '';
    var $template;		// reference associated 'XMLColTemplate'
    var $index = array();	// array of element keys
    var $current;		// reference of the current 'XmlColElement'

    /** collection constructor.
     *   build a new collection associated with 'filename'; the name of the
     *   collection is the basename of the file.
     */
    function XmlCollection($filename, $mode='') {
	// readable?
	$this->path = $filename;
	$this->mode = $mode;
	$this->xml =& new XMLFile();
	$this->xml->read_file($this->path);
	// print "'$this->path' read<br>\n";
	$root = &$this->xml->roottag;
	$this->name = $root->name;
	// print "collection name: '$this->name'<br>\n";
	$this->template =& new XmlColTemplate($root->tags[0],
	                                      dirname($this->path),
					      $this->mode);
	for ( $i = 1 ; $i < count ($root->tags) ; $i++ ) {
	    $elt = &new XmlColElement($root->tags[$i], $this->template);
	    $this->index[$elt->key] =& $elt;
	    // print '##'.$elt->key.'## '. $elt->valueAt ('nom')."\n";
	}
    }

    /** Build a new key identifier from the element values
     */
    function build_key($items) {
        $key_elems = explode('.', $this->template->idcons);
        $new_key = '';
        foreach ($key_elems as $an_elem) {
            $i = $this->template->get_field_index($an_elem);
            // print "$i ==> ".$items[$i];
            $new_key .= normalizeName($items[$i]).'.';
        }
        $new_key = trim($new_key, '. ');
        return $new_key;
    }
    
    /** Is this key in the collection
     */
    function has_key($key) {
	return array_key_exists($key, $this->index);
    }

    /** get the collection size
     */
    function count() {
	return count($this->index);
    }

    /** get the numer of fields in each element
     */
    function items_count() {
	return $this->template->fields_number;
    }

    /** get a collection element by its id
     */
    function &getAnElt($id) {
	if ( $this->has_key($id) ) {
	    return $this->index[$id];
	} else {
	    return null;
	}
    }

    /** Get a list of selected elements sorted by ids
     */
    function sorted_ids($order='', $limit='', $select='', $select2='') {
	if ( $select ) {
	    list ($chp, $value) = explode('.', $select);
	    if ( $select2 ) {
		list ($chp2, $value2) = explode('.', $select2);
	    } else { $value2 = ''; }
	}
	$tmp_ar = array();
	foreach (array_keys ($this->index) as $key) {
	    $id = $this->index[$key]->ref->get_attribute('id');
	    if ( $select ) {
		$t =& $this->index[$key]->ref->find_tag_by_path($chp);
		$lv = $t->get_cdata();
		if ( $select2 ) {
		    $t =& $this->index[$key]->ref->find_tag_by_path($chp2);
		    $lv2 = $t->get_cdata();
		} else { $lv2 = ''; }
//		print ($chp."  ".$value."  ".$lv."<br>\n");
		$selected = ($lv == $value && $lv2 == $value2);
	    } else {
		$selected = true;
	    }
	    if ( $limit ) {
	        if ( $order == 'rev' ) {
		    if ( $id < $limit ) {
			if ( $selected ) { $tmp_ar[] = $id; }
		    }
		} else {
		    if ( $id > $limit ) {
			if ( $selected ) { $tmp_ar[] = $id; }
		    }
		}
	    } else { $tmp_ar[] = $id; }
	}
	if ( $order == 'rev' ) { rsort($tmp_ar); }
 	else { sort($tmp_ar); }
	return $tmp_ar;
    }

    /** Make a form 'select' block with all the collection is as keywords.
     *  @param itemName : mandatory, the name of the field in the form
     *  @param current : if not empty, this value is selected by default
     *  @param addMsg : if not empty add a message for new value
     *  @param amPos : if addMsg is true, fixes the position of this msg 
     *                 ('first' or 'last')
     *  @return : the composed XHTML text
     */
    function getIdsAsSelectBlock($itemName, $current='', $addMsg='', 
                                 $amPos='first', $width='15em') {
        $idls = $this->sorted_ids();
        $txt = "<select name='".$itemName."' style='width: ".$width."'>\n";
        if ( ! $current ) {     // not in modification mode
            if ( ! $addMsg ) { $addMsg = "&lt;- choisir -&gt;"; }
            $txt .= "<option value='' selected>".$addMsg."</option>\n";
        } else {
            if ( $addMsg && $amPos == 'first' ) {
                $txt .= "<option value='new'>".$addMsg."</option>\n";
            }
        }
        foreach ($idls as $val) {
            $txt .= "  <option value='$val'";
            if ( $val == $current ) { $txt .= " selected='selected'"; }
            $txt .= ">$val</option>\n";
        }
        if ( $addMsg && $amPos != 'first' ) {
            $txt .= "<option value='new'>".$addMsg."</option>\n";
        }
        $txt .= "</select>\n";
        return $txt;
    }
    
    /** Compare  un champ ventuellmeent multiple
     */
    function compare_mvalue($value, $select) {
        // print "select=$select -- value=$value<br />\n";
        if ( $select == '' ) { return true; }
        if ( $value == '' ) { return false; }
        $vals = explode(',', $select);
        if ( count($vals) == 1 ) {
            $ret = ($select == $value);
        } else {
            $ret = false;
            foreach ($vals as $val) {
                $ret = ($value == $val);
                if ($ret) break;
            }
        }
        //print "s=$select; v=$value; r=$ret<br>";
        return $ret;
    }
    
    /** Get a list of selected element references, sorted on specified fields
     */
    function sorted_keys($tags, $order='', $limit='', $select='', $select2='') {
	$tgar = explode('.', $tags);
	if ( $select ) {
	    list ($chp, $value) = explode('.', $select, 2);
	    if ( $select2 ) {
		list($chp2, $value2) = explode('.', $select2, 2);
	    } else { $value2 = ''; }
	}
	$tmp_ar = array();
	foreach (array_keys($this->index) as $key) {
	    $lk = '';
	    foreach ($tgar as $ta) {
                if ( $ta = 'elt-key' ) {
                    $lk .= $key;
                } else {
                    $t =& $this->index[$key]->ref->find_tag_by_path($ta);
                    if ($t) { $lk .= $t->get_cdata(); }
                }
	    }
	    if ( $select ) {
		$t =& $this->index[$key]->ref->find_tag_by_path($chp);
                if ($t) { $lv = $t->get_cdata(); } else { $lv = ''; }
		if ( $select2 ) {
		    $t =& $this->index[$key]->ref->find_tag_by_path($chp2);
                    if ($t) { $lv2 = $t->get_cdata(); } else { $lv2 = ''; }
		} else { $lv2 = ''; }
		//print ($chp." # ".$value." - ".$lv." # ".$value2." - ".$lv2."<br>\n");
//		$selected = ($lv == $value && $lv2 == $value2);
                $selected = $this->compare_mvalue($lv, $value) &&
                            $this->compare_mvalue($lv2, $value2);
	    } else {
		$selected = true;
	    }
            // print "$selected<br>";

	    if ( $limit ) {
	        if ( $order == 'rev' ) {
		    if ( strcmp ($lk, $limit) < 0 ) {
			if ( $selected ) { $tmp_ar[$lk] = $key; }
		    }
		} else {
		    if ( strcmp ($lk, $limit) >= 0 ) {
			if ( $selected ) { $tmp_ar[$lk] = $key; }
		    }
		}
	    } else { if ( $selected ) {$tmp_ar[$lk] = $key; }}
	}
//        print_r($tmp_ar);
	if ( $order == 'rev' ) { krsort($tmp_ar); }
 	else { ksort($tmp_ar); }
	$ret = array_values($tmp_ar);
	return $ret;
    }

    /**
     */
    function asHTMLarray($akeys) {
	$txt = $this->template->thead('90%');
	foreach ($akeys as $key) {
	    $txt .= $this->index[$key]->asHTMLarrayLine();
	}
	$txt .= "</table>\n";
	return $txt;
    }

    /** Show elements in a HTML text build from a template
     */
    function asTemplateLines($tline, $akeys) {
	$txt = '';
	foreach ($akeys as $key) {
	    $txt .= $this->index[$key]->substTemplateLine($tline);
	}
	return $txt;
    }

    /** Update an element from a fields array
     */
    function updateElt($key, $items) {
	$this->current =& $this->getAnElt($key);
	$this->current->update($items);
    }

    /** Remove a collection element
     */
    function removeElt($key) {
	$root = &$this->xml->roottag;
	$index = $root->index_of_subtag_by_attr('id', $key);
	$root->remove_subtag($index);
	unset ($this->index[$key]);
	unset ($this->current);
    }

    /** Create a new element from a fields array
     */
    function createElt($key, $items) {
	$root =& $this->xml->roottag;
	$ref =& $root->add_subtag($this->template->name, array('id' => $key));
	$elt = new XmlColElement(&$ref, $this->template, $key, $items);
	$this->index[$key] = &$elt;
    }

    /** Show an element as HTML array
     */
    function showElt($key) {
	$this->current = &$this->getAnElt($key);
	return $this->current->show();
    }

    /** Show an element with a template
     */
    function showEltTmpl($key, $tmpl) {
	$this->current = &$this->getAnElt($key);
	return $this->current->substTemplateLine($tmpl);
    }

    /** Save the collection file
     */
    function saveFile($filepath='') {
	if (!$filepath) { $filepath = $this->path; }
	// creation .bak
	$this->xml->write_file($filepath);
	return $filepath;
    }
}


/*
$collec = new XmlCollection ('../intra/data/adresses.xml');
$ar = $collec->sorted_keys ('nom.prenom');
print_r ($ar);
$ar = $collec->sorted_ids();
print_r ($ar);

$collec->xml->show ('txt');
print $collec->asHTMLarray();
print $collec->template->form('suite.php', 'window');
//print $collec->template->form('suite.php', 'window',
//                              $collec->getAnElt ('daniele.deveaux'));
$file = $collec->saveFile ('test.xml');
print "<p>&nbsp;</p>\n<p>'$file' written</p>\n";

//$collec2 = new XmlCollection ('../intra/data/associations.xml');
//print $collec2->asHTMLarray();
//print $collec2->template->form('suite.php', 'window');
*/
?>
